package com.bxm.localnews.merchant.service.lottery.impl;

import com.bxm.localnews.merchant.domain.lottery.ActivityAwardMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryParticipatorMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.dto.LocationDTO;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryCoreInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotterySharePosterDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryAwardEntityWithBLOBs;
import com.bxm.localnews.merchant.entity.lottery.LotteryEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryPhaseEntity;
import com.bxm.localnews.merchant.integration.LocationIntegrationService;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryChangeStatusParam;
import com.bxm.localnews.merchant.param.activity.LotteryPhasePosterDetailParam;
import com.bxm.localnews.merchant.service.enums.LotteryPhaseStatusEnum;
import com.bxm.localnews.merchant.service.enums.LotteryStatusEnum;
import com.bxm.localnews.merchant.service.enums.LotteryUserTypeEnum;
import com.bxm.localnews.merchant.service.lottery.*;
import com.bxm.localnews.merchant.service.lottery.cache.LotteryCacheManage;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.merchant.service.enums.LotteryStatusEnum.MERCHANT_OFFLINE;

/**
 * @author liujia
 * @date 2020-10-27 10:59
 **/
@Service
@Slf4j
public class LotteryPhaseServiceImpl implements LotteryPhaseService {

    private final LotteryAwardService lotteryAwardService;

    private LotteryService lotteryService;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final SequenceCreater sequenceCreater;

    private final LotteryCacheManage lotteryCacheManage;

    private final LotteryParticipatorMapper lotteryParticipatorMapper;

    private final ActivityAwardMapper activityAwardMapper;

    private final LotteryMapper lotteryMapper;

    private final LotteryParticipatorService lotteryParticipatorService;

    private final MerchantInfoFacadeService merchantInfoFacadeService;

    private final UserIntegrationService userIntegrationService;

    private final LocationIntegrationService locationIntegrationService;

    private final LotteryPushService lotteryPushService;

    private final LotteryStatisticsService lotteryStatisticsService;

    @Autowired
    public LotteryPhaseServiceImpl(LotteryAwardService lotteryAwardService,
                                   LotteryPhaseMapper lotteryPhaseMapper,
                                   SequenceCreater sequenceCreater,
                                   LotteryCacheManage lotteryCacheManage,
                                   LotteryParticipatorMapper lotteryParticipatorMapper,
                                   LotteryParticipatorService lotteryParticipatorService,
                                   ActivityAwardMapper activityAwardMapper,
                                   MerchantInfoFacadeService merchantInfoFacadeService,
                                   UserIntegrationService userIntegrationService,
                                   LotteryMapper lotteryMapper,
                                   LocationIntegrationService locationIntegrationService,
                                   LotteryPushService lotteryPushService,
                                   LotteryStatisticsService lotteryStatisticsService) {
        this.lotteryAwardService = lotteryAwardService;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.sequenceCreater = sequenceCreater;
        this.lotteryCacheManage = lotteryCacheManage;
        this.lotteryParticipatorMapper = lotteryParticipatorMapper;
        this.lotteryParticipatorService = lotteryParticipatorService;
        this.activityAwardMapper = activityAwardMapper;
        this.merchantInfoFacadeService = merchantInfoFacadeService;
        this.userIntegrationService = userIntegrationService;
        this.lotteryMapper = lotteryMapper;
        this.locationIntegrationService = locationIntegrationService;
        this.lotteryPushService = lotteryPushService;
        this.lotteryStatisticsService = lotteryStatisticsService;
    }

    private LotteryService getLotteryService() {
        if (null == lotteryService) {
            lotteryService = SpringContextHolder.getBean(LotteryService.class);
        }
        return lotteryService;
    }

    @Override
    public Message create(Long lotteryId) {
        log.info("活动[{}]开启新一期的夺宝活动", lotteryId);

        LotteryCoreInfoDTO detailDTO = lotteryMapper.getLotteryInfo(lotteryId);

        // 获取活动状态，判断活动是否处于活跃上架状态
        if (!Objects.equals(detailDTO.getStatus(), LotteryStatusEnum.ACTIVED.getCode())) {
            log.info("[{}]活动已下架，无法开启下一期活动", lotteryId);
            return Message.build(false, "活动已下架，无法开启活动");
        }

        // 如果存在商家，则判断商家是否在线
        if (detailDTO.getMerchantId() != null) {
            MerchantInfo merchantInfo = merchantInfoFacadeService.getMerchantInfo(detailDTO.getMerchantId());

            if (!Objects.equals(merchantInfo.getIsShelf(), 1)) {
                log.info("活动[{}]对应的商家[{}]不处于上架状态", lotteryId, detailDTO.getMerchantId());

                LotteryChangeStatusParam changeParam = new LotteryChangeStatusParam();
                changeParam.setLotteryId(lotteryId);
                changeParam.setEnable(false);

                getLotteryService().changeStatus(changeParam, MERCHANT_OFFLINE);

                return Message.build(false, "商家已下架，无法开启活动");
            }
        }

        Long phaseId = sequenceCreater.nextLongId();
        // 活动的当前期数+1作为最新一期活动的序号
        Integer currentPhaseNum = detailDTO.getCurrentPhaseNum() + 1;
        // 扣除库存
        Message message = lotteryAwardService.changeStock(detailDTO.getAwardId(), -1);

        if (message.isSuccess()) {
            // 设置活动的最后活动期ID和活动期数
            getLotteryService().setLastPhase(lotteryId, phaseId, currentPhaseNum);

            // 插入活动信息
            saveNewPhase(detailDTO, phaseId, currentPhaseNum);

            // 设置其他活动期的最后一期活动（用于定位到当前活动）
            lotteryPhaseMapper.updateLastPhaseId(lotteryId, phaseId);

            afterCreate(detailDTO, phaseId, currentPhaseNum);
        }

        log.info("活动[{}]开启结果：{}", lotteryId, message);

        return message;
    }

    private void afterCreate(LotteryCoreInfoDTO detailDTO, Long phaseId, Integer currentPhaseNum) {
        // 如果允许虚拟用户参加活动
        if (Objects.equals(detailDTO.getVirtualFlag(), 1)) {
            lotteryParticipatorService.addVirtual(phaseId);
        }

        // 给赞助商家推送消息
        if (detailDTO.getMerchantUserId() != null) {
            lotteryPushService.startPhase(detailDTO.getMerchantUserId(),
                    currentPhaseNum,
                    phaseId,
                    detailDTO.getLotteryId());
        }

        // 初始化统计数据
        lotteryStatisticsService.setup(detailDTO.getLotteryId(), phaseId);
    }

    private void saveNewPhase(LotteryCoreInfoDTO detailDTO, Long phaseId, Integer currentPhaseNum) {
        LotteryPhaseEntity entity = new LotteryPhaseEntity();
        entity.setId(phaseId);
        entity.setStatus(LotteryPhaseStatusEnum.GOING.getCode());
        entity.setLotteryId(detailDTO.getLotteryId());
        entity.setAwardId(detailDTO.getAwardId());
        entity.setCreateTime(new Date());
        entity.setTargetNum(detailDTO.getTargetNum());
        entity.setCurrentNum(0);
        entity.setPhaseNum(currentPhaseNum);

        lotteryPhaseMapper.insert(entity);
    }

    @Override
    public Message cancel(Long lotteryId, Long phaseId, boolean focus) {
        log.info("活动[{}]的[{}]下架", lotteryId, phaseId);

        // 判断活动参与者是否存在真实参与用户，如果存在则活动不能取消
        if (!focus && lotteryParticipatorMapper.countUserType(phaseId, LotteryUserTypeEnum.ACTURAL.getCode()) > 0) {
            log.info("活动[{}]-[{}]有真实用户参与，无法取消", lotteryId, phaseId);

            return Message.build(false, "活动有真实用户参与，无法取消");
        }

        // 更新活动状态为已取消
        lotteryPhaseMapper.updateStatus(phaseId, LotteryPhaseStatusEnum.CANCEL.getCode());
        // 清除活动的所有活动期对应的最后一期活动ID
        lotteryPhaseMapper.updateLastPhaseId(lotteryId, null);

        return Message.build();
    }

    @Override
    public Message close(Long lotteryId, Long phaseId) {
        log.info("活动[{}]的[{}]完成开奖", lotteryId, phaseId);

        LotteryCoreInfoDTO lotteryInfo = lotteryMapper.getLotteryInfo(lotteryId);

        // 删除活动中的最后一期ID
        getLotteryService().setLastPhase(lotteryId, null, null);
        // 删除活动对应的活动期的ID
        lotteryPhaseMapper.updateLastPhaseId(lotteryId, null);
        // 更新当前活动期状态
        lotteryPhaseMapper.updateStatus(phaseId, LotteryPhaseStatusEnum.FINISH.getCode());
        // 删除缓存
        lotteryCacheManage.clearPhaseCache(phaseId);

        if (null != lotteryInfo.getMerchantUserId()) {
            lotteryPushService.finishPhase(lotteryInfo.getMerchantId(),
                    lotteryInfo.getMerchantUserId(),
                    lotteryInfo.getCurrentPhaseNum());
        }

        return Message.build();
    }

    @Override
    public Message resume(Long lotteryId, Long phaseId) {
        log.info("活动[{}]的[{}]重新开启", lotteryId, phaseId);

        // 将活动重新更新为激活状态
        lotteryPhaseMapper.updateStatus(phaseId, LotteryPhaseStatusEnum.GOING.getCode());
        // 将活动对应的活动期更新为当前活动期ID（用于定位到最新活动）
        lotteryPhaseMapper.updateLastPhaseId(lotteryId, phaseId);

        return Message.build();
    }


    @Override
    public LotterySharePosterDTO getSharePosterInfo(LotteryPhasePosterDetailParam param) {
        LotteryPhaseEntity lotteryPhaseEntity = lotteryPhaseMapper.selectById(param.getPhaseId());
        if (lotteryPhaseEntity == null) {
            return new LotterySharePosterDTO();
        }
        LotterySharePosterDTO resultDTO = new LotterySharePosterDTO();
        resultDTO.setLotteryId(lotteryPhaseEntity.getLotteryId());
        resultDTO.setPhaseId(lotteryPhaseEntity.getId());
        resultDTO.setTargetNum(lotteryPhaseEntity.getTargetNum());

        Long lotteryId = lotteryPhaseEntity.getLotteryId();
        LotteryEntity lotteryEntity = lotteryMapper.selectByPrimaryKey(lotteryId);
        resultDTO.setTitle(lotteryEntity.getTitle());

        //奖品id
        Long awardId = lotteryPhaseEntity.getAwardId();
        //获取奖品信息
        LotteryAwardEntityWithBLOBs lotteryAwardEntity = activityAwardMapper.selectByPrimaryKey(awardId);
        resultDTO.setPrice(lotteryAwardEntity.getPrice());
        resultDTO.setImg(lotteryAwardEntity.getFirstImg());

        Long merchantId = lotteryAwardEntity.getMerchantId();
        if (merchantId != null) {
            MerchantInfo merchantInfo = merchantInfoFacadeService.getMerchantInfo(merchantId);
            resultDTO.setAddress(merchantInfo.getAddress());
        } else {
            //获取地区名称，设置到地址字段中
            LocationDTO locationInfo = locationIntegrationService.getLocationByGeocode(param.getAreaCode());
            if (locationInfo != null) {
                resultDTO.setAddress(locationInfo.getName());
            }
        }
        //获取用户id获取用户信息
        Long currentUserId = param.getUserId();

        UserInfoDTO shareUser = userIntegrationService.getUserFromRedisDB(currentUserId);
        if (shareUser != null) {
            resultDTO.setUserId(currentUserId);
            resultDTO.setNickName(shareUser.getNickname());
            resultDTO.setHeadImg(shareUser.getHeadImg());
        }

        return resultDTO;
    }
}






















