package com.bxm.localnews.merchant.service.lottery.impl;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.common.service.VerificationCodeService;
import com.bxm.localnews.merchant.domain.lottery.ActivityPrizeMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryWinnerMapper;
import com.bxm.localnews.merchant.dto.activity.*;
import com.bxm.localnews.merchant.entity.lottery.ActivityPrizeEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryWinnerEntity;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryPrizeCheckParam;
import com.bxm.localnews.merchant.param.activity.LotteryPrizeParam;
import com.bxm.localnews.merchant.param.activity.LotteryWinnerManagePageParam;
import com.bxm.localnews.merchant.service.enums.ActivityPrizeStatusEnum;
import com.bxm.localnews.merchant.service.lottery.LotteryPrizeService;
import com.bxm.localnews.merchant.service.lottery.LotteryPushService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Objects;

import static com.github.pagehelper.page.PageMethod.startPage;

/**
 * 夺宝活动中奖奖品相关服务实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2020/10/27 9:59 上午
 */
@Service
@Slf4j
public class LotteryPrizeServiceImpl implements LotteryPrizeService {
    @Resource
    private ActivityPrizeMapper activityPrizeMapper;

    @Resource
    private SequenceCreater sequenceCreater;

    @Resource
    private VerificationCodeService verificationCodeService;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private LotteryPushService lotteryPushService;

    @Resource
    private LotteryPhaseMapper lotteryPhaseMapper;

    @Resource
    private LotteryWinnerMapper lotteryWinnerMapper;

    @Override
    public PageWarper<LotteryWinnerManageDTO> selectByPage(LotteryWinnerManagePageParam param) {
        startPage(param.getPageNum(), param.getPageSize());
        return new PageWarper<>(activityPrizeMapper.selectByPage(param));
    }

    @Override
    public ActivityPrizeManageDTO getPrizeDetail(Long prizeId) {
        ActivityPrizeEntity activityPrizeEntity = activityPrizeMapper.selectByPrimaryKey(prizeId);
        ActivityPrizeManageDTO detailInfo = new ActivityPrizeManageDTO();

        BeanUtils.copyProperties(activityPrizeEntity, detailInfo);
        detailInfo.setPrizeId(activityPrizeEntity.getId());
        return detailInfo;
    }

    @Override
    public void save(ActivityPrizeManageDTO param) {
        // 添加中奖奖品的物流信息，将奖品状态设置为已发奖（其他类型的奖品特有）
        ActivityPrizeEntity updatePrizeEntity = new ActivityPrizeEntity();
        updatePrizeEntity.setId(param.getPrizeId());
        updatePrizeEntity.setExpressCompany(param.getExpressCompany());
        updatePrizeEntity.setExpressNum(param.getExpressNum());
        updatePrizeEntity.setDeliveryAddress(param.getDeliveryAddress());
        updatePrizeEntity.setStatus(ActivityPrizeStatusEnum.GRANT.getCode());

        activityPrizeMapper.updateByPrimaryKeySelective(updatePrizeEntity);
    }

    @Override
    public Message savePrizeInfo(ActivityPrizeEntity entity) {
        // 创建一个七位数的核销码
        Long code = verificationCodeService.gen(RedisConfig.PRIZE_VERIFICATION_CODE_BUCKET, 100000L);

        entity.setId(sequenceCreater.nextLongId());
        entity.setCode(String.valueOf(code));

        return Message.build(activityPrizeMapper.insert(entity));
    }

    @Override
    public LotteryCheckResultDTO check(LotteryPrizeCheckParam param) {
        ActivityPrizeEntity prizeEntity = activityPrizeMapper.selectByPrimaryKey(param.getPrizeId());

        if (null == prizeEntity) {
            return new LotteryCheckResultDTO(2);
        }

        if (Objects.equals(prizeEntity.getStatus(), ActivityPrizeStatusEnum.CHECKED.getCode())) {
            return new LotteryCheckResultDTO(3);
        }

        ActivityPrizeEntity updatePrizeEntity = new ActivityPrizeEntity();
        updatePrizeEntity.setId(param.getPrizeId());
        updatePrizeEntity.setStatus(ActivityPrizeStatusEnum.CHECKED.getCode());
        updatePrizeEntity.setCheckTime(new Date());

        int result = activityPrizeMapper.updateByPrimaryKeySelective(updatePrizeEntity);

        if (result > 0) {
            // 核销成功后，推送消息
            LotteryWinnerEntity winner = lotteryWinnerMapper.selectByPrimaryKey(prizeEntity.getBizId());

            if (null != winner) {
                LotteryPhaseCoreInfoDTO phaseInfo = lotteryPhaseMapper.getCorePhaseInfo(winner.getPhaseId());

                if (null != phaseInfo.getMerchantId()) {
                    lotteryPushService.checkSuccess(phaseInfo.getMerchantId(),
                            phaseInfo.getMerchantUserId(),
                            phaseInfo.getPhaseNum());
                }
            }
            return new LotteryCheckResultDTO(1);
        }

        return new LotteryCheckResultDTO(3);
    }

    @Override
    public ActivityPrizeInfoDTO getPrizeInfo(LotteryPrizeParam param) {
        String prizeCode = param.getCode();

        if (StringUtils.isBlank(prizeCode) || !StringUtils.isNumeric(prizeCode)) {
            return invalidResult();
        }
        ActivityPrizeInfoDTO prizeInfo = activityPrizeMapper.getPrizeByCode(prizeCode);

        if (prizeInfo == null) {
            return invalidResult();
        }

        if (!Objects.equals(prizeInfo.getMerchantId(), param.getMerchantId())) {
            log.warn("核销码的所属商家与请求商家不一致，请求商家ID:{}，核销码为：{}，核销码所属商家为：{}",
                    param.getMerchantId(),
                    param.getCode(),
                    prizeInfo.getMerchantId());

            return invalidResult();
        }

        prizeInfo.setWinnerUserVipFlag(userIntegrationService.isVip(prizeInfo.getWinnerUserId()));
        return prizeInfo;
    }

    private ActivityPrizeInfoDTO invalidResult() {
        return ActivityPrizeInfoDTO.builder()
                .status(3)
                .build();
    }
}
























