package com.bxm.localnews.merchant.service.lottery.impl;

import com.bxm.localnews.merchant.common.utils.ValidateUtils;
import com.bxm.localnews.merchant.domain.lottery.ActivityAwardMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryAreaMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryMapper;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.dto.AreaInfoDTO;
import com.bxm.localnews.merchant.dto.LocationDTO;
import com.bxm.localnews.merchant.dto.activity.*;
import com.bxm.localnews.merchant.entity.lottery.LotteryAreaEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryAwardEntityWithBLOBs;
import com.bxm.localnews.merchant.entity.lottery.LotteryEntity;
import com.bxm.localnews.merchant.integration.LocationIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryChangeStatusParam;
import com.bxm.localnews.merchant.param.activity.LotteryManageParam;
import com.bxm.localnews.merchant.param.activity.LotteryRemoveParam;
import com.bxm.localnews.merchant.service.enums.LotteryStatusEnum;
import com.bxm.localnews.merchant.service.lottery.*;
import com.bxm.localnews.merchant.service.task.LotteryAutoEnableCallback;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTask;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 2020-10-26 19:41
 **/
@Slf4j
@Service
public class LotteryServiceImpl implements LotteryService {

    private final LotteryMapper lotteryMapper;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final LotteryAreaMapper lotteryAreaMapper;

    private final ActivityAwardMapper activityAwardMapper;

    private final LocationIntegrationService locationIntegrationService;

    private LotteryPhaseService lotteryPhaseService;

    private final SequenceCreater sequenceCreater;

    private final ScheduleService scheduleService;

    private final LotteryAutoEnableCallback lotteryAutoEnableCallback;

    private final LotteryPushService lotteryPushService;

    private final LotteryAwardService lotteryAwardService;

    private final LotteryMerchantService lotteryMerchantService;

    @Autowired
    public LotteryServiceImpl(LotteryMapper lotteryMapper,
                              LotteryAreaMapper lotteryAreaMapper,
                              LocationIntegrationService locationIntegrationService,
                              SequenceCreater sequenceCreater,
                              ScheduleService scheduleService,
                              LotteryAutoEnableCallback lotteryAutoEnableCallback,
                              LotteryPushService lotteryPushService,
                              LotteryAwardService lotteryAwardService,
                              LotteryMerchantService lotteryMerchantService,
                              LotteryPhaseMapper lotteryPhaseMapper,
                              ActivityAwardMapper activityAwardMapper) {
        this.lotteryMapper = lotteryMapper;
        this.lotteryAreaMapper = lotteryAreaMapper;
        this.locationIntegrationService = locationIntegrationService;
        this.sequenceCreater = sequenceCreater;
        this.scheduleService = scheduleService;
        this.lotteryAutoEnableCallback = lotteryAutoEnableCallback;
        this.lotteryPushService = lotteryPushService;
        this.lotteryAwardService = lotteryAwardService;
        this.lotteryMerchantService = lotteryMerchantService;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.activityAwardMapper = activityAwardMapper;
    }

    private LotteryPhaseService getLotteryPhaseService() {
        if (lotteryPhaseService == null) {
            lotteryPhaseService = SpringContextHolder.getBean(LotteryPhaseService.class);
        }
        return lotteryPhaseService;
    }

    @Override
    public Message changeStatus(LotteryChangeStatusParam param, LotteryStatusEnum status) {
        if (null != status) {
            log.info("活动[{}]状态变更，上下架：{},状态变更：{}", param.getLotteryId(), param.getEnable(), status.name());
        } else {
            log.info("活动[{}]状态变更，上下架：{}", param.getLotteryId(), param.getEnable());
        }

        LotteryEntity updateEntity = new LotteryEntity();
        updateEntity.setId(param.getLotteryId());
        updateEntity.setModifyTime(new Date());

        LotteryEntity entity = lotteryMapper.selectByPrimaryKey(param.getLotteryId());

        if (param.getEnable()) {
            // 如果是启用状态
            updateEntity.setEnable(1);
            // 活动未到达开启时间，则显示为待开启
            if (DateUtils.before(entity.getStartTime(), new Date())) {
                updateEntity.setStatus(LotteryStatusEnum.ACTIVED.getCode());
            } else {
                updateEntity.setStatus(LotteryStatusEnum.UNDER_PLANNING.getCode());
            }
        } else {
            if (null != status) {
                updateEntity.setStatus(status.getCode());
            }
            updateEntity.setEnable(0);

        }

        Message message = Message.build(lotteryMapper.updateByPrimaryKeySelective(updateEntity));

        if (message.isSuccess()) {
            if (param.getEnable()) {
                if (entity.getCurrentPhaseId() != null) {
                    // 如果存在最后一期活动，则说明是重新开启，则重新启用活动
                    getLotteryPhaseService().resume(entity.getId(), entity.getCurrentPhaseId());
                } else if (DateUtils.before(entity.getStartTime(), new Date())) {
                    // 如果活动已经在投放时间之后，则创建新的活动期数，否则将由定时任务进行创建
                    getLotteryPhaseService().create(entity.getId());
                }
            } else {
                // 取消当前的活动(是否取消成功)
                getLotteryPhaseService().cancel(entity.getId(), entity.getCurrentPhaseId(), false);
            }
        }

        return message;
    }

    @Override
    public Message remove(LotteryRemoveParam param) {
        LotteryEntity lotteryEntity = lotteryMapper.selectByPrimaryKey(param.getLotteryId());

        if (Objects.equals(LotteryStatusEnum.ACTIVED.getCode(), lotteryEntity.getStatus())) {
            return Message.build(false, "上架中的活动不支持删除操作");
        }

        LotteryEntity entity = new LotteryEntity();
        entity.setId(param.getLotteryId());
        entity.setDeleteUserId(param.getCurrentUserId());
        entity.setDeleteFlag(1);
        entity.setDeleteTime(new Date());

        return Message.build(lotteryMapper.remove(entity));
    }

    @Override
    public LotteryDetailDTO get(Long lotteryId) {
        LotteryEntity entity = lotteryMapper.selectByPrimaryKey(lotteryId);

        LotteryAwardEntityWithBLOBs awardInfo = activityAwardMapper.selectByPrimaryKey(entity.getAwardId());

        LotteryDetailDTO detailDTO = new LotteryDetailDTO();
        BeanUtils.copyProperties(entity, detailDTO);

        detailDTO.setLotteryId(entity.getId());

        int useStock = awardInfo.getUseStock() == null ? 0 : awardInfo.getUseStock();
        int totalStock = awardInfo.getStock() == null ? 0 : awardInfo.getStock();
        int resultStock = totalStock - useStock;

        //展示库存信息
        detailDTO.setStock(Math.max(resultStock, 0));
        //回显奖品名称
        detailDTO.setAwardTitle(awardInfo.getTitle());

        if (Objects.equals(0, entity.getGlobalFlag())) {
            // 填充地区信息
            detailDTO.setAreaInfos(loadAreaInfo(lotteryId));
        }


        return detailDTO;
    }

    @Override
    public Message setLastPhase(Long lotteryId, Long lastPhaseId, Integer lastPhaseNum) {
        LotteryEntity updateEntity = new LotteryEntity();
        updateEntity.setId(lotteryId);
        updateEntity.setCurrentPhaseId(lastPhaseId);
        updateEntity.setCurrentPhaseNum(lastPhaseNum);

        if (null == lastPhaseId) {
            return Message.build(lotteryMapper.removeLastPhase(lotteryId));
        }

        return Message.build(lotteryMapper.updateByPrimaryKeySelective(updateEntity));
    }

    private List<AreaInfoDTO> loadAreaInfo(Long lotteryId) {
        List<String> areaCodes = lotteryAreaMapper.queryByLotteryId(lotteryId);

        return areaCodes.stream().map(areaCode -> {
            LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(areaCode);

            AreaInfoDTO areaInfo = new AreaInfoDTO();
            areaInfo.setAreaCode(areaCode);
            areaInfo.setAreaName(locationByGeocode.getName());
            return areaInfo;
        }).collect(Collectors.toList());
    }

    @Override
    public Message saveOrUpdate(LotteryDetailDTO param) {
        LotteryEntity entity = new LotteryEntity();
        BeanUtils.copyProperties(param, entity);
        entity.setId(param.getLotteryId());
        // 活动新增、编辑后默认均为下架状态
        entity.setEnable(0);
        entity.setStatus(LotteryStatusEnum.UNDER_PLANNING.getCode());

        String errorMsg = ValidateUtils.fastValid(entity);
        if (null != errorMsg) {
            return Message.build(false, errorMsg);
        }

        Message message = check(entity);
        if (!message.isSuccess()) {
            return message;
        }

        int result;

        if (null == param.getLotteryId()) {
            entity.setId(sequenceCreater.nextLongId());
            entity.setCreateTime(new Date());
            entity.setCreator(param.getCurrentUserId());
            entity.setDeleteFlag(0);
            entity.setCurrentPhaseNum(0);

            result = lotteryMapper.insert(entity);

            afterCreate(entity);
        } else {
            LotteryEntity existsLottery = lotteryMapper.selectByPrimaryKey(param.getLotteryId());
            if (null != existsLottery.getCurrentPhaseId()) {
                //将活动期设置为下架状态
                getLotteryPhaseService().cancel(entity.getId(), existsLottery.getCurrentPhaseId(), false);
            }

            entity.setModifyTime(new Date());

            result = lotteryMapper.updateByPrimaryKeySelective(entity);
        }

        createTask(entity);

        lotteryAreaMapper.removeByLotteryId(entity.getId());

        if (Objects.equals(entity.getGlobalFlag(), 0)) {
            saveAreaInfo(entity.getId(), param);
        }

        return Message.build(result);
    }

    private Message check(LotteryEntity entity) {
        if (Objects.equals(entity.getActualFlag(), 0) && Objects.equals(entity.getVirtualFlag(), 0)) {
            return Message.build(false, "马甲号中奖的活动必须允许马甲号参与");
        }
        return Message.build();
    }

    private void afterCreate(LotteryEntity entity) {
        LotteryAwardDetailDTO awardDetailDTO = lotteryAwardService.get(entity.getAwardId());

        if (null != awardDetailDTO.getMerchantId()) {

            LotteryMerchantInfoDTO merchantInfo = lotteryMerchantService.get(awardDetailDTO.getMerchantId());
            if (Objects.nonNull(merchantInfo) && Objects.nonNull(merchantInfo.getMerchantUserId())) {
                lotteryPushService.createLottery(awardDetailDTO.getMerchantId(),
                        merchantInfo.getMerchantUserId(),
                        entity.getStartTime());
            }
        }
    }

    private void saveAreaInfo(Long lotteryId, LotteryDetailDTO param) {
        if (StringUtils.isNotBlank(param.getAreaCodeStr())) {
            String[] areaCodeList = param.getAreaCodeStr().split(",");
            for (String areaCode : areaCodeList) {
                LotteryAreaEntity entity = new LotteryAreaEntity();
                entity.setId(sequenceCreater.nextLongId());
                entity.setLotteryId(lotteryId);
                entity.setAreaCode(areaCode.trim());

                lotteryAreaMapper.insert(entity);
            }
        }
    }

    /**
     * 创建或编辑活动时，创建对应的定时任务，用于在活动设定的投放时间开启时创建夺宝活动
     * 如果夺宝活动到期后没有上架，则不进行处理
     *
     * @param entity 夺宝活动配置信息
     */
    private void createTask(LotteryEntity entity) {
        String taskName = lotteryAutoEnableCallback.getClass().getSimpleName() + "_" + entity.getId();

        scheduleService.remove(taskName);

        if (null == entity.getStartTime() || DateUtils.before(entity.getStartTime(), new Date())) {
            return;
        }

        OnceTask expiredTask = OnceTaskBuilder.builder(taskName, entity.getStartTime(), lotteryAutoEnableCallback)
                .callbackParam(entity.getId())
                .description("定时开启夺宝活动：" + entity.getId())
                .build();
        scheduleService.push(expiredTask);
    }

    @Override
    public PageWarper<LotteryOverviewDTO> query(LotteryManageParam param) {
        List<LotteryOverviewDTO> result = lotteryMapper.queryByPage(param);

        for (LotteryOverviewDTO lotteryOverviewDTO : result) {
            if (!lotteryOverviewDTO.getGlobalFlag()) {
                lotteryOverviewDTO.setAreaList(loadAreaInfo(lotteryOverviewDTO.getLotteryId()));
            }
        }

        return new PageWarper<>(result);
    }

    @Override
    public LotteryDetailFacadeDTO queryLastEnableLotteryInfo(String areaCode) {
        return lotteryPhaseMapper.queryLastEnableLotteryInfo(areaCode);
    }
}





















