package com.bxm.localnews.merchant.service.lottery.strategy;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinResultDTO;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseJoinParam;
import com.bxm.localnews.merchant.service.enums.LotteryJoinResultEnum;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import com.bxm.newidea.component.redis.DistributedLock;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.UUID;

import static com.bxm.localnews.merchant.common.config.RedisConfig.ACTIVITY_LOTTERY_RESOUCE_KEY;

/**
 * 用户参与夺宝活动的策略处理
 *
 * @author liujia
 * @date 2020-10-27 16:06
 **/
@Component
@Slf4j
public class JoinStrategyManage {

    private final List<IJoinStrategy> strategies;

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final DistributedLock distributedLock;

    @Autowired
    public JoinStrategyManage(List<IJoinStrategy> strategies,
                              LotteryPhaseMapper lotteryPhaseMapper,
                              DistributedLock distributedLock) {
        this.strategies = strategies;
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.distributedLock = distributedLock;
    }

    /**
     * 根据参数查找对应的处理策略，实现用户参与夺宝活动的处理流程
     *
     * @param context 参数上下文
     * @return 参与结果，存在多种业务分支逻辑，详见具体的策略实现类
     */
    @Transactional(rollbackFor = Exception.class)
    public LotteryJoinResultDTO execute(LotteryJoinContext context) {
        if (log.isDebugEnabled()) {
            log.debug("用户参与活动，参与参数：[{}]", context);
        }

        LotteryPhaseJoinParam param = context.getParam();

        String resourceId = ACTIVITY_LOTTERY_RESOUCE_KEY.copy()
                .appendKey(param.getPhaseId())
                .appendKey(param.getUserId())
                .gen();

        String requestId = UUID.randomUUID().toString();

        if (distributedLock.lock(resourceId, requestId)) {
            context.setPhaseInfo(lotteryPhaseMapper.getCorePhaseInfo(param.getPhaseId()));

            for (IJoinStrategy strategy : strategies) {
                if (strategy.match(context)) {
                    LotteryJoinResultDTO resultDTO = strategy.execute(context);

                    log.info("参与夺宝活动完成，参与方式：{},参与参数：{},参与结果：{}",
                            strategy.getClass().getSimpleName(),
                            JSON.toJSONString(param),
                            JSON.toJSONString(resultDTO));

                    distributedLock.unlock(resourceId, requestId);
                    return resultDTO;
                }
            }

        }

        log.error("参与活动失败，请求的参与方式不存在或存在重复点击，请求参数：{}", JSON.toJSONString(param));

        return LotteryJoinResultDTO.builder()
                .success(false)
                .joinResult(LotteryJoinResultEnum.FINISH.getCode())
                .build();
    }
}
