package com.bxm.localnews.merchant.service.lottery.strategy.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.merchant.domain.lottery.LotteryPhaseMapper;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinResultDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinStateDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryMerchantInfoDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCoreInfoDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryParticipatorEntity;
import com.bxm.localnews.merchant.integration.BizLogIntegrationService;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryCheckJoinTimesParam;
import com.bxm.localnews.merchant.param.activity.LotteryMerchantFollowParam;
import com.bxm.localnews.merchant.service.config.ActivityProperties;
import com.bxm.localnews.merchant.service.enums.LotteryJoinResultEnum;
import com.bxm.localnews.merchant.service.enums.LotteryJoinSourceEnum;
import com.bxm.localnews.merchant.service.enums.LotteryPhaseStatusEnum;
import com.bxm.localnews.merchant.service.lottery.LotteryDrawService;
import com.bxm.localnews.merchant.service.lottery.LotteryMerchantService;
import com.bxm.localnews.merchant.service.lottery.LotteryParticipatorService;
import com.bxm.localnews.merchant.service.lottery.LotteryStatisticsService;
import com.bxm.localnews.merchant.service.lottery.cache.LotteryCacheManage;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import com.bxm.localnews.merchant.service.lottery.event.LotteryJoinEvent;
import com.bxm.localnews.merchant.service.lottery.strategy.IJoinStrategy;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Objects;

import static com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum.JOINED_LOTTERY;

/**
 * 参与夺宝活动的逻辑处理策略抽象类
 *
 * @author liujia
 * @date 2020-10-27 16:04
 **/
@Slf4j
public abstract class AbstractJoinStrategy implements IJoinStrategy {

    private LotteryParticipatorService lotteryParticipatorService;

    private LotteryDrawService lotteryDrawService;

    @Resource
    private LotteryPhaseMapper lotteryPhaseMapper;

    @Resource
    protected LotteryCacheManage lotteryCacheManage;

    @Resource
    protected ActivityProperties activityProperties;

    @Resource
    protected UserIntegrationService userIntegrationService;

    @Resource
    private LotteryMerchantService lotteryMerchantService;

    @Resource
    protected LotteryStatisticsService lotteryStatisticsService;

    @Resource
    protected BizLogIntegrationService bizLogIntegrationService;

    LotteryParticipatorService getLotteryParticipatorService() {
        if (null == lotteryParticipatorService) {
            lotteryParticipatorService = SpringContextHolder.getBean(LotteryParticipatorService.class);
        }
        return lotteryParticipatorService;
    }

    private LotteryDrawService getLotteryDrawService() {
        if (lotteryDrawService == null) {
            lotteryDrawService = SpringContextHolder.getBean(LotteryDrawService.class);
        }
        return lotteryDrawService;
    }

    @Override
    public LotteryJoinResultDTO execute(LotteryJoinContext context) {
        // 检查用户是否符合参与活动的条件
        LotteryJoinResultDTO check = check(context);

        try {
            if (check.getSuccess()) {
                // 记录用户参与记录
                saveParticipator(context);
                // 参与成功的后置处理
                check = afterExecute(context);
            }

            if (check.getSuccess()) {
                check.setCode(context.getJoinCode());
                if (check.getJoinResult() == null) {
                    // 设置参与成功
                    check.setJoinResult(LotteryJoinResultEnum.SUCCESS.getCode());
                }
            } else {
                log.info("活动参与失败，请求参数：{}", JSON.toJSONString(context));
            }
        } catch (Exception e) {
            log.error("用户参与夺宝活动失败,参与参数：{}", JSON.toJSONString(context));
            log.error(e.getMessage(), e);

            rollback(context);
            throw e;
        }

        return check;
    }

    /**
     * 前置检查，子类选择性扩展
     */
    LotteryJoinResultDTO preCheck(LotteryJoinContext context) {
        return LotteryJoinResultDTO.buildDefault();
    }

    /**
     * 发生异常时，对关键数据进行回滚
     *
     * @param context 参数上下文对象
     */
    private void rollback(LotteryJoinContext context) {
        // 退还活动参与人次
        lotteryCacheManage.decrementJoinNum(context.getPhaseInfo().getPhaseId());
    }

    /**
     * 参与活动成功的后置处理
     *
     * @param context 参数上下文
     */
    LotteryJoinResultDTO afterExecute(LotteryJoinContext context) {
        // 用户参与成功后触发开奖逻辑
        triggerDraw(context);

        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();

        // 如果开启了自动关注并且是商家赞助活动并且未关注商家，在参与活动成功后则自动进行关注
        if (activityProperties.getEnableAutoFollow()
                && null != phaseInfo.getMerchantUserId()
                && !Boolean.TRUE.equals(context.getFollowMerchantUser())) {
            follow(context, phaseInfo);
        }

        // 如果是虚拟用户，则强制关注
        if (null != phaseInfo.getMerchantUserId()
                && null != phaseInfo.getMerchantId()
                && LotteryJoinSourceEnum.VIRTUAL.equals(context.getJoinSource())) {
            follow(context, phaseInfo);
        }

        // 参与成功增加标志位
        userIntegrationService.updateUserJudgeMarker(context.getParam().getUserId(), JOINED_LOTTERY, true);
        // 用户参与成功事件
        lotteryStatisticsService.post(new LotteryJoinEvent(phaseInfo.getLotteryId(), phaseInfo.getPhaseId()));

        return LotteryJoinResultDTO.buildDefault();
    }

    private void follow(LotteryJoinContext context, LotteryPhaseCoreInfoDTO phaseInfo) {
        // 判断是否已经建立邀请关系
        if (!userIntegrationService.isFollow(context.getParam().getUserId(), phaseInfo.getMerchantUserId())) {
            lotteryMerchantService.execAsyncFollow(LotteryMerchantFollowParam.builder()
                    .merchantUserId(phaseInfo.getMerchantUserId())
                    .userId(context.getParam().getUserId())
                    .phaseId(phaseInfo.getPhaseId())
                    .build());
        }
    }

    /**
     * 当参与人数为开奖人数少一人时，触发活动开奖
     *
     * @param context 参数上下文
     */
    private void triggerDraw(LotteryJoinContext context) {
        // 最后一条马甲号参与，不触发开奖
        if (Boolean.TRUE.equals(context.getSkipTriggerDraw())) {
            return;
        }

        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();

        if (phaseInfo.getTargetNum() <= context.getCurrentJoinNum() + 1) {
            log.info("触发活动[{}]开奖，目标人数：{},数据库中当前人数：{},缓存中当前人数：{}",
                    phaseInfo.getPhaseId(),
                    phaseInfo.getTargetNum(),
                    phaseInfo.getCurrentNum(),
                    context.getCurrentJoinNum());

            getLotteryDrawService().execAsyncDraw(context);
        }
    }

    private void fillUserInfo(LotteryJoinContext context) {
        UserInfoDTO cacheUser = context.getUserInfo();
        if (cacheUser == null) {
            cacheUser = userIntegrationService.getUserFromRedisDB(context.getParam().getUserId());
            context.setUserInfo(cacheUser);
        }
    }

    void setJoinTimesInfo(LotteryJoinContext context) {
        fillUserInfo(context);

        LotteryCheckJoinTimesParam param = LotteryCheckJoinTimesParam.builder()
                .maxTimes(context.getPhaseInfo().getMaxTimes())
                .maxType(context.getPhaseInfo().getMaxType())
                .phaseId(context.getPhaseInfo().getPhaseId())
                .platform(context.getParam().getPlatform())
                .userCreateTime(context.getUserInfo().getCreateTime())
                .userId(context.getParam().getUserId())
                .build();

        LotteryJoinStateDTO joinInfo = getLotteryParticipatorService().getJoinInfo(param);
        context.setJoinTimesInfo(joinInfo);
    }

    /**
     * 保存用户的参与记录
     *
     * @param context 参数上下文
     */
    private void saveParticipator(LotteryJoinContext context) {
        Long userId = context.getParam().getUserId();

        fillUserInfo(context);
        UserInfoDTO cacheUser = context.getUserInfo();

        Long currentJoinNum = lotteryCacheManage.incrementJoinNum(context.getPhaseInfo().getPhaseId());
        context.setCurrentJoinNum(currentJoinNum);

        LotteryParticipatorEntity entity = new LotteryParticipatorEntity();
        entity.setPhaseId(context.getPhaseInfo().getPhaseId());
        entity.setLotteryId(context.getPhaseInfo().getLotteryId());
        entity.setUserId(userId);
        entity.setNickName(cacheUser.getNickname());
        entity.setHeadImg(cacheUser.getHeadImg());
        // 夺宝券号码
        entity.setCode(String.valueOf(activityProperties.getJoinCodeStartIndex() + currentJoinNum));

        // 强制指定参与时间（用于操作开奖规则）
        if (context.getJoinTime() == null) {
            entity.setCreateTime(new Date());
        } else {
            entity.setCreateTime(context.getJoinTime());
        }

        fillParticipator(entity);


        getLotteryParticipatorService().save(entity);

        // 保存用户持有的夺宝券码，虚拟用户不需要保存
        if (!Objects.equals(LotteryJoinSourceEnum.VIRTUAL.getCode(), entity.getSource())) {
            lotteryCacheManage.addJoinCode(entity.getPhaseId(),
                    entity.getUserId(),
                    entity.getCode(),
                    entity.getSource());
        }

        lotteryPhaseMapper.addJoinNum(context.getPhaseInfo().getPhaseId());

        context.setJoinCode(entity.getCode());
    }

    /**
     * 根据当前策略进行不同的数据填充
     * 包括用户类型，参与来源，用户信息
     *
     * @param entity 用户参与数据
     */
    abstract void fillParticipator(LotteryParticipatorEntity entity);

    /**
     * 检查用户是否可以参与夺宝活动
     *
     * @param context 上下文参数
     * @return 检查结果
     */
    private LotteryJoinResultDTO check(LotteryJoinContext context) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();

        // 如果是最后一位虚拟用户，则跳过相关检查
        if (Boolean.TRUE.equals(context.getSkipTriggerDraw())) {
            return LotteryJoinResultDTO.buildDefault();
        }

        // 活动已下架或已结束，提示用户活动已结束
        boolean unexpectStatus = !Objects.equals(phaseInfo.getStatus(), LotteryPhaseStatusEnum.GOING.getCode());

        if (unexpectStatus) {
            LotteryJoinResultDTO resultDTO = buildFinishResult(phaseInfo);
            resultDTO.setLastPhaseId(phaseInfo.getLastPhaseId());
            return resultDTO;
        }

        // 增加用户参与人次
        Long joinNum = lotteryCacheManage.getJoinNum(phaseInfo.getPhaseId());
        // 通过redis中的参与人数来判断参与人数是否已达到上限
        boolean complete = joinNum + 2 > phaseInfo.getTargetNum();

        if (complete) {
            log.info("参与人数达到上限，不允许继续参与");
            return buildFinishResult(phaseInfo);
        }

        LotteryJoinResultDTO resultDTO = preCheck(context);

        if (null == resultDTO) {
            resultDTO = LotteryJoinResultDTO.buildDefault();
        }
        return resultDTO;
    }

    private LotteryJoinResultDTO buildFinishResult(LotteryPhaseCoreInfoDTO phaseInfo) {
        LotteryJoinResultDTO resultDTO = LotteryJoinResultDTO.builder()
                .joinResult(LotteryJoinResultEnum.FINISH.getCode())
                .success(false)
                .build();

        if (null != phaseInfo.getLastPhaseId()) {
            resultDTO.setLastPhaseId(phaseInfo.getLastPhaseId());
        }

        return resultDTO;
    }

    /**
     * 检查用户是否关注了商家
     * app内参与、站外参与需要检查
     *
     * @param context 参数上下文
     * @return 检查结果
     */
    LotteryJoinResultDTO checkFollow(LotteryJoinContext context) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();
        Long merchantUserId = phaseInfo.getMerchantUserId();

        // 开启了自动关注，则直接返回
        if (activityProperties.getEnableAutoFollow() || null == merchantUserId) {
            return LotteryJoinResultDTO.buildDefault();
        }

        // 如果商家关注自己则返回成功
        if (merchantUserId.equals(context.getParam().getUserId())) {
            return LotteryJoinResultDTO.buildDefault();
        }

        boolean follow = userIntegrationService.isFollow(context.getParam().getUserId(), merchantUserId);
        context.setFollowMerchantUser(follow);

        if (!follow) {
            LotteryMerchantInfoDTO merchantInfoDTO = lotteryMerchantService.get(phaseInfo.getMerchantId());

            return LotteryJoinResultDTO.builder()
                    .joinResult(LotteryJoinResultEnum.UNFOLLOW.getCode())
                    // 商家是否有福利
                    .hasAward(phaseInfo.getWelfareType() > 0)
                    // 商家信息
                    .merchantInfoDTO(merchantInfoDTO)
                    .success(true)
                    .build();
        }

        return LotteryJoinResultDTO.buildDefault();
    }
}





















