package com.bxm.localnews.merchant.service.lottery.strategy.impl;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinResultDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCoreInfoDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryParticipatorEntity;
import com.bxm.localnews.merchant.integration.UserAccountIntegrationService;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseJoinParam;
import com.bxm.localnews.merchant.service.enums.LotteryJoinResultEnum;
import com.bxm.localnews.merchant.service.enums.LotteryJoinSourceEnum;
import com.bxm.localnews.merchant.service.enums.LotteryUserTypeEnum;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Objects;

/**
 * 用户在App内参与夺宝活动
 * 扣除用户参与次数，同时需要扣除用户小红花
 *
 * @author liujia
 * @date 2020-10-27 16:04
 **/
@Component
@Slf4j
public class AppJoinStrategy extends AbstractJoinStrategy {

    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    @Override
    LotteryJoinResultDTO preCheck(LotteryJoinContext context) {
        setJoinTimesInfo(context);

        // 判断用户的参与次数是否足够
        LotteryJoinResultDTO resultDTO = checkTimes(context);

        if (resultDTO.getSuccess()) {
            resultDTO = deductGold(context);
        }

        return resultDTO;
    }

    /**
     * 检查用户参与活动的次数是否足够
     *
     * @param context 参数上下文
     * @return 检查结果
     */
    private LotteryJoinResultDTO checkTimes(LotteryJoinContext context) {
        // 获取用户的参与次数
        if (context.getJoinTimesInfo().getJoinTimes() <= 0) {
            return LotteryJoinResultDTO.builder()
                    .joinResult(LotteryJoinResultEnum.MAX_TIMES.getCode())
                    .success(false)
                    .build();
        }

        return LotteryJoinResultDTO.buildDefault();
    }

    /**
     * 扣除用户的小红花，如果扣除失败则提示数量不足
     *
     * @param context 参数上下文
     * @return 扣除结果
     */
    private LotteryJoinResultDTO deductGold(LotteryJoinContext context) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();
        if (phaseInfo.getExpendGold() <= 0) {
            return LotteryJoinResultDTO.buildDefault();
        }

        // 如果是今日注册的新用户，并且还有免费参与机会，则不扣除小红花
        if (context.getJoinTimesInfo().getNewbie()) {
            Long userId = context.getParam().getUserId();

            log.info("新用户[{}]参与夺宝活动[{}]", userId, phaseInfo.getPhaseId());

            if (context.getJoinTimesInfo().getFree()) {
                return LotteryJoinResultDTO.buildDefault();
            }
        }

        LotteryPhaseJoinParam param = context.getParam();

        boolean result = userAccountIntegrationService.deductGoldWithLottery(param.getUserId(),
                phaseInfo.getExpendGold(),
                phaseInfo.getPhaseId(),
                phaseInfo.getLotteryTitle());

        if (!result) {
            return LotteryJoinResultDTO.builder()
                    .joinResult(LotteryJoinResultEnum.GOLD_NOT_ENOUGH.getCode())
                    .success(false)
                    .build();
        }

        return LotteryJoinResultDTO.buildDefault();
    }

    @Override
    LotteryJoinResultDTO afterExecute(LotteryJoinContext context) {
        LotteryJoinResultDTO resultDTO = super.afterExecute(context);

        if (resultDTO.getSuccess()) {
            resultDTO = checkFollow(context);

            // 扣除新用户的免费参与机会
            if (context.getJoinTimesInfo().getNewbie()) {
                Long userId = context.getParam().getUserId();

                log.info("新用户[{}]参与活动[{}]成功，扣除一次免费参与机会", userId, context.getPhaseInfo().getPhaseId());

                lotteryCacheManage.consumeFreeJoinTime(userId);
            }

            // 上报用户参与参与成功
            bizLogIntegrationService.point(PointReportParam.build(context.getParam())
                    .e("3034")
                    .ev("127." + context.getPhaseInfo().getPhaseId())
                    .put("uid", String.valueOf(context.getParam().getUserId()))
                    .put("a", context.getParam().getAreaCode())
            );
        }

        return resultDTO;
    }

    @Override
    void fillParticipator(LotteryParticipatorEntity entity) {
        entity.setSource(LotteryJoinSourceEnum.APP.getCode());
        entity.setUserType(LotteryUserTypeEnum.ACTURAL.getCode());
    }

    @Override
    public boolean match(LotteryJoinContext context) {
        // 非H5的均认为是站内参与
        return null == context.getJoinSource()
                && !Objects.equals(PlatformEnum.WEB.getCode(), context.getParam().getPlatform());
    }
}
