package com.bxm.localnews.merchant.service.lottery.strategy.impl;

import com.bxm.localnews.merchant.dto.activity.LotteryJoinResultDTO;
import com.bxm.localnews.merchant.dto.activity.LotteryPhaseCoreInfoDTO;
import com.bxm.localnews.merchant.entity.lottery.LotteryParticipatorEntity;
import com.bxm.localnews.merchant.service.enums.LotteryJoinSourceEnum;
import com.bxm.localnews.merchant.service.enums.LotteryUserTypeEnum;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import com.bxm.localnews.merchant.service.lottery.event.LotteryShowEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Objects;

/**
 * 马甲号用户参与活动
 * 马甲号参与不做任何判断，只要活动是一个有效活动即可
 *
 * @author liujia
 * @date 2020-10-28 10:09
 **/
@Component
@Slf4j
public class VirtualUserJoinStrategy extends AbstractJoinStrategy {

    @Override
    void fillParticipator(LotteryParticipatorEntity entity) {
        entity.setSource(LotteryJoinSourceEnum.VIRTUAL.getCode());
        entity.setUserType(LotteryUserTypeEnum.VIRTURAL.getCode());
    }

    @Override
    public boolean match(LotteryJoinContext context) {
        return LotteryJoinSourceEnum.VIRTUAL.equals(context.getJoinSource());
    }

    @Override
    LotteryJoinResultDTO preCheck(LotteryJoinContext context) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();

        // 最后一位参见的虚拟用户跳过对应的逻辑检查
        if (Boolean.TRUE.equals(context.getSkipTriggerDraw())) {
            return super.preCheck(context);
        }

        // 活动是否允许马甲号参与
        if (Objects.equals(phaseInfo.getVirtualFlag(), 0)) {
            log.info("活动[{}]不允许马甲号参与", phaseInfo.getPhaseId());
            return LotteryJoinResultDTO.builder()
                    .success(false)
                    .build();
        }

        if (Boolean.TRUE.equals(context.getSkipActualLimit())) {
            return super.preCheck(context);
        }

        // 如果是真实用户中奖的活动，参与的真实用户数量必须达到最低值，防止被马甲号填满
        if (Objects.equals(phaseInfo.getActualFlag(), 1)) {
            int num = phaseInfo.getTargetNum() - phaseInfo.getCurrentNum();
            boolean actualCondition = num <= activityProperties.getMinimumActual();

            if (actualCondition) {
                int actualNum = getLotteryParticipatorService().countUserType(phaseInfo.getPhaseId(), LotteryUserTypeEnum.ACTURAL);

                if (num <= activityProperties.getMinimumActual() - actualNum) {
                    log.info("活动[{}]剩余参与人数必须保障最低真实用户参与，马甲号参与失败.总共人数：{},当前参与人数：{}，真实用户参与人数：{}",
                            phaseInfo.getPhaseId(),
                            phaseInfo.getTargetNum(),
                            phaseInfo.getCurrentNum(),
                            actualNum);

                    return LotteryJoinResultDTO.builder()
                            .success(false)
                            .build();
                }
            }
        }

        return super.preCheck(context);
    }

    @Override
    LotteryJoinResultDTO afterExecute(LotteryJoinContext context) {
        LotteryPhaseCoreInfoDTO phaseInfo = context.getPhaseInfo();
        // 马甲号参与成功补充一次曝光
        lotteryStatisticsService.post(new LotteryShowEvent(phaseInfo.getLotteryId(), phaseInfo.getPhaseId()));
        return super.afterExecute(context);
    }
}
