package com.bxm.localnews.merchant.service.lottery.strategy.impl;

import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.merchant.dto.activity.LotteryJoinResultDTO;
import com.bxm.localnews.merchant.entity.lottery.ActivityInviteHistoryEntity;
import com.bxm.localnews.merchant.entity.lottery.LotteryParticipatorEntity;
import com.bxm.localnews.merchant.param.activity.LotteryPhaseJoinParam;
import com.bxm.localnews.merchant.service.config.ActivityProperties;
import com.bxm.localnews.merchant.service.enums.LotteryJoinResultEnum;
import com.bxm.localnews.merchant.service.enums.LotteryJoinSourceEnum;
import com.bxm.localnews.merchant.service.enums.LotteryUserTypeEnum;
import com.bxm.localnews.merchant.service.lottery.LotteryInviteService;
import com.bxm.localnews.merchant.service.lottery.context.LotteryJoinContext;
import com.bxm.localnews.merchant.service.lottery.strategy.JoinStrategyManage;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Objects;

/**
 * 站外参与夺宝活动
 * 扣除用户的参与次数，但不扣除小红花
 *
 * @author liujia
 * @date 2020-10-27 16:26
 **/
@Component
@Slf4j
public class WebJoinStrategy extends AbstractJoinStrategy {

    @Resource
    private LotteryInviteService lotteryInviteService;

    private JoinStrategyManage joinStrategyManage;

    @Resource
    private ActivityProperties activityProperties;

    private JoinStrategyManage getJoinStrategyManage() {
        if (null == joinStrategyManage) {
            joinStrategyManage = SpringContextHolder.getBean(JoinStrategyManage.class);
        }
        return joinStrategyManage;
    }

    @Override
    public boolean match(LotteryJoinContext context) {
        return null == context.getJoinSource()
                && Objects.equals(PlatformEnum.WEB.getCode(), context.getParam().getPlatform());
    }

    @Override
    void fillParticipator(LotteryParticipatorEntity entity) {
        entity.setSource(LotteryJoinSourceEnum.WEB.getCode());
        entity.setUserType(LotteryUserTypeEnum.ACTURAL.getCode());
    }

    @Override
    LotteryJoinResultDTO preCheck(LotteryJoinContext context) {
        setJoinTimesInfo(context);

        LotteryJoinResultDTO check = checkTimes(context);

        if (check.getSuccess()) {
            return super.preCheck(context);
        }

        return check;
    }

    /**
     * 站外用户允许每日免费参与一次，不限制夺宝活动
     *
     * @param context 参数上下文
     * @return 检查结果
     */
    private LotteryJoinResultDTO checkTimes(LotteryJoinContext context) {
        // 如果站外不限制参与次数，则直接返回【默认不开启，为测试准备用】
        if (activityProperties.getNoLimitWebJoinTimes()) {
            return LotteryJoinResultDTO.buildDefault();
        }

        if (context.getJoinTimesInfo().getJoinTimes() <= 0) {
            log.info("用户[{}]已无站外参与活动次数", context.getParam().getUserId());

            return LotteryJoinResultDTO.builder()
                    .joinResult(LotteryJoinResultEnum.MAX_TIMES.getCode())
                    .success(false)
                    .build();
        }

        return LotteryJoinResultDTO.buildDefault();
    }

    @Override
    LotteryJoinResultDTO afterExecute(LotteryJoinContext context) {
        LotteryJoinResultDTO resultDTO = super.afterExecute(context);

        if (resultDTO.getSuccess()) {
            resultDTO = checkFollow(context);

            LotteryPhaseJoinParam param = context.getParam();

            // 消耗用户的全局参与次数
            lotteryCacheManage.consumeGlobalJoinTimes(param.getUserId());

            // 给分享人增加夺宝券
            addInviteCode(param, context);

            // 上报用户参与参与成功
            bizLogIntegrationService.point(PointReportParam.build(context.getParam())
                    .e("3034")
                    .ev("127." + context.getPhaseInfo().getPhaseId())
                    .put("uid", String.valueOf(context.getParam().getUserId()))
                    .put("a", context.getParam().getAreaCode())
            );
        }
        return resultDTO;
    }

    private void addInviteCode(LotteryPhaseJoinParam param, LotteryJoinContext context) {
        if (log.isDebugEnabled()) {
            log.debug("用户[{}]站外夺宝活动成功，开始给上级[{}]增加夺宝券", param.getUserId(), param.getShareUserId());
        }

        // 如果存在分享夺宝活动的用户，并且当前用户未给分享人助力过，则给分享人增加一张夺宝券
        if (null == param.getShareUserId() || Objects.equals(param.getUserId(), param.getShareUserId())) {
            if (log.isDebugEnabled()) {
                log.debug("用户[{}]站外参与夺宝活动[{}]，不存在分享人，不做处理",
                        param.getUserId(),
                        param.getPhaseId());
            }
            return;
        }

        // 分享人如果是商家，商家用户不增加额外的夺宝券
        Long merchantUserId = context.getPhaseInfo().getMerchantUserId();
        if (param.getShareUserId().equals(merchantUserId)) {
            log.info("商家[{}]分享的夺宝活动[{}]，不给商家发券", merchantUserId, param.getPhaseId());
            return;
        }

        if (activityProperties.getMatchPhaseId()) {
            // 分享的期数ID和参与的期数ID一致，才增加分享人的夺宝券（如果改需求了就找陈媛媛）
            if (!Objects.equals(param.getSourcePhaseId(), param.getPhaseId())) {
                log.info("用户[{}]站外参与的的活动[{}]与原始分享活动[{}]不一致",
                        param.getUserId(),
                        param.getPhaseId(),
                        param.getSourcePhaseId());
                return;
            }
        }

        if (!lotteryInviteService.match(param.getUserId(), param.getShareUserId(), param.getPhaseId())) {
            log.info("用户[{}]已经分享人[{}]助力过该活动[{}]",
                    param.getUserId(),
                    param.getShareUserId(),
                    param.getPhaseId());
            return;
        }

        LotteryPhaseJoinParam inviteParam = new LotteryPhaseJoinParam();
        inviteParam.setPhaseId(param.getPhaseId());
        inviteParam.setUserId(param.getShareUserId());

        LotteryJoinContext inviteContext = LotteryJoinContext.builder()
                .joinSource(LotteryJoinSourceEnum.INVITE)
                .param(inviteParam)
                .build();

        LotteryJoinResultDTO resultDTO = getJoinStrategyManage().execute(inviteContext);
        if (resultDTO.getSuccess()) {
            log.info("用户[{}]站外参与成功，给上级[{}]增加夺宝券成功", param.getUserId(), param.getShareUserId());
            // 记录邀请关系
            saveInviteHistory(context);
        }
    }

    private void saveInviteHistory(LotteryJoinContext context) {
        LotteryPhaseJoinParam param = context.getParam();

        ActivityInviteHistoryEntity entity = new ActivityInviteHistoryEntity();
        entity.setBizId(param.getPhaseId());
        entity.setUserId(param.getUserId());
        entity.setInviteUserId(param.getShareUserId());
        entity.setCreateTime(new Date());
        entity.setNickName(context.getUserInfo().getNickname());
        entity.setHeadImg(context.getUserInfo().getHeadImg());

        lotteryInviteService.save(entity);
    }
}
