package com.bxm.localnews.merchant.integration;

import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.facade.UserFeignService;
import com.bxm.localnews.merchant.param.UserFollowTypeParam;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.compress.utils.Lists;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

/**
 * @Author: pf.w
 * @Date: 2020/7/14 10:20
 */
@Slf4j
@Service
public class UserIntegrationService {

    @Resource
    private UserFeignService userFeignService;

    /**
     * 获取用户的redis缓存信息
     *
     * @param userId 用户ID
     * @return 用户信息
     */
    public UserInfoDTO getUserFromRedisDB(Long userId) {
        try {
            ResponseEntity<UserInfoDTO> response = userFeignService.getUserFromCache(userId);
            if (Objects.nonNull(response) && response.hasBody()) {
                return response.getBody();
            }

        } catch(Exception e) {
            log.error("请求用户服务失败userId: {}", userId, e);
        }

        UserInfoDTO userInfoDTO = new UserInfoDTO();
        userInfoDTO.setId(userId);
        return userInfoDTO;
    }

    /**
     * 设置用户之间的关注关系，formUserId关注toUserId
     *
     * @param formUserId 关注人
     * @param toUserId   被关注人
     */
    public void follow(Long formUserId, Long toUserId) {

        if (Objects.isNull(formUserId) || Objects.isNull(toUserId)) {
            log.warn("参数错误，无法进行关注, formUserId: {} toUserId: {} ", formUserId, toUserId);
            return;
        }

        UserFollowTypeParam param = new UserFollowTypeParam();
        param.setFromUserId(formUserId);
        param.setToUserId(toUserId);
        param.setType((byte) 0);

        try {
            userFeignService.follow(param);
        } catch(Exception e) {
            log.error("请求用户服务失败, formUserId: {} toUserId: {} ", formUserId, toUserId, e);
        }

    }

    /**
     * 判断两个用户之间是否相互关注
     *
     * @param formUserId 关注人
     * @param toUserId   被关注人
     * @return true表示已经关注，或相互关注
     */
    public Boolean isFollow(Long formUserId, Long toUserId) {

        if (Objects.isNull(formUserId) || Objects.isNull(toUserId)) {
            log.warn("参数错误，无法判断是否关注, formUserId: {} toUserId: {} ", formUserId, toUserId);
            return false;
        }

        try {
            ResponseEntity<Boolean> responseEntity = userFeignService.isFollow(formUserId, toUserId);
            if (Objects.nonNull(responseEntity) && responseEntity.hasBody()) {
                return responseEntity.getBody();
            }

        } catch(Exception e) {
            log.error("请求用户服务失败, formUserId: {} toUserId: {} ", formUserId, toUserId, e);
        }
        return false;
    }

    /**
     * 获取一定数量的虚拟用户信息
     *
     * @param num 获取数量
     * @return 虚拟用户信息
     */
    public List<UserInfoDTO> getVirtualUserList(Integer num) {
        ResponseEntity<List<UserInfoDTO>> responseEntity = userFeignService.getVirtualUserList(num);

        if (responseEntity.hasBody()) {
            return responseEntity.getBody();
        }

        return Lists.newArrayList();
    }

    public Boolean isVip(Long userId) {
        ResponseEntity<Boolean> responseEntity = userFeignService.checkUserIsVip(userId);
        if (responseEntity.hasBody()) {
            return responseEntity.getBody();
        }

        return false;
    }

    /**
     * 更新标志位
     *
     * @param userId
     * @param judgeMarker
     * @return
     */
    public Boolean updateUserJudgeMarker(Long userId, LocalNewsUserJudgeMarkerEnum judgeMarker, boolean add) {

        if (log.isDebugEnabled()) {
            log.debug("userId: {} 更新标志位judgeMarker: {} ", userId, judgeMarker);
        }

        try {
            ResponseEntity<Boolean> responseEntity = userFeignService.updateUserJudgeMarker(userId, judgeMarker, add);

            if (Objects.nonNull(responseEntity)) {
                return responseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("更新标志位失败 userId: {} judgeMarker: {}", userId, judgeMarker, e);
        }

        return false;
    }
}
