package com.bxm.localnews.merchant.security.service.impl;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.common.constant.MerchantMemberEnum;
import com.bxm.localnews.merchant.domain.security.MerchantMemberMapper;
import com.bxm.localnews.merchant.dto.CreateStatusDto;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.UserInviteHistoryBean;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.integration.UserInviteIntegrationService;
import com.bxm.localnews.merchant.security.enums.MerchantRoleEnum;
import com.bxm.localnews.merchant.security.facade.dto.MechantTeamInfosDTO;
import com.bxm.localnews.merchant.security.facade.dto.RelationshipDto;
import com.bxm.localnews.merchant.security.facade.param.CreateParam;
import com.bxm.localnews.merchant.security.facade.param.MemberTeamParam;
import com.bxm.localnews.merchant.security.facade.service.MerchantMemberFacadeService;
import com.bxm.localnews.merchant.security.facade.vo.MerchantMemberVo;
import com.bxm.localnews.merchant.security.service.MemberMessageService;
import com.bxm.localnews.merchant.security.service.MerchantAuthService;
import com.bxm.localnews.merchant.security.service.MerchantMemberService;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import static java.util.Objects.nonNull;

/**
 * @author liujia
 * @date 12/24/20 3:03 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class MerchantMemberFacadeServiceImpl implements MerchantMemberFacadeService {

    private RedisHashMapAdapter redisHashMapAdapter;

    private MerchantMemberMapper merchantMemberMapper;

    private UserInviteIntegrationService userInviteIntegrationService;

    private UserIntegrationService userIntegrationService;

    private MerchantInfoFacadeService merchantInfoFacadeService;

    private MemberMessageService memberMessageService;

    private RedisStringAdapter redisStringAdapter;

    private MerchantMemberService merchantMemberService;

    private MerchantAuthService getMerchantAuthService() {
        return SpringContextHolder.getBean(MerchantAuthService.class);
    }

    @Override
    public PageWarper<MechantTeamInfosDTO> getMemberInfo(MemberTeamParam param) {
        return merchantMemberService.getMemberList(param);
    }

    @Override
    public List<MerchantMemberVo> getAllEffectiveMember() {
        return merchantMemberMapper.getMerchantMemberList();
    }

    @Override
    public MerchantMemberVo getBoosByMerchantId(Long merchantId) {
        MerchantMemberVo bossMember = merchantMemberMapper.getBossByMerchantId(merchantId);

        if (bossMember == null) {
            bossMember = new MerchantMemberVo();
            bossMember.setUserId(0L);

            log.error("异常数据，商家[{}]不存在老板", merchantId);
        }
        return bossMember;
    }

    @Override
    public RelationshipDto getRelationship(Long userId) {

        MerchantMemberVo merchantMemberVo = getMemberInfo(userId);
        if (merchantMemberVo != null) {
            //用户是老板
            if (Objects.equals(merchantMemberVo.getRole(), MerchantRoleEnum.MEMBER_ROLE_BOSS.getRoleCode())) {
                return getRelationshipDto(0, merchantMemberVo.getMerchantId(), merchantMemberVo.getUserId(), null, null);
            }
            //用户是员工
            if (Objects.equals(merchantMemberVo.getRole(), MerchantRoleEnum.MEMBER_ROLE_EMPLOYEE.getRoleCode())) {
                return getRelationshipDto(1, merchantMemberVo.getMerchantId(), null, merchantMemberVo.getUserId(), null);
            }
        }

        //用户不是老板也不是员工  找到用户的一级师傅和二级师傅
        UserInviteHistoryBean userInviteHistoryBean = userInviteIntegrationService.selectByUserId(userId);

        if (nonNull(userInviteHistoryBean)) {
            //用户的一级师傅是老板或员工
            if (userInviteHistoryBean.getInviteUserId() != null && userInviteHistoryBean.getInviteUserId().intValue() > 0) {
                RelationshipDto relationshipDto = getBindInfo(userInviteHistoryBean.getInviteUserId());
                if (nonNull(relationshipDto)) {
                    return relationshipDto;
                }
            }
            //用户的二级师傅是老板或员工
            if (userInviteHistoryBean.getInviteSuperUserId() != null && userInviteHistoryBean.getInviteSuperUserId().intValue() > 0) {
                RelationshipDto relationshipDto = getBindInfo(userInviteHistoryBean.getInviteSuperUserId());
                if (nonNull(relationshipDto)) {
                    return relationshipDto;
                }
            }
        }

        return null;
    }

    private RelationshipDto getBindInfo(Long userId) {
        MerchantMemberVo vo = getMemberInfo(userId);
        if (vo != null) {
            //师傅是老板  师傅和老板是同一个人
            if (Objects.equals(vo.getRole(), MerchantRoleEnum.MEMBER_ROLE_BOSS.getRoleCode())) {
                return getRelationshipDto(2, vo.getMerchantId(), vo.getUserId(), null, vo.getUserId());
            }
            //师傅是员工
            if (Objects.equals(vo.getRole(), MerchantRoleEnum.MEMBER_ROLE_EMPLOYEE.getRoleCode())) {
                return getRelationshipDto(3, vo.getMerchantId(), null, vo.getUserId(), vo.getUserId());
            }
        }
        return null;
    }


    private RelationshipDto getRelationshipDto(Integer type, Long merchantId, Long bossUserId, Long employeeUserId, Long inviteSuperUserId) {
        RelationshipDto relationshipDto = new RelationshipDto();
        relationshipDto.setType(type);
        relationshipDto.setMerchantId(merchantId);
        relationshipDto.setBossUserId(bossUserId);
        relationshipDto.setEmployeeUserId(employeeUserId);
        relationshipDto.setInviteSuperUserId(inviteSuperUserId);
        return relationshipDto;
    }

    @Override
    public MerchantMemberVo getMemberInfo(Long userId) {
        return merchantMemberService.getMemberInfo(userId);
    }

    @Override
    public int memberNum(Long merchantId) {
        return merchantMemberMapper.memberNum(merchantId);
    }

    @Override
    @Async
    public void bossCreate(Long merchantId, Long userId) {
        try {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
            MerchantMemberVo merchantMemberVo = new MerchantMemberVo();
            merchantMemberVo.setMerchantId(merchantId);
            merchantMemberVo.setUserId(userId);
            if (userInfoDTO != null && userInfoDTO.getState() != null) {
                merchantMemberVo.setNickName(userInfoDTO.getNickname());
                merchantMemberVo.setHeadImg(userInfoDTO.getHeadImg());
            }
            merchantMemberVo.setRelation(MerchantMemberEnum.RELATION_TYPE_YES.getType());
            merchantMemberVo.setRole(MerchantRoleEnum.MEMBER_ROLE_BOSS.getRoleCode());
            merchantMemberVo.setCreateTime(new Date());
            merchantMemberVo.setBindTime(new Date());
            merchantMemberMapper.insertSelective(merchantMemberVo);

            // 初始化老板的相关权限
            getMerchantAuthService().initAuth(merchantId, userId, MerchantRoleEnum.MEMBER_ROLE_BOSS);

            //消息存入缓存
            redisHashMapAdapter.put(RedisConfig.MEMBER_INFO_KEY, String.valueOf(merchantMemberVo.getUserId()), merchantMemberVo);
            //记录日志
            memberMessageService.addLog(merchantId, "老板加入团队成功", null);
        } catch (Exception e) {
            log.error("老板" + userId + "加入团队失败");
        }
    }


    @Override
    public Message create(CreateParam createParam) {
        CreateStatusDto createStatusDto = new CreateStatusDto();
        MerchantMemberVo merchantMemberVo = new MerchantMemberVo();
        BeanUtils.copyProperties(createParam, merchantMemberVo);
        MerchantInfo merchantInfo = merchantInfoFacadeService.getMerchantInfo(createParam.getMerchantId());
        if (merchantInfo == null) {
            log.debug("找不到该商户信息" + createParam.getMerchantId());
            return Message.build(false, "找不到该商户信息");
        }

        //获取用户信息
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(createParam.getUserId());

        String nickname;
        String headImg;
        int register;

        if (userInfoDTO != null && userInfoDTO.getState() != null) {
            //已注册 用户信息获取
            nickname = userInfoDTO.getNickname();
            headImg = userInfoDTO.getHeadImg();
            createStatusDto.setRegister(0);
            register = 0;
        } else {
            //未注册
            nickname = createParam.getNickName();
            headImg = createParam.getHeadImg();
            createStatusDto.setRegister(1);
            register = 1;
        }

        //绑定了公司的用户  员工正在团队中
        MerchantMemberVo memberInfo = getMemberInfo(createParam.getUserId());
        if (memberInfo != null) {
            MerchantInfo vo = merchantInfoFacadeService.getMerchantInfo(memberInfo.getMerchantId());
            if (vo != null) {
                createStatusDto.setContent("您已经是“" + vo.getShortName() + "”的员工，无法绑定其他店铺");
                log.debug("您已经是“" + vo.getShortName() + "”的员工，无法绑定其他店铺");
            }
            createStatusDto.setType(1);
            createStatusDto.setRegister(register);
            return Message.build(true).addParam("createStatusDto", createStatusDto);
        }

        merchantMemberVo.setNickName(nickname);
        merchantMemberVo.setHeadImg(headImg);
        merchantMemberVo.setRelation(MerchantMemberEnum.RELATION_TYPE_YES.getType());
        merchantMemberVo.setRole(MerchantRoleEnum.MEMBER_ROLE_EMPLOYEE.getRoleCode());
        merchantMemberVo.setCreateTime(new Date());
        merchantMemberVo.setBindTime(new Date());
        merchantMemberVo.setBindPopType(0);

        //曾经绑定过  已经解绑了
        if (merchantMemberMapper.selectByMerchantAndUserId(createParam.getMerchantId(), createParam.getUserId()) != null) {
            merchantMemberVo.setCreateTime(null);
            merchantMemberVo.setModifyTime(new Date());
            //被删除过 标志位恢复
            merchantMemberVo.setDeleteFlag(0);
            merchantMemberMapper.updateByUserIdAndMerchantId(merchantMemberVo);
        } else {
            //没有绑定过
            merchantMemberMapper.insertSelective(merchantMemberVo);

            getMerchantAuthService().initAuth(createParam.getMerchantId(), createParam.getUserId(), MerchantRoleEnum.MEMBER_ROLE_EMPLOYEE);
        }

        if (!Objects.equals(createParam.getPush(), 0)) {
            //店铺老板推送
            memberMessageService.pushMerchantMemberMsg(merchantMemberVo, merchantInfo, MerchantMemberEnum.PUSH_TOBOSS_SUCESSS);
        }

        //员工推送
        memberMessageService.pushMerchantMemberMsg(merchantMemberVo, merchantInfo, MerchantMemberEnum.PUSH_TOEMPLOYEE_SUCESSS);
        //消息存入缓存
        redisHashMapAdapter.put(RedisConfig.MEMBER_INFO_KEY, String.valueOf(merchantMemberVo.getUserId()), merchantMemberVo);
        redisStringAdapter.set(RedisConfig.MERCHANT_MESSAGE_KEY.copy().appendKey(merchantInfo.getId()), nickname + "和您绑定了了员工关系");

        //记录日志
        memberMessageService.addLog(createParam.getMerchantId(), nickname + "和您绑定了了员工关系", null);

        //消息返回
        createStatusDto.setType(0);
        createStatusDto.setContent("恭喜你成为" + merchantInfo.getShortName() + "的员工");

        return Message.build(true).addParam("createStatusDto", createStatusDto);
    }

    @Override
    public void updateHeadImgAndNickName(Long userId, String headImg, String nickName) {
        merchantMemberMapper.updateHeadImgAndNickName(userId, headImg, nickName);
    }
}
