package com.bxm.localnews.merchant.security.service.impl;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.common.constant.MerchantMemberEnum;
import com.bxm.localnews.merchant.domain.security.MerchantMemberMapper;
import com.bxm.localnews.merchant.dto.MerchantUserInfoDto;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.EmployeeRemoveParam;
import com.bxm.localnews.merchant.param.RemoveRecordParam;
import com.bxm.localnews.merchant.security.entity.MemberShortInfoVo;
import com.bxm.localnews.merchant.security.facade.dto.MechantTeamInfosDTO;
import com.bxm.localnews.merchant.security.facade.param.MemberTeamParam;
import com.bxm.localnews.merchant.security.facade.vo.MerchantMemberVo;
import com.bxm.localnews.merchant.security.param.MerchantMemberPageParam;
import com.bxm.localnews.merchant.security.service.MemberMessageService;
import com.bxm.localnews.merchant.security.service.MerchantMemberService;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author liujia
 */
@Service
@AllArgsConstructor
public class MerchantMemberImpl extends BaseService implements MerchantMemberService {

    private MerchantMemberMapper merchantMemberMapper;

    private UserIntegrationService userIntegrationService;

    private RedisHashMapAdapter redisHashMapAdapter;

    private MemberMessageService memberMessageService;

    private RedisStringAdapter redisStringAdapter;

    private final MerchantInfoFacadeService merchantInfoDbService;

    @Override
    public PageWarper<MechantTeamInfosDTO> getMemberList(MemberTeamParam param) {
        List<MemberShortInfoVo> listFromDb = merchantMemberMapper.getMemberForIdAndNickname(param);
        if (org.apache.commons.collections.CollectionUtils.isEmpty(listFromDb)) {
            return new PageWarper<>();
        }

        List<MechantTeamInfosDTO> result = listFromDb.stream().map(item -> {
            MechantTeamInfosDTO mechantTeamInfosDTO = new MechantTeamInfosDTO();
            BeanUtils.copyProperties(item, mechantTeamInfosDTO);
            return mechantTeamInfosDTO;
        }).collect(Collectors.toList());

        result.forEach(bean -> {
            UserInfoDTO user = userIntegrationService.getUserFromRedisDB(bean.getUserId());
            if (Objects.nonNull(user)) {
                bean.setPhone(Objects.nonNull(user.getPhone()) ? user.getPhone() : null);
                bean.setAreaName(Objects.nonNull(user.getLocationName()) ? user.getLocationName() : null);
            }
        });
        return new PageWarper<>(result);
    }

    @Override
    public Message remove(EmployeeRemoveParam employeeRemoveParam) {
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(employeeRemoveParam.getMerchantId());
        if (merchantInfo == null) {
            return Message.build(false, "找不到该商户信息");
        }

        MerchantMemberVo merchantMemberVo = getMemberInfo(employeeRemoveParam.getUserId());
        if (merchantMemberVo == null) {
            return Message.build(false, "没有找到该用户的绑定店铺");
        }
        merchantMemberVo.setRelation(MerchantMemberEnum.RELATION_TYPE_NO.getType());
        merchantMemberVo.setModifyTime(new Date());
        merchantMemberMapper.updateByUserId(merchantMemberVo);

        memberMessageService.pushMerchantMemberMsg(merchantMemberVo, merchantInfo, MerchantMemberEnum.PUSH_TOBOSS_FAIL);
        memberMessageService.pushMerchantMemberMsg(merchantMemberVo, merchantInfo, MerchantMemberEnum.PUSH_TOEMPLOYEE_FAIL);
        memberMessageService.addLog(merchantMemberVo.getMerchantId(), merchantMemberVo.getNickName() + "和您解除了员工关系", "解除绑定");
        redisHashMapAdapter.remove(RedisConfig.MEMBER_INFO_KEY, merchantMemberVo.getUserId() + "");
        redisStringAdapter.set(RedisConfig.MERCHANT_MESSAGE_KEY.copy().appendKey(merchantMemberVo.getMerchantId()), merchantMemberVo.getNickName() + "和您解除了员工关系");
        return Message.build(true);
    }

    @Override
    public Message removeRecord(RemoveRecordParam removeRecordParam) {
        if (merchantMemberMapper.selectByPrimaryKey(removeRecordParam.getId()) == null) {
            return Message.build(false, "没有找到该条绑定记录");
        }
        MerchantMemberVo merchantMemberVo = new MerchantMemberVo();
        merchantMemberVo.setId(removeRecordParam.getId());
        merchantMemberVo.setDeleteFlag(1);
        merchantMemberVo.setDeleteTime(new Date());
        merchantMemberVo.setModifyTime(new Date());
        merchantMemberMapper.updateByPrimaryKeySelective(merchantMemberVo);
        return Message.build(true);
    }

    @Override
    public PageWarper<MerchantMemberVo> getEmployeeList(MerchantMemberPageParam merchantMemberPageParam) {
        List<MerchantMemberVo> list = merchantMemberMapper.queryListByPage(merchantMemberPageParam);
        if (!CollectionUtils.isEmpty(list)) {
            return new PageWarper<>(list);
        }
        return new PageWarper<>(Lists.newArrayList());
    }

    @Override
    public Message getMerchantUserInfo(Long userId) {
        MerchantMemberVo merchantMemberVo = getMemberInfo(userId);
        if (merchantMemberVo == null) {
            return Message.build(false, "没有找到该用户团队的相关信息");
        }
        //根据用户id查到商户id
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(merchantMemberVo.getMerchantId());
        if (merchantInfo == null) {
            return Message.build(false, "没有找到该用户绑定店铺的相关信息");
        }
        MerchantUserInfoDto merchantUserInfoDto = new MerchantUserInfoDto();
        merchantUserInfoDto.setShortName(merchantInfo.getShortName());
        merchantUserInfoDto.setHeadImg(merchantMemberVo.getHeadImg());
        merchantUserInfoDto.setNickname(merchantMemberVo.getNickName());
        merchantUserInfoDto.setCreateTime(merchantMemberVo.getBindTime());
        return Message.build(true).addParam("merchantUserInfoDto", merchantUserInfoDto);
    }

    @Override
    public MerchantMemberVo getMemberInfo(Long userId) {
        MerchantMemberVo merchantMemberVo = redisHashMapAdapter.get(RedisConfig.MEMBER_INFO_KEY, userId + "", MerchantMemberVo.class);
        if (merchantMemberVo == null) {
            //查库
            MerchantMemberVo vo = merchantMemberMapper.selectByUserId(userId);
            if (vo != null) {
                redisHashMapAdapter.put(RedisConfig.MEMBER_INFO_KEY, vo.getUserId() + "", vo);
                return vo;
            }
            return null;
        } else {
            return merchantMemberVo;
        }
    }
}















