package com.bxm.localnews.msg.controller;

import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.common.vo.SmsJson;
import com.bxm.localnews.mq.common.constant.SmsTemplateEnum;
import com.bxm.localnews.mq.common.model.dto.SendSmsResult;
import com.bxm.localnews.mq.common.model.dto.SmsSupplyDTO;
import com.bxm.localnews.msg.service.SmsSupplyService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.tools.Validater;
import com.bxm.newidea.component.util.WebUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

/**
 * 短信相关的服务接口
 *
 * @author liujia
 */
@Api(tags = "5-02 短信接口", description = "短信发送与验证接口")
@RestController
@RequestMapping("api/sms")
public class SmsController {

    @Resource
    private SmsSupplyService smsSupplyService;

    @ApiOperation(value = "5-02-1发送验证短信")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "type", value = "短信类型 1:登录（短信登录方式）" +
                    "2：绑定手机号（app绑定手机号码） " +
                    "3：注册（h5页面注册用户），" +
                    "6：忘记密码（app端忘记密码）," +
                    "7:普通手机短信验证码（蓝领，商户使用）"),
            @ApiImplicitParam(name = "phone", value = "接收验证码的手机号码")
    })
    @PostMapping("/vcode")
    public SmsJson getVerificationCode(Byte type, String phone, HttpServletRequest httpServletRequest) {
        if (type == null || StringUtils.isEmpty(phone)) {
            return SmsJson.build(ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误"));
        }
        if (!Validater.checkPhone(phone)) {
            return SmsJson.build(ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "手机号码不正确"));
        }
        SmsTemplateEnum templateEnum = SmsTemplateEnum.getTemplateByType(type);

        if (templateEnum != null) {
            SmsSupplyDTO smsSupplyDTO = new SmsSupplyDTO();
            smsSupplyDTO.setPhoneNo(phone);
            smsSupplyDTO.setSmsTemplateEnum(templateEnum);
            smsSupplyDTO.setIp(WebUtils.getIpAddr(httpServletRequest));
            SendSmsResult result = smsSupplyService.sendSmsByVCodeTemplate(phone, templateEnum,
                    WebUtils.getIpAddr(httpServletRequest));
            if (result.getState()) {
                return SmsJson.build(ResultUtil.genFailedResult(RespCode.OK, result.getMsg()));
            } else {
                return SmsJson.build(ResultUtil.genFailedResult(RespCode.BAD_REQUEST, result.getMsg()));
            }
        }

        return SmsJson.build(ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误"));
    }

    /**
     * 检查验证码是否正确
     *
     * @param type  验证码类型
     * @param phone 手机号码
     * @param code  验证码
     * @return 验证结果
     */
    @ApiOperation(value = "5-02-2短信验证码检查")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "type", value = "短信类型 1:登录（短信登录方式）" +
                    "2：绑定手机号（app绑定手机号码） " +
                    "3：注册（h5页面注册用户），" +
                    "6：忘记密码（app端忘记密码）," +
                    "7:普通手机短信验证码（蓝领，商户使用）"),
            @ApiImplicitParam(name = "phone", value = "接收短信的手机号码"),
            @ApiImplicitParam(name = "code", value = "用户填写的验证码")
    })
    @GetMapping("/check")
    public Json checkCode(Byte type, String phone, String code) {

        SmsTemplateEnum templateEnum = SmsTemplateEnum.getTemplateByType(type);
        boolean result = false;

        if (templateEnum != null) {
            result = this.smsSupplyService.verifySmsCode(phone, code, templateEnum);
        }

        if (result) {
            return ResultUtil.genFailedResult(RespCode.OK, "验证通过");
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "验证失败");
    }

    @ApiOperation(value = "5-02-3发送自定义短信")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "自定义短信内容"),
            @ApiImplicitParam(name = "phone", value = "接收验证码的手机号码")
    })
    @PostMapping("/custom")
    public SmsJson sendCustomSms(String content, String phone) {
        if (StringUtils.isBlank(content) || !Validater.checkPhone(phone)) {
            return SmsJson.build(ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误"));
        }
        SmsSupplyDTO smsSupplyDTO = new SmsSupplyDTO();
        smsSupplyDTO.setPhoneNo(phone);
        smsSupplyDTO.setContent(content);
        SendSmsResult result = smsSupplyService.sendSmsByCustomize(phone, content);
        int respCode = 0;
        if (!result.getState()) {
            respCode = RespCode.BAD_REQUEST;
        }
        return SmsJson.build(ResultUtil.genFailedResult(respCode, result.getMsg()));
    }
}