package com.bxm.localnews.msg.platform.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.push.PushResponseCallback;
import com.bxm.localnews.msg.push.Response;
import com.bxm.localnews.msg.vo.MsgUserTokenBean;
import com.oppo.push.server.Notification;
import com.oppo.push.server.Result;
import com.oppo.push.server.Sender;
import com.oppo.push.server.Target;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import java.util.concurrent.atomic.AtomicBoolean;

/**
 * oppo推送平台
 * 相关文档：https://open.oppomobile.com/wiki/doc#id=10203
 *
 * @author liujia
 * @date 2019/12/5 15:33
 */
@Component
@Slf4j
public class OppoPushPlatform extends AbstractPushPlatform implements InitializingBean {

    /**
     * oppo推送成功的状态码
     */
    private static final int SUCCESS_CODE = 0;

    private Sender sender;

    private AtomicBoolean init = new AtomicBoolean(false);

    private void initSender() {
        if (init.compareAndSet(false, true)) {
            try {
                sender = new Sender(appPushProperties.getOppoAppKey(), appPushProperties.getOppoAppSecret());
            } catch (Exception e) {
                logger.error("oppo 推送初始化失败");
                logger.error(e.getMessage(), e);
            }
        }
    }

    @Override
    public void push(PushMessage message, MsgUserTokenBean userToken, PushResponseCallback callback) {
        if (sender == null) {
            callback.apply(message, userToken, Response.fail("init failed"));
            return;
        }

        Notification notification = buildNotification(message);

        Target target = Target.build(userToken.getToken());

        try {
            Result result = sender.unicastNotification(notification, target);

            log.debug("推送结果：{},推送内容：{}", result, message);

            if (SUCCESS_CODE == result.getReturnCode().getCode()) {
                callback.apply(message, userToken, Response.success());
            } else {
                log.info("OPPO推送失败，失败结果：{},推送参数：{}", result, message);
                callback.apply(message, userToken, Response.fail(FAILED));
            }
        } catch (Exception e) {
            callback.apply(message, userToken, Response.fail(e.getMessage()));
            logger.error(e.getMessage(), e);
        }

    }

    private Notification buildNotification(PushMessage pushMessage) {

        JSONObject paramJson = new JSONObject();
        paramJson.put("message", pushMessage.getPayloadInfo().toJsonString());

        Notification notification = new Notification();
        notification.setTitle(pushMessage.getTitle());
        notification.setContent(pushMessage.getContent());
        notification.setStyle(1);
        //需要与客户端保持一致（安卓提供）
        notification.setChannelId("11111");
        notification.setAppMessageId(pushMessage.getPayloadInfo().getMsgId().toString());
        notification.setClickActionType(4);
        notification.setClickActionActivity("com.receiver.PushRouteActivity");
        notification.setActionParameters(paramJson.toJSONString());
        notification.setNetworkType(0);
        return notification;
    }

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.OPPO;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        initSender();
        if (sender == null) {
            throw new IllegalArgumentException("初始化oppo推送平台失败，请检查是否配置了对应的参数");
        }
    }
}
