package com.bxm.localnews.msg.service;

import com.bxm.localnews.mq.common.constant.PushReceiverRuleEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.msg.config.AppPushProperties;
import com.bxm.localnews.msg.strategy.IPushDistributeStrategy;
import com.bxm.localnews.msg.strategy.impl.ConditionPushDistributeStrategy;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Map;

/**
 * 执行消息具体的推送处理，根据不同的推送策略进行分发
 * 将消息分发到对应的消息队列，交由消费方进行具体的推送处理
 *
 * @author liujia
 */
@Component
@Slf4j
public class PushMessageDispatcher {

    private SequenceCreater sequenceCreater;

    private AppPushProperties appPushProperties;

    private Map<PushReceiverRuleEnum, IPushDistributeStrategy> strategyMap;

    private final ConditionPushDistributeStrategy conditionPushDistributeStrategy;

    @Autowired
    public PushMessageDispatcher(SequenceCreater sequenceCreater,
                                 AppPushProperties appPushProperties,
                                 List<IPushDistributeStrategy> pushStrategies,
                                 ConditionPushDistributeStrategy conditionPushDistributeStrategy) {
        this.sequenceCreater = sequenceCreater;
        this.appPushProperties = appPushProperties;
        this.conditionPushDistributeStrategy = conditionPushDistributeStrategy;

        strategyMap = Maps.newHashMap();
        for (IPushDistributeStrategy pushStrategy : pushStrategies) {
            strategyMap.put(pushStrategy.support(), pushStrategy);
        }
    }

    /**
     * 根据消息设定的推送范围，决定消息队列的通道与处理方式
     *
     * @param message 推送消息参数
     */
    public void push(PushMessage message) {
        //填充默认消息
        setDefaultProperty(message);

        if (log.isDebugEnabled()) {
            log.debug("消息推送开始,message:{}", message);
        }

        PushReceiveScope receiveScope = message.getPushReceiveScope();

        if (null != receiveScope) {
            IPushDistributeStrategy pushStrategy = strategyMap.getOrDefault(receiveScope.getPushReceiverRule(),
                    conditionPushDistributeStrategy);

            pushStrategy.send(message);
        } else {
            log.error("消息未设置接收对象，消息内容：{}", message);
        }

        if (log.isDebugEnabled()) {
            log.debug("消息推送到消息队列结束,messageId:{}", message);
        }

    }

    /**
     * 推送消息有必须存在的基础属性，进行补充，防止推送失败
     *
     * @param message 推送消息
     */
    private void setDefaultProperty(PushMessage message) {
        //设置标题
        if (StringUtils.isEmpty(message.getTitle())) {
            message.setTitle(appPushProperties.getDefaulPushTitle());
        }

        // 非群推消息和消息ID为空时，填充消息ID
        if (!message.isStatistics() && StringUtils.isEmpty(message.getPayloadInfo().getMsgId())) {
            long msgId = sequenceCreater.nextLongId();
            message.getPayloadInfo().addExtend("msgId", msgId);
            message.getPayloadInfo().setMsgId(msgId);
        }
    }
}
