package com.bxm.localnews.msg.service.impl;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.SendTypeEunm;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.PushMessageStatusEnum;
import com.bxm.localnews.msg.dto.AdminPushMessageDTO;
import com.bxm.localnews.msg.dto.AreaCode;
import com.bxm.localnews.msg.dto.LocationDTO;
import com.bxm.localnews.msg.dto.PushMessageStatistical;
import com.bxm.localnews.msg.integration.LocationIntegrationService;
import com.bxm.localnews.msg.param.PushMessageParam;
import com.bxm.localnews.msg.push.PushExecutor;
import com.bxm.localnews.msg.service.AdminPushMessageService;
import com.bxm.localnews.msg.service.MessageGroupCounterService;
import com.bxm.localnews.msg.service.MessageGroupService;
import com.bxm.localnews.msg.vo.AdminPushMessage;
import com.bxm.localnews.msg.vo.MsgGroupPushBean;
import com.bxm.localnews.msg.vo.MsgGroupPushCounterBean;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
public class AdminPushMessageServiceImpl implements AdminPushMessageService {

    private final MessageGroupService messageGroupService;

    private final LocationIntegrationService locationIntegrationService;

    private final PushExecutor pushExecutor;

    private final MessageGroupCounterService messageGroupCounterService;

    @Autowired
    public AdminPushMessageServiceImpl(MessageGroupService messageGroupService,
                                       LocationIntegrationService locationIntegrationService,
                                       PushExecutor pushExecutor,
                                       MessageGroupCounterService messageGroupCounterService) {
        this.messageGroupService = messageGroupService;
        this.locationIntegrationService = locationIntegrationService;
        this.pushExecutor = pushExecutor;
        this.messageGroupCounterService = messageGroupCounterService;
    }

    @SuppressWarnings("unchecked")
    @Override
    public PageWarper<AdminPushMessageDTO> queryPushMessageList(PushMessageParam pushMessageParam) {
        //V3.4.0 进行适配，相应逻辑统一迁移到群推消息中
        List<MsgGroupPushBean> groupMessageByPage = messageGroupService.getGroupMessageByPage(pushMessageParam);

        List<AdminPushMessageDTO> messageList = groupMessageByPage.stream()
                .map(this::covertPushMessage)
                .collect(Collectors.toList());

        PageWarper pageWarper = new PageWarper(groupMessageByPage);
        pageWarper.setList(messageList);
        return pageWarper;
    }

    private AdminPushMessageDTO covertPushMessage(MsgGroupPushBean groupPushBean) {
        AdminPushMessageDTO pushMessage = new AdminPushMessageDTO();
        BeanUtils.copyProperties(groupPushBean, pushMessage);

        // 不对应的字段
        pushMessage.setTiming(groupPushBean.getIsTiming());
        pushMessage.setFloatNotify(groupPushBean.getIsFloatNotify());
        pushMessage.setMute(groupPushBean.getIsMute());
        pushMessage.setVibrate(groupPushBean.getIsVibrate());
        pushMessage.setCreateUser(getAdminUserId().toString());

        if (StringUtils.isNotEmpty(pushMessage.getAreaScope())) {
            List<AreaCode> areaCodes = Lists.newArrayList();

            for (String areaCode : Splitter.on(",").split(pushMessage.getAreaScope())) {
                LocationDTO areaInfo = locationIntegrationService.getLocationByGeocode(areaCode);
                if (null != areaInfo) {
                    AreaCode area = new AreaCode();
                    area.setAreaCode(areaInfo.getCode());
                    area.setAreaName(areaInfo.getName());

                    areaCodes.add(area);
                }
            }

            pushMessage.setAreaCodeList(areaCodes);
        }
        return pushMessage;
    }

    /**
     * 将原管理后台的请求对象转换为重构后的群退对象
     *
     * @param message 推送消息
     * @return 适配结果
     */
    private MsgGroupPushBean adapter(AdminPushMessage message) {
        MsgGroupPushBean groupPushBean = new MsgGroupPushBean();
        BeanUtils.copyProperties(message, groupPushBean);
        groupPushBean.setIsFloatNotify(message.getFloatNotify());
        groupPushBean.setIsMute(message.getMute());
        groupPushBean.setIsTiming(message.getTiming());
        groupPushBean.setIsVibrate(message.getVibrate());

        groupPushBean.setCreateUserId(getAdminUserId());
        // 设置额外参数，携带到用户服务
        // groupPushBean.setExtendParams(message.get);

        return groupPushBean;
    }

    private Long getAdminUserId() {
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();

        if (requestAttributes instanceof ServletRequestAttributes) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) requestAttributes;
            HttpServletRequest request = servletRequestAttributes.getRequest();

            String currentUserId = request.getHeader("currentUserId");

            if (NumberUtils.isDigits(currentUserId)) {
                return Long.valueOf(currentUserId);
            }
        }

        return 0L;
    }

    @Override
    public AdminPushMessage getDetail(Long id) {
        MsgGroupPushBean groupPushBean = messageGroupService.get(id);
        return covertPushMessage(groupPushBean);
    }

    @Override
    public Message saveOrUpdatePushMessage(AdminPushMessage adminPushMessage) {
        Message message = Message.build();
        message = checkValidMessage(adminPushMessage, message);
        if (!message.isSuccess()) {
            return message;
        }

        MsgGroupPushBean groupBean = adapter(adminPushMessage);

        return messageGroupService.save(groupBean);
    }

    /**
     * 检测参数是否合法
     */
    private Message checkValidMessage(AdminPushMessage adminPushMessage, Message message) {
        Date now = new Date();
        if (SendTypeEunm.SEND_TIMING.getType().equals(adminPushMessage.getTiming() + "") &&
                adminPushMessage.getPushTime().getTime() < now.getTime()) {
            return message.setSuccess(false).setMessage("推送时间不能小于当前时间");
        }
        return message;
    }

    @Override
    public Message updateStatusPushMessage(Long id, Byte status) {
        PushMessageStatusEnum statusEnum = PushMessageStatusEnum.getByStatus(status);

        return messageGroupService.changeStatus(id, statusEnum);
    }

    @Override
    public Message testPushMessage(Long id, Long userId) {
        PushMessage pushMessage = messageGroupService.loadCache(id, true);

        if (null == pushMessage) {
            return Message.build(false, "消息不存在");
        }

        pushMessage.assign(userId);
        pushMessage.setPersistence(false);

        pushExecutor.push(pushMessage);

        return Message.build();
    }

    @Override
    public Message testPushMessage(Long id, String userIds) {
        String[] userIdData = StringUtils.split(userIds, ",");
        List<Long> userIdList = Stream.of(userIdData).map(Long::new).collect(Collectors.toList());

        for (Long userId : userIdList) {
            testPushMessage(id, userId);
        }

        return Message.build();
    }

    @Override
    public Message immediatelyPushMessage(Long id) {
        Message message = Message.build();
        MsgGroupPushBean groupMessage = messageGroupService.get(id);

        if (groupMessage == null) {
            return message.setSuccess(false).setMessage("该消息不存在");
        }

        if (PushMessageStatusEnum.HAS_BEEN_SEND.getType().equals(groupMessage.getStatus())) {
            return message.setSuccess(false).setMessage("该消息已推送");
        }

        if (PushMessageStatusEnum.BEING_SEND.getType().equals(groupMessage.getStatus())) {
            return message.setSuccess(false).setMessage("该消息正在推送中，请刷新后查看");
        }

        // 如果是立即推送，则直接触发
        if (SendTypeEunm.SEND_NOW.getType().equals(String.valueOf(groupMessage.getIsTiming()))) {
            groupMessage.setPushTime(new Date());
            groupMessage.setStatus(PushMessageStatusEnum.BEING_SEND.getType());
            messageGroupService.pushNow(groupMessage);
        }

        // 如果是定时任务，则修改状态并创建定时任务
        if (SendTypeEunm.SEND_TIMING.getType().equals(String.valueOf(groupMessage.getIsTiming()))) {
            messageGroupService.createTimer(groupMessage);
            messageGroupService.changeStatus(id, PushMessageStatusEnum.STAY_SEND);
        }

        return Message.build();
    }

    @Override
    public Collection<PushMessageStatistical> getStatistical(Long id) {
        MsgGroupPushBean pushMessage = messageGroupService.get(id);

        List<MsgGroupPushCounterBean> messageCounterList = messageGroupCounterService.getMessageCounter(id);

        Map<String, PushMessageStatistical> tempMap = Maps.newHashMap();

        for (MsgGroupPushCounterBean counter : messageCounterList) {
            PlatformTypeEnum platform = PlatformTypeEnum.getByCode(counter.getPushType());

            PushMessageStatistical statistical = tempMap.get(platform.getLabel());
            if (statistical == null) {
                statistical = new PushMessageStatistical();
                statistical.setPushPlatform(platform.getLabel());
            }

            statistical.addOpenTotal(counter.getCallback());
            statistical.addPushSuccess(counter.getSuccess());
            statistical.addPushFail(counter.getFail());
            statistical.addUserTotal(counter.getSuccess(), counter.getFail());

            tempMap.put(platform.getLabel(), statistical);
        }

        if (tempMap.size() == 0) {
            return buildEmptyResult();
        }

        List<PushMessageStatistical> finalResult = Lists.newArrayList();
        finalResult.addAll(tempMap.values());
        finalResult.add(total(pushMessage, finalResult));

        long index = 1L;
        for (PushMessageStatistical statistical : finalResult) {
            statistical.setId(index++);
        }

        return finalResult;
    }

    private List<PushMessageStatistical> buildEmptyResult() {
        List<PushMessageStatistical> emptyList = Lists.newArrayList();
        //组装平台数据
        PlatformTypeEnum[] platforms = PlatformTypeEnum.values();

        long index = 1L;

        for (PlatformTypeEnum platform : platforms) {
            if (platform.isApp()) {
                PushMessageStatistical pushMessageStatistical = new PushMessageStatistical();
                pushMessageStatistical.setId(index++);
                pushMessageStatistical.setPushPlatform(platform.getLabel());
                pushMessageStatistical.setOpenRate(0d);
                pushMessageStatistical.setUserTotal(0L);
                pushMessageStatistical.setPushSuccess(0L);
                pushMessageStatistical.setOpenTotal(0L);
                emptyList.add(pushMessageStatistical);
            }
        }
        return emptyList;
    }

    private PushMessageStatistical total(MsgGroupPushBean pushMessage, Collection<PushMessageStatistical> result) {
        PushMessageStatistical calcTotal = new PushMessageStatistical();
        calcTotal.setPushPlatform("ALL");
        Long openTotal = 0L;
        Long successTotal = 0L;
        Long failTotal = 0L;

        for (PushMessageStatistical pms : result) {
            failTotal += pms.getPushFail();
            openTotal += pms.getOpenTotal();
            successTotal += pms.getPushSuccess();
        }
        calcTotal.setUserTotal(Long.valueOf(pushMessage.getUserTotal()));
        calcTotal.setOpenTotal(openTotal);
        calcTotal.setPushSuccess(successTotal);
        calcTotal.setPushFail(failTotal);

        if (calcTotal.getUserTotal() == 0L) {
            calcTotal.setOpenRate(0D);
        } else if (successTotal == 0L) {
            calcTotal.setOpenRate(1D);
        } else {
            calcTotal.setOpenRate((double) openTotal / successTotal);
        }

        return calcTotal;
    }
}
