package com.bxm.localnews.msg.service.impl;

import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.domain.MsgGroupPushCounterMapper;
import com.bxm.localnews.msg.service.MessageGroupCounterService;
import com.bxm.localnews.msg.vo.MsgGroupPushCounterBean;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

@Service
@Slf4j
public class MessageGroupCounterServiceImpl implements MessageGroupCounterService {

    private final MsgGroupPushCounterMapper msgGroupPushCounterMapper;

    private final SequenceCreater sequenceCreater;

    /**
     * 初始化标记，在实际需要使用到消费队列的时候再初始化
     */
    private AtomicBoolean init = new AtomicBoolean(false);

    /**
     * 统计更新数据的缓存队列
     * 减少数据写入次数，会丢失一定的数据
     */
    private LinkedBlockingQueue<MsgGroupPushCounterBean> counterQueue = new LinkedBlockingQueue<>(10000);

    @Autowired
    public MessageGroupCounterServiceImpl(MsgGroupPushCounterMapper msgGroupPushCounterMapper,
                                          SequenceCreater sequenceCreater) {
        this.msgGroupPushCounterMapper = msgGroupPushCounterMapper;
        this.sequenceCreater = sequenceCreater;
    }

    private void initConsumerThread() {
        ScheduledExecutorService executor = Executors.newScheduledThreadPool(1,
                new NamedThreadFactory("consumer-counter", true));

        executor.scheduleAtFixedRate(this::execBatchSave, 500, 500, TimeUnit.MILLISECONDS);
    }

    private void execBatchSave() {
        try {
            Multimap<Long, MsgGroupPushCounterBean> counterMap = ArrayListMultimap.create();

            int batchSize = 0;
            while (CollectionUtils.isNotEmpty(counterQueue)) {
                MsgGroupPushCounterBean counter = counterQueue.poll();

                if (null != counter) {
                    Collection<MsgGroupPushCounterBean> counterList = counterMap.get(counter.getGroupMsgId());

                    boolean hit = false;

                    if (CollectionUtils.isNotEmpty(counterList)) {
                        // 如果存在相同消息的相同推送平台，则进行数据累加。批量推送，大部分数据会是相同ID，相同平台
                        for (MsgGroupPushCounterBean subCounter : counterList) {
                            if (Objects.equals(subCounter.getGroupMsgId(), counter.getGroupMsgId()) &&
                                    Objects.equals(subCounter.getPushType(), counter.getPushType())) {
                                subCounter.setFail(sum(counter.getFail(), subCounter.getFail()));
                                subCounter.setSuccess(sum(counter.getSuccess(), subCounter.getSuccess()));
                                subCounter.setCallback(sum(counter.getCallback(), subCounter.getCallback()));

                                hit = true;
                            }
                        }
                    }

                    if (!hit) {
                        counterMap.put(counter.getGroupMsgId(), counter);
                    }

                    if (batchSize > 1000) {
                        saveBatch(counterMap);
                        counterMap = ArrayListMultimap.create();
                        break;
                    }

                    batchSize++;
                }
            }

            if (counterMap.size() > 0) {
                saveBatch(counterMap);
            }
        } catch (Exception e) {
            // 拦截所有异常，防止线程中断
            log.error("消费群推统计数据异常，当前队列剩余数量：{}", counterQueue.size());
            log.error(e.getMessage(), e);
        }
    }

    private void saveBatch(Multimap<Long, MsgGroupPushCounterBean> counterMap) {

        for (MsgGroupPushCounterBean counter : counterMap.values()) {
            log.info("写入一次统计数据，msgId:{},success：{},fail:{},callback:{}",
                    counter.getGroupMsgId(),
                    counter.getSuccess(),
                    counter.getFail(),
                    counter.getCallback());

            msgGroupPushCounterMapper.addCounter(counter);
        }
    }

    private int sum(Integer num, Integer subNum) {
        if (num == null) {
            num = 0;
        }
        if (subNum == null) {
            subNum = 0;
        }
        return num + subNum;
    }

    @Override
    public void saveCounter(PushMessage message) {
        log.info("初始化消息统计数据，msgId:{}", message.getMsgId());
        // 初始化统计数据
        msgGroupPushCounterMapper.deleteByMsgId(message.getMsgId());

        for (PlatformTypeEnum platformType : PlatformTypeEnum.values()) {
            if (platformType.isApp()) {
                MsgGroupPushCounterBean pushCounter = MsgGroupPushCounterBean.builder()
                        .id(sequenceCreater.nextLongId())
                        .groupMsgId(message.getMsgId())
                        .callback(0)
                        .fail(0)
                        .success(0)
                        .pushType(platformType.getCode())
                        .createTime(new Date())
                        .build();

                msgGroupPushCounterMapper.insert(pushCounter);
            }
        }
    }

    @Override
    public void addCounter(MsgGroupPushCounterBean counter) {
        if (init.compareAndSet(false, true)) {
            initConsumerThread();
        }

        if (!counterQueue.offer(counter)) {
            log.error("队列已满，丢弃写入数据：{}", counter);
        }
    }

    @Override
    public List<MsgGroupPushCounterBean> getMessageCounter(Long msgId) {
        return msgGroupPushCounterMapper.getMesageCounter(msgId);
    }
}
