package com.bxm.localnews.msg.service.impl;

import com.bxm.localnews.mq.common.constant.SendTypeEunm;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.msg.config.PushMessageStatusEnum;
import com.bxm.localnews.msg.constant.PushMsgConstant;
import com.bxm.localnews.msg.domain.MsgGroupPushMapper;
import com.bxm.localnews.msg.param.PushMessageParam;
import com.bxm.localnews.msg.param.PushMsgBuildParam;
import com.bxm.localnews.msg.service.MessageGroupService;
import com.bxm.localnews.msg.service.PushMessageDispatcher;
import com.bxm.localnews.msg.timer.PushMessageTask;
import com.bxm.localnews.msg.vo.MsgGroupPushBean;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.bxm.newidea.component.vo.Message;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

import static com.bxm.localnews.msg.constant.RedisConfig.BLACK_MSG_ID;
import static com.bxm.localnews.msg.constant.RedisConfig.INCREMENT_STATISTICS_MSG_ID;

/**
 * @author liujia
 */
@Service
@Slf4j
public class MessageGroupServiceImpl implements MessageGroupService {

    private final MsgGroupPushMapper msgGroupPushMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final PushMessageHandler pushMessageHandler;

    private final ScheduleService scheduleService;

    private final PushMessageTask pushMessageTask;

    private final PushMessageDispatcher dispatcher;

    /**
     * 推送消息本地缓存
     */
    private LoadingCache<Long, PushMessage> cache;

    /**
     * 黑名单的本地缓存
     */
    private LoadingCache<Long, Boolean> blackCache;

    @Autowired
    public MessageGroupServiceImpl(MsgGroupPushMapper msgGroupPushMapper,
                                   RedisStringAdapter redisStringAdapter,
                                   PushMessageHandler pushMessageHandler,
                                   ScheduleService scheduleService,
                                   PushMessageTask pushMessageTask,
                                   PushMessageDispatcher dispatcher) {
        this.msgGroupPushMapper = msgGroupPushMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.pushMessageHandler = pushMessageHandler;
        this.scheduleService = scheduleService;
        this.pushMessageTask = pushMessageTask;
        this.dispatcher = dispatcher;

        initLocalCache();
    }

    @Override
    public Message save(MsgGroupPushBean groupPushBean) {
        Date now = new Date();
        int result;

        if (groupPushBean.getId() == null) {
            groupPushBean.setId(incrementMsgId());

            groupPushBean.setCreateTime(now);
            groupPushBean.setUserTotal(0);
            groupPushBean.setModifyTime(now);
            //消息有效期默认24小时
            if (groupPushBean.getPeriodTime() == null) {
                groupPushBean.setPeriodTime(24);
            }
            groupPushBean.setStatus(PushMessageStatusEnum.STAY_CONFIRM.getType());

            result = msgGroupPushMapper.insert(groupPushBean);
        } else {
            MsgGroupPushBean existsGroupMsg = msgGroupPushMapper.selectByPrimaryKey(groupPushBean.getId());
            if (PushMessageStatusEnum.HAS_BEEN_SEND.getType().equals(existsGroupMsg.getStatus())
                    || PushMessageStatusEnum.BEING_SEND.getType().equals(existsGroupMsg.getStatus())) {
                return Message.build(false, "消息推送中或已推送，无法编辑");
            }

            groupPushBean.setModifyTime(now);
            result = msgGroupPushMapper.updateByPrimaryKeySelective(groupPushBean);
        }


        // 立即发送，后续需要增加审核与二次确认的流程
        if (SendTypeEunm.SEND_NOW.getType().equals(String.valueOf(groupPushBean.getIsTiming()))) {
            removeTimer(groupPushBean.getId());
            removeBlack(groupPushBean.getId());
        } else if (SendTypeEunm.SEND_TIMING.getType().equals(String.valueOf(groupPushBean.getIsTiming()))) {
            removeBlack(groupPushBean.getId());
        }

        return Message.build(result);
    }

    @Override
    public Message pushNow(MsgGroupPushBean groupPushBean) {
        dispatcher.push(pushMessageHandler.generateMessage(groupPushBean));
        // 用户主动触发推送时，移除黑名单
        removeBlack(groupPushBean.getId());

        MsgGroupPushBean modifyPushInfo = new MsgGroupPushBean();
        modifyPushInfo.setId(groupPushBean.getId());
        modifyPushInfo.setPushTime(new Date());
        modifyPushInfo.setStatus(groupPushBean.getStatus());

        msgGroupPushMapper.updateByPrimaryKeySelective(groupPushBean);

        return Message.build();
    }

    @Override
    public void createTimer(MsgGroupPushBean entity) {
        PushMsgBuildParam timerParam = new PushMsgBuildParam();
        timerParam.setMessageId(entity.getId());
        timerParam.setPushMessage(pushMessageHandler.generateMessage(entity));
        timerParam.setStartTime(entity.getPushTime());

        removeTimer(entity.getId());

        scheduleService.push(OnceTaskBuilder.builder(
                PushMessageTask.generateTaskName(entity.getId()),
                timerParam.getStartTime(),
                pushMessageTask)
                .callbackParam(timerParam)
                .build());
    }

    private void removeTimer(Long msgId) {
        String taskName = PushMessageTask.generateTaskName(msgId);

        log.info("移除定时任务:{}", taskName);

        scheduleService.remove(taskName);
    }

    @Override
    public Long incrementMsgId() {
        return redisStringAdapter.incrementWithDefault(INCREMENT_STATISTICS_MSG_ID,
                PushMsgConstant.MIN_STATISTICS_MSG_ID);
    }

    @Override
    public boolean isGroupMsg(Long msgId) {
        if (msgId == null) {
            return false;
        }
        return msgId > PushMsgConstant.MIN_STATISTICS_MSG_ID && msgId < PushMsgConstant.MAX_STATISTICS_MSG_ID;
    }

    private void removeBlack(Long msgId) {
        redisStringAdapter.remove(BLACK_MSG_ID.copy().appendKey(msgId));
    }

    private void addBlack(Long msgId) {
        // 加入黑名单，召回正在推送的消息
        redisStringAdapter.set(BLACK_MSG_ID.copy().appendKey(msgId), Boolean.TRUE, 24 * 3600L);
    }

    @Override
    public boolean isBlackMsgId(Long msgId) {
        Boolean hit = blackCache.getUnchecked(msgId);
        return Boolean.TRUE.equals(hit);
    }

    @Override
    public Message changeStatus(Long msgId, PushMessageStatusEnum status) {
        if (null == status) {
            return Message.build(false, "变更状态不存在");
        }

        if (null == msgId) {
            return Message.build(false, "该消息不存在");
        }

        MsgGroupPushBean updateEntity = new MsgGroupPushBean();
        updateEntity.setId(msgId);
        updateEntity.setStatus(status.getType());

        // 如果是取消推送则取消定时器、增加黑名单
        if (PushMessageStatusEnum.CANCEL.equals(status)) {
            removeTimer(msgId);
            addBlack(msgId);
        }

        // 如果是待确认，则移除定时器
        if (PushMessageStatusEnum.STAY_CONFIRM.equals(status)) {
            removeTimer(msgId);
        }

        // 如果是待发送，则移除黑名单
        if (PushMessageStatusEnum.STAY_SEND.equals(status)) {
            removeBlack(msgId);
        }

        int result = msgGroupPushMapper.updateByPrimaryKeySelective(updateEntity);

        cache.refresh(msgId);
        return Message.build(result);
    }

    @Override
    public PushMessage loadCache(Long msgId, boolean forceRefresh) {
        if (forceRefresh) {
            cache.refresh(msgId);
        }
        return cache.getUnchecked(msgId);
    }

    @Override
    public MsgGroupPushBean get(Long msgId) {
        return msgGroupPushMapper.selectByPrimaryKey(msgId);
    }

    @Override
    public List<MsgGroupPushBean> getGroupMessageByPage(PushMessageParam param) {
        return msgGroupPushMapper.selectByPage(param);
    }

    @Override
    public void addPushTotal(Long msgId, int userTotal) {
        msgGroupPushMapper.addUserTotal(msgId, userTotal);
    }

    private void initLocalCache() {
        cache = CacheBuilder.newBuilder()
                .maximumSize(100)
                .expireAfterAccess(5, TimeUnit.SECONDS)
                .build(new CacheLoader<Long, PushMessage>() {
                    @Override
                    public PushMessage load(Long key) {
                        MsgGroupPushBean msgGroupPushBean = msgGroupPushMapper.selectByPrimaryKey(key);

                        if (null != msgGroupPushBean) {
                            return pushMessageHandler.generateMessage(msgGroupPushBean);
                        }
                        return null;
                    }
                });

        blackCache = CacheBuilder.newBuilder()
                .maximumSize(1000)
                .expireAfterAccess(5, TimeUnit.SECONDS)
                .build(new CacheLoader<Long, Boolean>() {
                    @Override
                    public Boolean load(Long key) {
                        return Boolean.TRUE.equals(redisStringAdapter.get(BLACK_MSG_ID.copy().appendKey(key), Boolean.class));
                    }
                });
    }
}

















