package com.bxm.localnews.msg.service.impl;

import com.bxm.localnews.mq.common.constant.MessageTypeEnum;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.msg.config.MessageTypeProperties;
import com.bxm.localnews.msg.service.PushMessageDispatcher;
import com.bxm.localnews.msg.service.PushMsgSupplyService;
import com.bxm.localnews.msg.timer.PushMessageTask;
import com.bxm.localnews.msg.utils.IntervalPeriodUtils;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Map;

import static com.bxm.localnews.msg.constant.RedisConfig.INTERACTION_NUM;
import static com.bxm.localnews.msg.constant.RedisConfig.LAST_INTERACTION_MSG;
import static org.apache.commons.lang.StringUtils.isBlank;

@Service("pushMsgSupplyService")
@Slf4j
public class PushMsgSupplyServiceImpl extends BaseService implements PushMsgSupplyService {

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private ScheduleService scheduleService;

    @Resource
    private PushMessageTask pushMessageTask;

    @Resource
    private MessageTypeProperties messageTypeProperties;

    @Resource
    private PushMessageDispatcher pushMessageDispatcher;

    @Override
    public void pushMsg(PushMessage message) {
        pushMessageDispatcher.push(message);
    }

    @Override
    public void pushInteraction() {
        int index = IntervalPeriodUtils.getPreIntervalIndex();

        //获取未读消息数量
        Map<String, Long> intervalNumMap = Maps.newHashMap();
        KeyGenerator key;

        for (MessageTypeEnum messageType : messageTypeProperties.getInteractionType()) {
            key = INTERACTION_NUM.copy().appendKey(index).appendKey(messageType.name());
            //不同类型下的用户与未读数量MAP
            Map<String, Long> typeMap = redisHashMapAdapter.entries(key, Long.class);

            if (null != typeMap) {
                typeMap.forEach((userIdStr, num) -> intervalNumMap.merge(userIdStr, num, Long::sum));
            }
        }

        if (intervalNumMap.isEmpty()) {
            log.info("{}没有发生用户互动消息", DateUtils.formatDateTime(new Date()));
            return;
        }

        //获取最后未读消息
        Map<String, PushMessage> lastMsgMap = Maps.newHashMap();

        for (MessageTypeEnum messageType : messageTypeProperties.getInteractionType()) {
            key = LAST_INTERACTION_MSG.copy().appendKey(index).appendKey(messageType.name());
            Map<String, PushMessage> typeMap = redisHashMapAdapter.entries(key, PushMessage.class);

            if (null != typeMap) {
                lastMsgMap.putAll(typeMap);
            }
        }

        intervalNumMap.forEach((userIdStr, num) -> {
            PushMessage message = lastMsgMap.get(userIdStr);

            if (num == 0 || null == message) {
                log.info("未读消息数量为0，或消息为null.用户ID：[{}]", userIdStr);
            } else {
                Long userId = Long.valueOf(userIdStr);
                if (num == 1) {
                    pushSingle(message, userId);
                } else {
                    pushSum(message, userId, num);
                }
            }
        });
    }

    /**
     * 用户仅有一条互动消息时
     *
     * @param message 互动消息
     * @param userId  对应用户
     */
    private void pushSingle(PushMessage message, Long userId) {
        PushPayloadInfo payloadInfo = message.getPayloadInfo();

        PushMessage pushMessage = PushMessage.build();
        pushMessage.setTitle(message.getTitle());
        if (isBlank(message.getContent())) {
            pushMessage.setContent(message.getTitle());
        } else {
            pushMessage.setContent(message.getContent());
        }
        pushMessage.setType(TemplateTypeEnum.NOTIFCTION);
        pushMessage.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
        pushMessage.setPayloadInfo(payloadInfo);
        pushMessage.setPersistence(false);

        pushMsg(pushMessage);
    }

    /**
     * 用户有多条互动消息时，进行消息合并
     *
     * @param message 最后一条互动消息
     * @param userId  对应用户
     * @param total   互动消息总数
     */
    private void pushSum(PushMessage message, Long userId, Long total) {
        PushPayloadInfo payloadInfo = message.getPayloadInfo();
        String name = payloadInfo.getExtend().get("nickname").toString();
        String title = name + " 等用户跟你有" + total + "条互动消息哦，快来看看吧";

        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.INTERACTION_MSG);
        info.setTitle(title);

        PushMessage pushMessage = PushMessage.build();
        pushMessage.setTitle("有人和你打招呼，快来看看吧");
        pushMessage.setContent(title);
        pushMessage.setType(TemplateTypeEnum.NOTIFCTION);
        pushMessage.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
        pushMessage.setPayloadInfo(info);
        //2.5.0 合并推送不显示在消息中心
        pushMessage.setPersistence(false);

        pushMsg(pushMessage);
    }
}
