package com.bxm.localnews.msg.strategy.impl;

import com.bxm.localnews.mq.common.constant.PushReceiverRuleEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.mq.common.param.UserSearchPageParam;
import com.bxm.localnews.msg.config.PushMessageStatusEnum;
import com.bxm.localnews.msg.integration.UserIntegrationService;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.newidea.component.vo.PageWarper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 通过设置检索条件，批量进行消息推送
 *
 * @author liujia
 * @date 2020-07-04 12:21
 **/
@Component
@Slf4j
public class ConditionPushDistributeStrategy extends AbstractPushDistributeStrategy {

    @Resource
    private UserIntegrationService userIntegrationService;

    private ThreadPoolTaskExecutor executor;

    @Override
    public PushReceiverRuleEnum support() {
        return PushReceiverRuleEnum.CONDITION;
    }

    @Override
    public void send(PushMessage message) {
        //条件推送默认均进行统计
        execPreprocess(message);

        PageWarper<Long> page = queryAndDeal(message, 1);

        //包含较多的分页数据时，多线程获取
        if (page.isHasNextPage()) {
            if (null == executor) {
                initThreadPool();
            }

            AtomicInteger pageNum = new AtomicInteger(1);

            for (int i = 0; i < page.getPages() - 1; i++) {
                executor.execute(() -> queryAndDeal(message, pageNum.incrementAndGet()));
            }
        }
    }

    private PageWarper<Long> queryAndDeal(PushMessage message, int pageNum) {
        PushReceiveScope pushReceiveScope = message.getPushReceiveScope();
        UserSearchPageParam pageParam = UserSearchPageParam.builder()
                .pushReceiverRule(pushReceiveScope.getPushReceiverRule())
                .ruleParam(pushReceiveScope.getRuleParam())
                .build();

        int pageSize = appPushProperties.getBucketNum() * appPushProperties.getPullMultiple();

        pageParam.setPageSize(pageSize);
        pageParam.setPageNum(pageNum);

        PageWarper<Long> page = userIntegrationService.queryUserByPage(pageParam);

        List<Long> result = page.getList();
        int queryNum = CollectionUtils.isEmpty(result) ? 0 : result.size();

        if (log.isDebugEnabled()) {
            log.debug("分页获取推送目标数据，每页数量:{}，当前页:{}，获取用户数量：{}", pageSize, pageNum, queryNum);
        }

        if (queryNum > 0) {
            splitTask(result, message.getMsgId(), pageNum, page.getPages());
        }
        if (queryNum < pageSize) {
            // 查询不到数据，更新为推送完成状态，实际数据还在异步推送
            getMessageGroupService().changeStatus(message.getPayloadInfo().getMsgId(), PushMessageStatusEnum.HAS_BEEN_SEND);
        }

        return page;
    }

    private void initThreadPool() {
        executor = new ThreadPoolTaskExecutor();
        executor.setCorePoolSize(2);
        executor.setQueueCapacity(10000);
        executor.setMaxPoolSize(10);
        executor.setKeepAliveSeconds(10);
        executor.setRejectedExecutionHandler(new ThreadPoolExecutor.AbortPolicy());
        executor.setThreadFactory(new NamedThreadFactory("batch-page-push"));

        executor.initialize();
    }
}
