package com.bxm.localnews.msg.config;

import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.msg.constant.MessageTypeEnum;
import com.bxm.localnews.msg.dto.MessageType;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

@Component
@ConfigurationProperties("message.config.message")
@Data
@Slf4j
public class MessageTypeProperties {

    /**
     * 消息中心栏目的配置信息，包括标签、图标、包含的子类型、排序等
     * key：类型名称，无用
     * value：类型详情
     */
    private Map<String, MesssageTypeConfigInfo> msgTypeConfig = Maps.newHashMap();

    /**
     * 小程序需要推送模板消息的类型
     */
    private List<Integer> template = Lists.newArrayList();

    /**
     * 将推送的消息类型转换为消息中心对应的栏目
     *
     * @param pushMsgType 推送消息类型
     * @return 消息中心栏目类型，详见{@link MessageTypeEnum}
     */
    public String convertType(Object pushMsgType) {
        String type = Objects.toString(pushMsgType);

        if (String.valueOf(PushMessageEnum.ADD_FUNS.getType()).equals(type)) {
            return PushMessageEnum.ADD_FUNS.name();
        }

        if (String.valueOf(PushMessageEnum.IM_MESSAGE.getType()).equals(type)) {
            return PushMessageEnum.IM_MESSAGE.name();
        }

        if (NumberUtils.isDigits(type)) {
            for (MesssageTypeConfigInfo msgType : msgTypeConfig.values()) {
                if (msgType.getSubTypes().contains(type)) {
                    return msgType.getType();
                }
            }
        }

        return pushMsgType.toString();
    }

    /**
     * 获取配置的消息栏目类型信息，根据排序返回
     *
     * @return 消息栏目类型
     */
    public List<String> getMessageTypes() {
        List<MesssageTypeConfigInfo> configInfos = msgTypeConfig.values().stream()
                .sorted(Comparator.comparingInt(MesssageTypeConfigInfo::getIndex))
                .collect(Collectors.toList());

        return configInfos.stream().map(MesssageTypeConfigInfo::getType).collect(Collectors.toList());
    }

    /**
     * 获取消息栏目中的聚合类型对应的子类型列表
     *
     * @param msgType 消息栏目
     * @return 子类型列表
     */
    public List<String> getSubTypes(String msgType) {
        for (MesssageTypeConfigInfo configInfo : msgTypeConfig.values()) {
            if (Objects.equals(msgType, configInfo.getType())) {
                return configInfo.getSubTypes();
            }
        }

        return Lists.newArrayList();
    }

    public MessageType getMessageInfo(String msgType) {
        for (MesssageTypeConfigInfo configInfo : msgTypeConfig.values()) {
            if (Objects.equals(configInfo.getType(), msgType)) {
                return convert(configInfo);
            }
        }

        log.warn("未配置的消息类型[{}]", msgType);
        return new MessageType();
    }

    /**
     * 将配置的消息类型转换为用户消息中心显示的歌声
     *
     * @param configInfo 配置消息对象
     * @return 消息中心分类对象
     */
    private MessageType convert(MesssageTypeConfigInfo configInfo) {
        MessageType messageType = new MessageType();
        messageType.setImg(configInfo.getIcon());
        messageType.setTitle(configInfo.getLabel());
        messageType.setMessageType(configInfo.getType());
        return messageType;
    }

    /**
     * 消息类型是否为互动消息
     *
     * @param msgType 消息类型
     * @return true表示为互动消息
     */
    public boolean isInteraction(String msgType) {
        return MessageTypeEnum.COMMENT.name().equals(msgType)
                || MessageTypeEnum.LIKE.name().equals(msgType)
                || MessageTypeEnum.INVITE.name().equals(msgType)
                || MessageTypeEnum.SHARE.name().equals(msgType);
    }

    /**
     * 互动消息的消息类型
     *
     * @return 消息列表
     */
    public Set<MessageTypeEnum> getInteractionType() {
        return ImmutableSet.of(MessageTypeEnum.COMMENT, MessageTypeEnum.LIKE,
                MessageTypeEnum.INVITE, MessageTypeEnum.SHARE);
    }
}
