package com.bxm.localnews.msg.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.util.DateFormatUtils;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.mq.common.constant.*;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.msg.config.MessageTypeProperties;
import com.bxm.localnews.msg.constant.MessageTypeEnum;
import com.bxm.localnews.msg.domain.MessageMapper;
import com.bxm.localnews.msg.dto.MessageType;
import com.bxm.localnews.msg.dto.MessageTypeDTO;
import com.bxm.localnews.msg.event.UserEventService;
import com.bxm.localnews.msg.integration.AdvertIntegrationService;
import com.bxm.localnews.msg.integration.AppVersionIntegrationService;
import com.bxm.localnews.msg.integration.PushTemplateMessageIntegrationService;
import com.bxm.localnews.msg.param.MessageListParam;
import com.bxm.localnews.msg.service.UserMessageService;
import com.bxm.localnews.msg.utils.IntervalPeriodUtils;
import com.bxm.localnews.msg.vo.Tuple;
import com.bxm.localnews.msg.vo.UserMessageBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.math.NumberUtils;
import org.joda.time.DateTimeComparator;
import org.springframework.beans.BeanUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.bxm.localnews.mq.common.constant.RedisConfig.*;
import static com.bxm.localnews.msg.constant.MessageTypeEnum.*;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

@Service
@Transactional(rollbackFor = Exception.class)
@RefreshScope
@Slf4j
public class UserMessageServiceImpl implements UserMessageService {

    @Resource
    private MessageMapper messageMapper;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private AdvertIntegrationService advertIntegrationService;

    @Resource
    private MessageTypeProperties messageTypeProperties;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private UserEventService userEventService;

    private DateTimeComparator dateTimeComparator = DateTimeComparator.getInstance();

    @Resource
    private PushTemplateMessageIntegrationService pushTemplateMessageIntegrationService;

    @Resource
    private SequenceCreater sequenceCreater;

    @Override
    public int getUnReadMsg(Long userId, Integer platform, String curVer) {
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);


        if (platform == PlatformEnum.APPLET.getCode()) {
            unreadNumMap.remove(MessageTypeEnum.BALANCE.name());
        } else {
            unreadNumMap.remove(MessageTypeEnum.INVITE.name());
        }

        Long unread;

        BasicParam param = new BasicParam();
        param.setCurVer(curVer);
        //3.0.0之后的版本，私聊不纳入到消息总数
        if (null != curVer && appVersionIntegrationService.afterMsgCenterRebuild(param)) {
            unread = 0L;
            for (Map.Entry<String, Long> entry : unreadNumMap.entrySet()) {
                if (!NumberUtils.isDigits(entry.getKey())) {
                    unread += entry.getValue();
                }
            }
        } else {
            unread = unreadNumMap.values().stream().mapToLong((v) -> v).sum();
        }

        if (log.isDebugEnabled()) {
            log.debug("{}在平台{},客户端版本：{},未读消息总数为：{}", userId, platform, curVer, unread);
        }

        return unread.intValue();
    }

    @Override
    public PageWarper<UserMessageBean> listMessage(MessageListParam messageListParam, BasicParam basicParam) {
        if (MessageTypeEnum.INTERACTION.name().equals(messageListParam.getMessageType())) {
            return new PageWarper<>(Lists.newArrayList());
        }

        constructionParam(messageListParam);

        if (CollectionUtils.isEmpty(messageListParam.getMsgTypeList())) {
            return new PageWarper<>(Lists.newArrayList());
        }

        PageWarper<UserMessageBean> msgPage = new PageWarper<>(messageMapper.listMessage(messageListParam));

        String messageType = messageListParam.getMessageType();
        //设置用户图标
        msgPage.getList().forEach(message -> {
            message.setReadStatus(message.getStatus());
            message.setLastTime(DateFormatUtils.format(message.getAddTime()));

            if (NumberUtils.isDigits(message.getMsgType())) {
                MessageBehaviorEnum behavior = PushMessageEnum.getMessageBehaviorByType(Integer.parseInt(message.getMsgType()));
                if (null != behavior) {
                    message.setHasDetail(MessageBehaviorEnum.JUMP.equals(behavior));
                }
            }

            //是否为互动消息
            if (messageTypeProperties.isInteraction(messageType)) {

                JSONObject linkParam = JSON.parseObject(message.getLinkParam());
                JSONObject extend = linkParam.getJSONObject("extend");

                message.setNickname(extend.getString("nickname"));
                message.setAction(extend.getString("action"));
                message.setIcon(extend.getString("icon"));
                message.setExtendContent(extend.getString("extendContent"));
                message.setExtendUrl(extend.getString("extendUrl"));
            }
        });
        return msgPage;
    }

    @Override
    public MessageTypeDTO getMessageType(String areaCode, Long userId, BasicParam basicParam) {
        MessageTypeDTO msgDto = new MessageTypeDTO();

        List<String> skipTypes = Lists.newArrayList();

        //提审时不显示余额 备注：平台：5 移除余额变动通知和系统通知 -------
        if (basicParam.getPlatform() == PlatformEnum.APPLET.getCode()
                || Boolean.TRUE.equals(appVersionIntegrationService.getPublishState(basicParam))) {
            skipTypes.add(MessageTypeEnum.BALANCE.name());
        }

        //平台：1和2 移除邀请通知
        if (basicParam.getPlatform() == PlatformEnum.ANDROID.getCode()
                || basicParam.getPlatform() == PlatformEnum.IOS.getCode()) {
            skipTypes.add(MessageTypeEnum.INVITE.name());
        }

        //设置用户的最后未读消息
        setMessageList(msgDto, userId, skipTypes, basicParam);

        //附加广告信息
        msgDto.setAdvertDTOS(advertIntegrationService.getAdvertByType("10", areaCode, userId));

        return msgDto;
    }

    /**
     * 获取消息中心显示的类型、标题、图标、未读数量。用于显示消息中心的列表
     *
     * @param msgDto    消息传输对象
     * @param userId    用户ID
     * @param skipTypes 不显示的消息类型
     */
    private void setMessageList(MessageTypeDTO msgDto, Long userId, List<String> skipTypes, BasicParam basicParam) {
        Map<String, Long> unreadNumMap = getUnreadMsgNumMap(userId);
        Map<String, MessageType> lastMsgMap = getLastMsgMap(userId);

        List<MessageType> msgList = lastMsgMap.values()
                .stream()
                .sorted((pre, next) ->
                        dateTimeComparator.compare(next.getSourceLastTime(), pre.getSourceLastTime()))
                .collect(Collectors.toList());

        List<MessageType> afterFilter = Lists.newArrayList();

        boolean beforeImVersion = appVersionIntegrationService.beforeImVersion(basicParam);
        boolean afterMsgCenterRebuild = appVersionIntegrationService.afterMsgCenterRebuild(basicParam);


        for (MessageType msgType : msgList) {
            String type = msgType.getMessageType();
            //跳过不在特定平台显示的消息
            if (null == type || skipTypes.contains(msgType.getMessageType())) {
                continue;
            }

            //低版本不显示即时通讯
            boolean showImMsg = (beforeImVersion || afterMsgCenterRebuild) && NumberUtils.isDigits(type);
            if (showImMsg) {
                continue;
            }

            //处理日期格式
            if (null != msgType.getSourceLastTime()) {
                msgType.setLastTime(DateFormatUtils.format(msgType.getSourceLastTime()));
            }

            Long unreadNum = unreadNumMap.get(type);
            msgType.setUnreadNum(unreadNum == null ? 0 : unreadNum.intValue());

            //三种特殊的消息类型处理
            if (!beforeImVersion) {
                if (LIKE.name().equals(type)) {
                    msgType.setTitle("点赞");
                    msgDto.setLikeType(msgType);
                    continue;
                } else if (COMMENT.name().equals(type)) {
                    msgType.setTitle("评论");
                    msgDto.setCommentType(msgType);
                    continue;
                } else if (ADD_FUNS.name().equals(type)) {
                    msgType.setTitle("新增粉丝");
                    msgDto.setFollowType(msgType);
                    continue;
                }
            } else {
                //旧版本不显示新增粉丝
                if (ADD_FUNS.name().equals(type)) {
                    continue;
                }
            }

            MessageType configTypeInfo = messageTypeProperties.getMessageInfo(type);

            log.debug("match msg type:{}", configTypeInfo);

            if (null != configTypeInfo) {
                msgType.setTitle(configTypeInfo.getTitle());
                msgType.setImg(configTypeInfo.getImg());
            } else {
                //未匹配的类型不再显示
                continue;
            }

            afterFilter.add(msgType);
        }
        msgDto.setMessageTypes(afterFilter);
        fillNull(msgDto, beforeImVersion);
    }

    /**
     * 点赞、评论、新增粉丝在某一个版本调整出了消息栏目到单独的顶部区域
     *
     * @param msgDto          消息栏目信息完整信息
     * @param beforeImVersion 是否在特定的版本之前
     */
    private void fillNull(MessageTypeDTO msgDto, boolean beforeImVersion) {
        if (null == msgDto.getLikeType()) {
            MessageType likeType = new MessageType();
            likeType.setMessageType(LIKE.name());
            likeType.setTitle("点赞");
            likeType.setUnreadNum(0);
            msgDto.setLikeType(likeType);
        }

        if (null == msgDto.getCommentType()) {
            MessageType commentType = new MessageType();
            commentType.setMessageType(COMMENT.name());
            commentType.setTitle("评论");
            commentType.setUnreadNum(0);
            msgDto.setCommentType(commentType);
        }

        if (null == msgDto.getFollowType()) {
            MessageType followType = new MessageType();
            followType.setMessageType(ADD_FUNS.name());
            followType.setTitle("新增粉丝");
            followType.setUnreadNum(0);
            msgDto.setFollowType(followType);
        }

        if (msgDto.getMessageTypes() == null) {
            msgDto.setMessageTypes(Lists.newArrayList());
        }

        for (String type : messageTypeProperties.getMessageTypes()) {
            boolean specTypeAndOldVersion = !beforeImVersion && (LIKE.name().equals(type)
                    || COMMENT.name().equals(type)
                    || ADD_FUNS.name().equals(type));
            if (specTypeAndOldVersion) {
                continue;
            }
            if (msgDto.getMessageTypes().stream().anyMatch(item -> type.equals(item.getMessageType()))) {
                continue;
            }
            MessageType innerType = new MessageType();
            MessageType configTypeInfo = messageTypeProperties.getMessageInfo(type);
            if (null != configTypeInfo) {
                innerType.setMessageType(type);
                innerType.setTitle(configTypeInfo.getTitle());
                innerType.setImg(configTypeInfo.getImg());

                msgDto.getMessageTypes().add(innerType);
            }
        }
    }

    private KeyGenerator getUnreadCacheKey(Long userId) {
        return UN_READ_MSG.copy().appendKey(userId.toString());
    }

    private KeyGenerator getLastMsgCacheKey(Long userId) {
        return LAST_MSG.copy().appendKey(userId.toString());
    }

    /**
     * 获取用户的各种类型的未读消息数量
     *
     * @param userId 用户ID
     * @return 未读消息map，key为消息类型，value为对应消息类型的未读数量
     */
    private Map<String, Long> getUnreadMsgNumMap(Long userId) {
        KeyGenerator key = getUnreadCacheKey(userId);
        Map<String, Long> result = redisHashMapAdapter.entries(key, Long.class);

        //如果不存在用户缓存，则检索一次数据库
        if (null == result || result.isEmpty()) {
            result = loadHistoryUnread(userId);
            redisHashMapAdapter.putAll(key, result);
        }

        return result;
    }

    /**
     * 从数据库加载各类型的历史未读数据
     *
     * @param userId 用户ID
     * @return 各消息类型的未读数量
     */
    private Map<String, Long> loadHistoryUnread(Long userId) {
        //查询各种推送消息的未读数量
        List<Tuple> unreadTypes = messageMapper.getUnreadMap(userId);

        Map<String, Long> result = Maps.newHashMap();

        for (Tuple tuple : unreadTypes) {
            //将推送消息的类型转换为消息中心显示的类型
            String msgType = messageTypeProperties.convertType(tuple.getKey());
            Long val = result.get(msgType);
            val = (val == null ? 0 : val) + tuple.getValue();

            result.put(msgType, val);
        }

        return result;
    }

    private Map<String, MessageType> getLastMsgMap(Long userId) {
        KeyGenerator key = getLastMsgCacheKey(userId);
        Map<String, MessageType> result;
        if (Boolean.TRUE.equals(redisHashMapAdapter.hasKey(key))) {
            result = redisHashMapAdapter.entries(key, MessageType.class);
        } else {
            result = loadLastMsgHistory(userId);
            redisHashMapAdapter.putAll(key, result);
        }

        return result;
    }

    private Map<String, MessageType> loadLastMsgHistory(Long userId) {
        Map<String, MessageType> result = Maps.newHashMap();

        List<UserMessageBean> lastMsgType = messageMapper.getLastMsgType(userId);
        List<Long> lastIdList = lastMsgType.stream().map(UserMessageBean::getId).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(lastIdList)) {
            return result;
        }

        List<UserMessageBean> lastMessages = messageMapper.getLastMsg(userId, lastIdList);

        for (UserMessageBean msg : lastMessages) {
            String msgType = messageTypeProperties.convertType(msg.getMsgType());
            MessageType messageType = build(msg, msgType);

            result.put(msgType, messageType);
        }

        return result;
    }

    @Override
    public void addUnreadMsg(PushMessage message, Long userId) {
        String msgType = messageTypeProperties.convertType(message.getPayloadInfo().getType());
        //添加对应类型的未读消息,数据迁移原因，初次加载时，尝试加载历史数据
        KeyGenerator key = getUnreadCacheKey(userId);

        if (Boolean.FALSE.equals(redisHashMapAdapter.hasKey(key))) {
            redisHashMapAdapter.putAll(key, loadHistoryUnread(userId));
            redisHashMapAdapter.expire(key, 7 * 24 * 3600);
        }

        redisHashMapAdapter.increment(key, msgType, 1);

        //添加对应类型的最后阅读消息
        addMessageTypeLastMsg(message, userId, msgType);

        //小程序模板消息推送
        //pushTemplateMessage(message, userId);

        //添加消息未读的事件
        addUnReadUserEvent(userId, msgType);

        //如果是互动消息，则计入到定期的互动消息中，统一进行推送
        if (messageTypeProperties.isInteraction(msgType)) {
            addInteractionCache(message, userId, msgType);
        }
    }

    @Override
    public void addLastMsg(PushMessage message) {
        Preconditions.checkArgument(null != message);
        Long userId = message.getPushReceiveScope().getUserId();
        String msgType = messageTypeProperties.convertType(message.getPayloadInfo().getType());

        addMessageTypeLastMsg(message, userId, msgType);
    }

    /**
     * 添加互动消息缓存
     *
     * @param message 推送消息
     * @param userId  目标用户
     * @param msgType 推送消息类型
     */
    private void addInteractionCache(PushMessage message, Long userId, String msgType) {
        int index = IntervalPeriodUtils.getIntervalIndex();
        //记录每个用户在时间段内最后一条互动消息
        KeyGenerator key = LAST_INTERACTION_MSG.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.put(key, userId.toString(), message);
        redisHashMapAdapter.expire(key, 60 * 60);

        //记录每个用户在时间段内的互动消息总数
        key = INTERACTION_NUM.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.increment(key, userId.toString(), 1);
        redisHashMapAdapter.expire(key, 60 * 60);
    }

    /**
     * 移除互动消息缓存
     *
     * @param userId  目标用户
     * @param msgType 推送消息类型
     */
    private void removeInteractionCache(Long userId, String msgType) {
        int index = IntervalPeriodUtils.getIntervalIndex();
        KeyGenerator key = LAST_INTERACTION_MSG.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.remove(key, userId.toString());

        key = INTERACTION_NUM.copy().appendKey(index).appendKey(msgType);
        redisHashMapAdapter.put(key, userId.toString(), 0);
    }

    private MessageType build(UserMessageBean message, String msgType) {
        MessageType messageType = new MessageType();

        messageType.setMessageType(msgType);
        messageType.setSourceLastTime(message.getAddTime());
        if (messageTypeProperties.isInteraction(msgType)) {
            JSONObject extend = JSONObject.parseObject(message.getLinkParam());
            String nickName = Objects.toString(extend.get("nickname"), "");
            String action = Objects.toString(extend.get("action"), "");

            messageType.setDescription(nickName + action);
        } else {
            messageType.setDescription(message.getContent());
        }

        return messageType;
    }

    /**
     * 将用户的最后一次未读消息写入到缓存。用于显示在消息列表中
     *
     * @param message 推送消息
     * @param userId  用户ID
     * @param msgType 消息中心栏目类型
     */
    private void addMessageTypeLastMsg(PushMessage message, Long userId, String msgType) {
        MessageType entity = new MessageType();
        KeyGenerator key = getLastMsgCacheKey(userId);

        //数据迁移原因，初次加载时，尝试加载历史数据
        if (Boolean.FALSE.equals(redisHashMapAdapter.hasKey(key))) {
            Map<String, MessageType> historyMap = loadLastMsgHistory(userId);
            redisHashMapAdapter.putAll(key, historyMap);
            redisHashMapAdapter.expire(key, 7 * 24 * 3600);
        }

        if (messageTypeProperties.isInteraction(msgType)) {
            Map<String, Object> extend = message.getPayloadInfo().getExtend();
            String nickName = Objects.toString(extend.get("nickname"), "");
            String action = Objects.toString(extend.get("action"), "");
            entity.setDescription(nickName + action);
        } else {
            entity.setDescription(message.getContent());
        }

        //如果是即时通讯消息
        if (PushMessageEnum.IM_MESSAGE.name().equals(msgType)) {
            Map<String, Object> extend = message.getPayloadInfo().getExtend();

            String nickname = Objects.toString(extend.get("nickname"), "");
            String headImg = Objects.toString(extend.get("headImg"), "");
            String customType = Objects.toString(extend.get("customType"), "");

            entity.setImg(headImg);
            entity.setTitle(nickname);
            entity.setMessageType(customType);

            msgType = customType;
        }

        entity.setSourceLastTime(new Date());
        entity.setMessageType(msgType);
        redisHashMapAdapter.put(key, msgType, entity);
    }

    @Override
    public void updateMessageByType(Long userId, String messageType) {
        //如果是用户的消息，则不需要更新数据库
        if (!NumberUtils.isDigits(messageType)
                && !MessageTypeEnum.ADD_FUNS.name().equals(messageType)) {
            MessageListParam messageListParam = new MessageListParam();
            messageListParam.setUserId(userId);
            messageListParam.setMessageType(messageType);
            constructionParam(messageListParam);

            this.messageMapper.updateMessageByType(messageListParam);
        }

        //对应的消息类型未读数量设置为0
        redisHashMapAdapter.put(getUnreadCacheKey(userId), messageType, 0L);

        //如果互动消息已读，则定时推送互动消息移除对应的类型信息
        if (messageTypeProperties.isInteraction(messageType)) {
            removeInteractionCache(userId, messageType);
        }

        //通知客户端设置未读消息数量（补偿策略，不清楚客户端有没有对已读消息做处理）
        addUnReadUserEvent(userId, messageType);
    }

    @Override
    public void addMessage(PushMessage message, Long userId) {
        UserMessageBean entity = new UserMessageBean();
        entity.setId(sequenceCreater.nextLongId());
        entity.setTitle(message.getTitle());
        entity.setContent(message.getContent());
        entity.setUserId(userId);
        entity.setMsgType(String.valueOf(message.getPayloadInfo().getType()));
        entity.setStatus((byte) 0);
        entity.setLinkParam(JSON.toJSONString(message.getPayloadInfo()));
        entity.setAddTime(new Date());

        addUnreadMsg(message, userId);

        if (message.isPersistence()) {
            this.messageMapper.insertSelective(entity);
        }
    }

    @Override
    public Map<String, Long> getUnReadNum(Long userId, String messageType) {
        Map<String, Long> resultMap = Maps.newHashMap();

        //获取用户的未读消息缓存
        Map<String, Long> unReadMap = getUnreadMsgNumMap(userId);

        if (StringUtils.isBlank(messageType)) {
            return resultMap;
        }
        if (!unReadMap.containsKey(messageType)) {
            return resultMap;
        }
        resultMap.put(messageType, unReadMap.get(messageType));

        return getUnreadMsgNumMap(userId);
    }

    /**
     * 添加用户事件，用于显示未读消息的小红点
     *
     * @param userId  用户ID
     * @param msgType 消息类型
     */
    private void addUnReadUserEvent(Long userId, String msgType) {
        /*
         * V3.4.0 不对3.0.0版本和小程序进行兼容，减少无谓的用户事件
         */
        if (MessageTypeEnum.BALANCE.name().equals(msgType)) {
            //余额变动仅在客户端显示
            //userEventService.add(buildPushPayloadInfo(userId));
        } else if (MessageTypeEnum.INVITE.name().equals(msgType)) {
            //邀请信息仅在小程序显示
            //userEventService.add(buildAppletPushPayloadInfo(userId));
        } else if (PushMessageEnum.IM_MESSAGE.name().equals(msgType)) {
            //用户私聊消息
            //userEventService.add(buildPushPayloadInfo(userId));
        } else {
            //系统消息
            userEventService.add(buildSystemPush(userId));
            //userEventService.add(buildPushPayloadInfo(userId));
            //userEventService.add(buildAppletPushPayloadInfo(userId));
        }
    }

    /**
     * 构建未读消息总数
     *
     * @param userId 用户ID
     * @return
     */
    private PushPayloadInfo buildSystemPush(Long userId) {
        return PushPayloadInfo.build().setType(UserEventEnum.UN_READ_SYSTEM_MSG.type)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 0, "3.0.0"));
    }

    /**
     * app用户事件
     *
     * @param userId 用户ID
     * @deprecated 不再对3.0.0版本之前进行兼容
     */
    private PushPayloadInfo buildPushPayloadInfo(Long userId) {
        return PushPayloadInfo.build().setType(UserEventEnum.USER_UN_READ_MSG.type)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 0, null));
    }

    /**
     * 小程序用户事件
     *
     * @param userId 用户ID
     * @deprecated 小程序未推广，不再推送消息
     */
    private PushPayloadInfo buildAppletPushPayloadInfo(Long userId) {
        return PushPayloadInfo.build().setType(UserEventEnum.APPLET_USER_UN_READ_MSG.type)
                .addExtend("userId", userId).addExtend("current", getUnReadMsg(userId, 5, null));
    }

    /**
     * 推送小程序模板消息
     *
     * @param pushMessage 推送消息
     * @param userId      目标用户ID
     */
    private void pushTemplateMessage(PushMessage pushMessage, Long userId) {
        int type = pushMessage.getPayloadInfo().getType();
        //判断消息类型是否需要推送小程序模板消息
        if (messageTypeProperties.getTemplate().contains(type)) {
            PushMessage cloneMsg = PushMessage.build();
            BeanUtils.copyProperties(pushMessage, cloneMsg);

            cloneMsg.setPushReceiveScope(PushReceiveScope.pushRule(PushReceiverRuleEnum.DEFAULT)
                    .addRuleParam(ReceiverRuleParamConstant.USER_IDS, userId));
            pushTemplateMessageIntegrationService.pushTemplateMessage(cloneMsg);
        }
    }

    /**
     * 构造参数信息
     * 根据查询的消息聚合类型获取对应的子类型
     */
    private void constructionParam(MessageListParam param) {
        //备注：小程序版本1.0.0和万事通2.3.0版本互动消息变成评论通知和点赞通知
        if (isNotEmpty(param.getMessageType())
                && MessageTypeEnum.INTERACTION.name().equals(param.getMessageType())) {
            List<String> msgTypeList = messageTypeProperties.getSubTypes(MessageTypeEnum.COMMENT.name());
            msgTypeList.addAll(messageTypeProperties.getSubTypes(LIKE.name()));
            param.setMsgTypeList(msgTypeList);
        } else {
            param.setMsgTypeList(messageTypeProperties.getSubTypes(param.getMessageType()));
        }
    }
}
