package com.bxm.localnews.news.controller;

import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.dto.ReplenishmentDTO;
import com.bxm.localnews.news.param.ForumPostInformParam;
import com.bxm.localnews.news.param.ForumPostListQueryParam;
import com.bxm.localnews.news.param.ForumPostListUserQueryParam;
import com.bxm.localnews.news.service.ForumService;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

@Api(tags = "4-10 社区版块相关信息获取")
@RestController
@RequestMapping("api/forum")
public class ForumController extends BaseController {

    @Resource
    private ForumService forumService;

    @ApiOperation(value = "4-10-1 根据地区获取版块列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "areaCode", value = "地区", required = true),
            @ApiImplicitParam(name = "enable", value = "查询版块范围：0全部 1已上架", required = true)
    })
    @GetMapping(value = "getForumListByArea")
    public Json<List<ForumVo>> getForumListByArea(String areaCode, Integer enable) {
        return ResultUtil.genSuccessResult(forumService.getForumList(areaCode, enable));
    }

    @ApiOperation(value = "4-10-2 根据地区获取话题列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "areaCode", value = "地区", required = true),
            @ApiImplicitParam(name = "enable", value = "查询话题范围：0全部 1已上架", required = true)
    })
    @GetMapping(value = "getTopicListByArea")
    public Json<List<TopicVo>> getTopicListByArea(String areaCode, Integer enable) {
        return ResultUtil.genSuccessResult(forumService.getTopicList(areaCode, enable));
    }

    @ApiOperation(value = "4-10-3 根据id获取版块", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true)
    })
    @GetMapping(value = "getForumById")
    public Json<ForumVo> getForumById(Long id) {
        return ResultUtil.genSuccessResult(forumService.selectForumById(id));
    }

    @ApiOperation(value = "4-10-4 根据id获取话题", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true)
    })
    @GetMapping(value = "getTopicById")
    public Json<TopicVo> getTopicById(Long id) {
        return ResultUtil.genSuccessResult(forumService.selectTopicById(id));
    }

    @ApiOperation(value = "4-10-5 根据版块或话题获取帖子列表", httpMethod = "GET")
    @GetMapping(value = "getPostPage")
    public Json<PageWarper<ForumPostVo>> getPostPage(ForumPostListQueryParam param) {
        return ResultUtil.genSuccessResult(forumService.getPostPage(param));
    }

    @ApiOperation(value = "4-10-6 根据用户及类型获取帖子列表", httpMethod = "GET")
    @GetMapping(value = "getPostPageByUser")
    public Json<PageWarper<ForumPostVo>> getPostPageByUser(ForumPostListUserQueryParam param) {
        return ResultUtil.genSuccessResult(forumService.getPostPageByUser(param));
    }

    @ApiOperation(value = "4-10-7 根据id获取帖子详情", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "帖子id", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "devcId", value = "设备id", required = false)
    })
    @GetMapping(value = "getPostDetailById")
    public Json<ForumPostVo> getPostDetailById(Long id, Long userId, String devcId) {
        return ResultUtil.genSuccessResult(forumService.getPostDetailById(id, userId, devcId));
    }

    @ApiOperation(value = "4-10-8 发布或编辑帖子", httpMethod = "POST")
    @PostMapping(value = "public/createOrUpdatePost")
    public Json<Boolean> createOrUpdatePost(@RequestBody ForumBasicVo forumBasicVo) {
        int result = forumService.createOrUpdatePost(forumBasicVo);
        if (-1 == result) {
            return ResultUtil.genFailedResult("发布内容包含敏感词");
        }
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "4-10-9 删除帖子", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "帖子id", required = true)
    })
    @PostMapping(value = "delete")
    public Json<Boolean> delete(Long id) {
        return ResultUtil.genSuccessResult(forumService.delete(id));
    }

    @ApiOperation(value = "4-10-10 点击次数+1", httpMethod = "POST", notes = "暂时不用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "帖子id", required = true)
    })
    @PostMapping(value = "addClick")
    public Json addClick(Long id) {
        return null;
    }

    @ApiOperation(value = "4-10-11 点赞、取消点赞", notes = "点赞、取消点赞")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "postId", value = "帖子ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "type", value = "0：取消点赞，1：点赞", required = true),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @PostMapping("like")
    public Json like(Long postId, Long userId, int type, String areaCode) {
        forumService.doLike(userId, postId, type, areaCode);
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation("4-10-12 收藏或取消收藏帖子")
    @RequestMapping(value = "collect", method = RequestMethod.POST)
    public Json collect(@RequestParam(value = "userId", required = true) Long userId,
                        @RequestParam(value = "postId", required = true) Long postId) {
        this.forumService.doCollect(userId, postId);
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation("4-10-13 帖子举报")
    @RequestMapping(value = "inform", method = RequestMethod.POST)
    public Json inform(ForumPostInformParam forumPostInformParam) {
        return ResultUtil.genSuccessResult(this.forumService.doInform(forumPostInformParam));
    }

    @ApiOperation(value = "4-10-14 帖子分享获得金币（返回类和新闻一样）", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "postId", value = "帖子ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @RequestMapping(value = "share", method = RequestMethod.POST)
    public Json<NewsCompleTaskDTO> share(@RequestParam(value = "postId") Long postId,
                                         @RequestParam(value = "userId") Long userId,
                                         @RequestParam(value = "areaCode") String areaCode) {
        return ResultUtil.genSuccessResult(this.forumService.doShare(userId, postId, areaCode));
    }

    @ApiOperation(value = "4-10-15 帖子详情相关推荐", httpMethod = "GET", notes = "如果是15条，则5条本地新闻 10条帖子，如果是5条，则2条本地，3条帖子")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "postId", value = "帖子ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = false),
            @ApiImplicitParam(name = "areaCode", value = "地区编号", required = false),
            @ApiImplicitParam(name = "size", value = "推荐个数", required = false)
    })
    @GetMapping(value = "detail/recommend")
    public Json<List<News4Client>> recommend(@RequestParam(value = "postId") Long postId,
                                             @RequestParam(value = "userId", required = false) Long userId,
                                             @RequestParam(value = "areaCode", required = false) String areaCode,
                                             @RequestParam(value = "size", required = false) Integer size) {
        return ResultUtil.genSuccessResult(forumService.listPostDetailRecommend(postId,userId,size,areaCode));
    }

    @ApiOperation(value = "4-10-16 获得热门[帖子]或者[新闻]用于补签", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = false)
    })
    @GetMapping(value = "hot")
    public Json<ReplenishmentDTO> recommend(@RequestParam(value = "userId", required = true) Long userId,
                                            @RequestParam(value = "areaCode",required = true) String areaCode) {
        return ResultUtil.genSuccessResult(forumService.getHotPostOrNews(userId,areaCode));
    }

}
