package com.bxm.localnews.news.controller;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.news.dto.VideoDto;
import com.bxm.localnews.news.dto.VideoReplyDto;
import com.bxm.localnews.news.param.*;
import com.bxm.localnews.news.service.*;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

@Api(tags = "4-08 视频相关接口", description = "视频获取，视频点赞、评论等相关接口")
@RestController
@RequestMapping("api/video")
public class VideoController extends BaseController {

    private final VideoViewService videoViewService;
    private final VideoReplyService videoReplyService;
    private final VideoBlackService videoBlackService;
    private final VideoShareService videoShareService;
    private VideoRecommendService videoRecommendService;
    private VideoLikeService videoLikeService;

    @Autowired
    public VideoController(VideoRecommendService videoRecommendService,
                           VideoLikeService videoLikeService,
                           VideoViewService videoViewService,
                           VideoReplyService videoReplyService,
                           VideoBlackService videoBlackService,
                           VideoShareService videoShareService) {
        this.videoRecommendService = videoRecommendService;
        this.videoLikeService = videoLikeService;
        this.videoViewService = videoViewService;
        this.videoReplyService = videoReplyService;
        this.videoBlackService = videoBlackService;
        this.videoShareService = videoShareService;
    }

    @ApiOperation(value = "4-08-1 获取视频列表", notes = "为指定的用户推荐一定量的视频数量,如果返回的结果数组长度为0，表示没有更多的视频可以推荐")
    @GetMapping("list")
    public Json<List<VideoDto>> getVideo(VideoQueryParam param) {
        return Json.build(videoRecommendService.execRecommend(param));
    }

    @ApiOperation(value = "4-08-2 屏蔽视频", notes = "用户对视频不感兴趣，或举报视频")
    @PostMapping("black")
    public Json black(VideoBlackParam param) {
        return Json.build(videoBlackService.produceVideoReply(param));
    }

    @ApiOperation(value = "4-08-3 点赞、取消点赞视频", notes = "视频点赞、取消点赞")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "videoId", value = "视频ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "type", value = "0：取消点赞，1：点赞")
    })
    @PostMapping("like")
    public Json like(Long videoId, Long userId, int type) {
        return Json.build(videoLikeService.save(videoId, userId, type));
    }

    @ApiOperation(value = "4-08-4 获取回复列表", notes = "获取单个视频下的回复清单,如果返回result数组为空或小于PageSize则表示无更多回复")
    @GetMapping("reply/list")
    public Json<PageWarper<VideoReplyDto>> getReplies(VideoReplyListQueryParam param) {
        return ResultUtil.genSuccessResult(videoReplyService.queryByPageSize(param));
    }

    @ApiOperation(value = "4-08-5 视频回复", notes = "视频留言，或对回复进行回复")
    @PostMapping("reply")
    public Json reply(VideoReplyParam param) {
        return Json.build(videoReplyService.replyVideo(param));
    }

    @ApiOperation(value = "4-08-6 回复点赞", notes = "对留言进行点赞")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "replyId", value = "回复ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "type", value = "0：取消点赞，1：点赞")
    })
    @PostMapping("like/reply")
    public Json likeReply(VideoReplyLikeParam param) {
        return Json.build(videoReplyService.produceVideoReplyLike(param));
    }

    @ApiOperation(value = "4-08-7 视频浏览记录", notes = "用户浏览视频结束后调用，保存用户浏览记录")
    @PostMapping("view")
    public Json view(VideoViewParam param) {
        return Json.build(videoViewService.produceVideoView(param));
    }


    @ApiOperation(value = "4-08-8 视频分享", notes = "用户分享视频结束后调用，保存用户分享记录")
    @PostMapping("share")
    public Json share(VideoShareParam param) {
        return Json.build(videoShareService.produceVideoShare(param));
    }


    @ApiOperation(value = "4-08-9 视频举报", notes = "用户可以在视频详情页面对视频进行举报")
    @PostMapping("inform")
    public Json inform(VideoInformParam param) {
        return Json.build(Message.build(videoBlackService.informVideo(param)));
    }

}
