package com.bxm.localnews.news.controller;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.common.annotation.TouTiaoAuth;
import com.bxm.localnews.common.constant.AppConst;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.common.vo.MPage;
import com.bxm.localnews.news.domain.MPArticleBlackMapper;
import com.bxm.localnews.news.domain.NewsBlackMapper;
import com.bxm.localnews.news.param.ExecGoldParam;
import com.bxm.localnews.news.service.NewsService;
import com.bxm.localnews.news.vo.*;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

@Api(tags = "4-02 新闻相关信息获取", description = "获取新闻详情")
@RestController
@RequestMapping("api/news")
public class NewsController {

    @Resource
    private NewsService newsService;

    @Resource
    private NewsBlackMapper newsBlackMapper;

    @Resource
    private MPArticleBlackMapper mpArticleBlackMapper;

    @ApiOperation(value = "4-02-1 获取新闻详情页跳转地址", httpMethod = "GET", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID"),
            @ApiImplicitParam(name = "type", value = "获取类型  1：详情地址   2：分享地址  " +
                    "3：搜索文章查看详情地址 4:广告分享链接（newsId为广告id）", required = true)
    })
    @GetMapping(value = "detailUrl")
    public Json<NewsUrlVo> getUrl(@RequestParam(value = "newsId") Long newsId,
                                  @RequestParam(value = "userId", required = false) Long userId,
                                  @RequestParam(value = "type") Byte type) {
        return this.newsService.getUrl(newsId, userId, type);
    }

    @ApiOperation(value = "4-02-2 新闻详情", notes = "获取新闻内容和详细信息", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID"),
            @ApiImplicitParam(name = "type", value = "页面类型 1：app详情页获取数据  2：分享详情页获取数据 " +
                    "3：app用户搜索的文章获取数据", required = true)
    })
    @GetMapping(value = "syDetail")
    public Json<NewsDetailVO> syDetail(@RequestParam(value = "newsId") Long newsId,
                                       @RequestParam(value = "userId", required = false) Long userId,
                                       @RequestParam(value = "type") Byte type) {
        return this.newsService.getSyDetail(newsId, userId, type);
    }

    @ApiOperation(value = "4-02-3 获取阅读文章获取金币", notes = "(app)阅读文章获取金币（满足条件调用）", httpMethod = "POST")
    @PostMapping(value = "read/gold")
    public Json<NewsGoldMeta> getGold(ExecGoldParam execGoldParam) {
        return this.newsService.execGetGold(execGoldParam);
    }

    @ApiOperation(value = "4-02-4 (app)阅读文章返回时调用", notes = "", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "viewTime", value = "阅读时间", required = true),
            @ApiImplicitParam(name = "tabType", value = "是否读取到文章末尾 1：否   2：是", required = true),
            @ApiImplicitParam(name = "advertShow", value = "广告位是否展示出来 1：否  2：是", required = true)
    })
    @RequestMapping(value = "read/back", method = RequestMethod.POST)
    public Json getBack(@RequestParam(value = "newsId") Long newsId,
                        @RequestParam(value = "userId") Long userId,
                        @RequestParam(value = "viewTime") Integer viewTime,
                        @RequestParam(value = "tabType") String tabType,
                        @RequestParam(value = "advertShow") String advertShow) {
        return this.newsService.getBack(newsId, userId, viewTime, tabType, advertShow);
    }

    @ApiOperation(value = "4-02-5 新闻/视频详情推荐", notes = "包含文章下面的广告默认是列表第一条", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true)
    })
    @RequestMapping(value = "detail/recommend", method = RequestMethod.GET)
    public Json<List<News4Client>> detailRecommend(@RequestParam(value = "newsId") Long newsId) {
        return this.newsService.queryRecommendNews(newsId);
    }

    @ApiOperation(value = "4-02-6 组图推荐", notes = "", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true)
    })
    @RequestMapping(value = "imgs/recommend", method = RequestMethod.GET)
    public Json<List<News4Client>> imgsRecommend(@RequestParam(value = "newsId") Long newsId) {
        return this.newsService.queryRecommendNews4ImgNews(newsId);
    }

    @ApiOperation(value = "4-02-7 新闻分享成功金币赠送", notes = "", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true)
    })
    @RequestMapping(value = "share/gold", method = RequestMethod.POST)
    @TouTiaoAuth
    public Json shareNews(@RequestParam(value = "newsId") Long newsId,
                          @RequestParam(value = "userId") Long userId) {
        return this.newsService.saveShareNews(newsId, userId);
    }

    @ApiOperation(value = "4-02-8 好友阅读分享的新闻（给分享人金币）", notes = "", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "viewTime", value = "阅读时间", required = true),
            @ApiImplicitParam(name = "tabType", value = "是否读取到文章末尾 1：否   2：是", required = true),
            @ApiImplicitParam(name = "advertShow", value = "广告位是否展示出来 1：否  2：是", required = true)
    })
    @RequestMapping(value = "readShare/gold", method = RequestMethod.POST)
    public Json getReadShareNewsGold(@RequestParam(value = "newsId") Long newsId,
                                     @RequestParam(value = "userId") Long userId,
                                     @RequestParam(value = "viewTime", required = false) Integer viewTime,
                                     @RequestParam(value = "tabType", required = false) String tabType,
                                     @RequestParam(value = "advertShow", required = false) String advertShow) {
        return this.newsService.saveReadShareNewsGold(newsId, userId, viewTime, tabType, advertShow);
    }

    @ApiOperation(value = "4-02-9 视频上/下刷新", notes = "", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "type", value = "类型", required = true),
            @ApiImplicitParam(name = "kindId", value = "新闻频道", required = true),
            @ApiImplicitParam(name = "cityName", value = "城市名称")
    })
    @RequestMapping(value = "refreshVideo", method = RequestMethod.GET)
    public Json<NewsMeta> refreshVideo(Long userId, Byte type, Integer kindId, String cityName) {
        return this.newsService.refreshVideo(userId, type, kindId);
    }

    @ApiOperation(value = "4-02-10 用户不感兴趣的文章", notes = "", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻ID", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", required = true),
            @ApiImplicitParam(name = "types", value = "屏蔽原因：1：不感兴趣，2：内容质量差，3：旧闻重复，4：不喜欢该作者", required = true)
    })
    @TouTiaoAuth
    @PostMapping("black")
    public Json black(Long userId, Long newsId, Integer[] types) {
        if (newsId == null) {
            return ResultUtil.genFailedResult("参数错误");
        }
        //XXX [zhaoyadong]时间：2018-10-8 此处直接调用mapper层
        if (newsId > AppConst.MP_ARTICLE_ID_PLUS) {
            this.mpArticleBlackMapper.add(userId, newsId - AppConst.MP_ARTICLE_ID_PLUS,
                    types != null ? JSON.toJSONString(types) : null);
        } else {
            this.newsBlackMapper.addBlack(userId, newsId, types != null ? JSON.toJSONString(types) : null);
        }
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "4-02-11 新闻查询", notes = "", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "keyword", value = "关键字", required = true)
    })
    @RequestMapping(value = "search", method = RequestMethod.GET)
    public Json<List<News>> searchNews(@RequestParam(value = "userId") Long userId,
                                       @RequestParam(value = "keyword") String keyword, MPage page) {
        page.checkPage();
        List<News> list = this.newsService.searchNews(userId, keyword, page);
        return ResultUtil.genSuccessResultWithPage(list, page);
    }

    @ApiOperation(value = "4-02-12 获取热门搜索标题", notes = "", httpMethod = "GET")
    @RequestMapping(value = "hotTitle", method = RequestMethod.GET)
    public Json<List<NewsSearchWord>> searchHotTitle() {
        return this.newsService.searchHotTitle();
    }

    @ApiOperation(value = "4-02-13 视频分类", notes = "", httpMethod = "GET")
    @RequestMapping(value = "video/kinds", method = RequestMethod.GET)
    public Json<List<NewsKind>> videoKinds() {
        return this.newsService.searchKinds();
    }

    @ApiOperation(value = "4-02-14 无效视频重新抓取接口", notes = "", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻id", required = true)
    })
    @RequestMapping(value = "video/regain", method = RequestMethod.GET)
    public Json<FailVideo> getFailVideo(Long newsId) {
        return this.newsService.getFailVideo(newsId);
    }

    @ApiOperation(value = "4-02-15 点击次数+1", notes = "", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "newsId", value = "新闻id", required = true)
    })
    @RequestMapping(value = "click/add", method = RequestMethod.GET)
    public Json addClick(Long newsId) {
        return this.newsService.addClick(newsId);
    }

}
