package com.bxm.localnews.integration;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.facade.MessageFeignService;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.PushReceiveScope;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.NewsMapper;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.domain.VideoMapper;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.NewsReplyLikeParam;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * @author zhaoyadong 2019/4/17 14:43
 * @desc
 */
@Service
public class MessageIntegrationService extends BaseService {

    @Autowired
    private MessageFeignService messageFeignService;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private UserProperties userProperties;

    @Autowired
    private ForumPostMapper forumPostMapper;

    @Autowired
    private VideoMapper videoMapper;

    @Autowired
    private NewsMapper newsMapper;

    @Autowired
    private NewsReplyMapper newsReplyMapper;

    public void addMessage(PushMessage pushMessage, Long userId) {

        messageFeignService.addMessage(pushMessage, userId);
    }

    /**
     * 带任务的消息推送
     *
     * @param shareUserId
     * @param userId
     * @param completeTaskDTO
     * @param code
     * @param extendUrl
     * @param extendContent
     * @param newsTitle
     */
    public void addMessage(Long shareUserId,
                           Long userId,
                           NewsCompleTaskDTO completeTaskDTO,
                           Long newsId,
                           Byte code,
                           String extendUrl,
                           String extendContent,
                           String newsTitle) {

        UserBean userEntity = userIntegrationService.selectUserFromCache(userId);
        String title, action, icon, nickname;
        PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.FORWARDING);
        if (Objects.isNull(userEntity) || StringUtils.isEmpty(userEntity.getWeixin())) {

            title = "有人悄悄阅读了你转发的内容，他没透露名字，奖励你" + completeTaskDTO.getGoldNum() + "朵小红花；";
            action = title;
            icon = userProperties.getDefaultHeadImgUrl();
            nickname = "";
        } else {
            title = userEntity.getNickname() + "阅读了你转发的内容，奖励你" + completeTaskDTO.getGoldNum() + "朵小红花；";
            icon = userEntity.getHeadImg();
            nickname = userEntity.getNickname();
            action = "阅读了你转发的内容,奖励" + completeTaskDTO.getGoldNum() + "小红花";
        }
        Long msgId = nextId();
        info.setMsgId(msgId);
        //添加用户图标
        info.addExtend("msgId", msgId);
        info.addExtend("newsId", newsId);
        info.addExtend("title", title);
        info.addExtend("icon", icon);
        info.addExtend("nickname", nickname);
        info.addExtend("action", action);
        info.addExtend("type", code);
        info.addExtend("extendUrl", extendUrl);
        info.addExtend("extendContent", extendContent);
        info.addExtend("flowerNum", completeTaskDTO.getGoldNum());
        PushMessage message = PushMessage.build();
        message.setTitle(title);
        message.setContent(newsTitle);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(shareUserId));
        message.setPayloadInfo(info);
        messageFeignService.addMessage(message, shareUserId);
        logger.debug("用户[{}]阅读了用户[{}]转发帖子，给用户[{}]发送推送,推送类型为[{}]，推送的数据：[{}]", userId,
                shareUserId, shareUserId, PushMessageEnum.FORWARDING.getType(),
                JSON.toJSONString(message));
    }

    /**
     * 帖子回复推送
     *
     * @param newsReply 评论信息
     */
    public void pushPostReplyMessage(NewsReply newsReply,ForumPostVo forumPostVo ) {
        if ( null != forumPostVo && !newsReply.getUserId().equals(forumPostVo.getUserId())) {
            String title = newsReply.getUserNickname() + "给你评论了";
            PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.BELONG_POST_REPLY);
            Long msgId = nextId();
            info.setMsgId(msgId);
            //添加用户图标
            info.addExtend("msgId", msgId);
            info.addExtend("newsId", newsReply.getNewsId());
            info.addExtend("postId", newsReply.getNewsId());
            info.addExtend("title", title);
            info.addExtend("icon", newsReply.getHeadImg());
            info.addExtend("nickname", newsReply.getUserNickname());
            info.addExtend("action", "给你评论了");
            info.addExtend("type", newsReply.getType());
            info.addExtend("postTitle", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
            info.addExtend("replyContent", newsReply.getReplyContent());
            this.fillExtendUrlOrContent(forumPostVo,info);
            PushMessage message = PushMessage.build();
            message.setTitle(title);
            message.setContent(newsReply.getReplyContent());
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPushReceiveScope(PushReceiveScope.pushSignle(forumPostVo.getUserId()));
            message.setPayloadInfo(info);
            this.addMessage(message, forumPostVo.getUserId());
            logger.debug("用户[{}]评论了用户[{}]的帖子，给用户[{}]发送推送,推送类型为[{}]， \n推送的数据：[{}]", newsReply.getUserId(),
                    forumPostVo.getUserId(), forumPostVo.getUserId(), PushMessageEnum.BELONG_POST_REPLY.getType(),
                    JSON.toJSONString(message));
        }

    }

    /**
     * 填补消息中的图片与内容
     * @param forumPostVo
     * @param info
     */
    private void fillExtendUrlOrContent(ForumPostVo forumPostVo, PushPayloadInfo info) {
        List<PostImgVo> img = forumPostVo.getPostImgList();
        if (!CollectionUtils.isEmpty(img)) {
            info.addExtend("extendUrl", img.get(0).getImgUrl());
        } else {
            info.addExtend("extendContent", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
        }
    }

    /**
     * 回复评论推送
     *
     * @param newsReply
     * @param pushMessageEnum
     */
    public void pushNewReplyMessage(NewsReply newsReply, PushMessageEnum pushMessageEnum,ForumPostVo forumPostVo, News news) {
        if (newsReply.getParentId() != 0 && Objects.nonNull(newsReply.getParentUserId()) && newsReply.getParentUserId() != 0) {
            UserBean userEntity = userIntegrationService.selectUserFromCache(newsReply.getParentUserId());
            if (Objects.nonNull(userEntity)) {
                String title = newsReply.getUserNickname() + "回复了你的评论";
                PushPayloadInfo info = PushPayloadInfo.build(pushMessageEnum);
                Long msgId = nextId();
                info.setMsgId(msgId);
                //添加用户图标
                info.addExtend("msgId", msgId);
                info.addExtend("icon", newsReply.getHeadImg());
                info.addExtend("replyId", newsReply.getId());
                info.addExtend("title", title);
                info.addExtend("newsId", newsReply.getNewsId());
                info.addExtend("type", newsReply.getType());
                info.addExtend("nickname", newsReply.getUserNickname());
                info.addExtend("action", "回复了你的评论");
                info.addExtend("replyContent", newsReply.getReplyContent());
                this.addPushPayloadInfo(newsReply, info, forumPostVo, news);
                PushMessage message = PushMessage.build();
                message.setTitle(title);
                if (newsReply.getDeleteFlag() == 0) {
                    message.setContent(newsReply.getReplyContent());
                } else {
                    message.setContent("该评论已删除！");
                }
                message.setType(TemplateTypeEnum.NOTIFCTION);
                message.setPushReceiveScope(PushReceiveScope.pushSignle(newsReply.getParentUserId()));
                message.setPayloadInfo(info);
                logger.info("推送的message:{}", JSONObject.toJSONString(message));
                this.addMessage(message, newsReply.getParentUserId());
                logger.debug("用户[{}]评论了用户[{}]的帖子，给用户[{}]发送推送,推送类型为[{}]， \n推送的数据：[{}]", newsReply.getUserId(),
                        newsReply.getParentUserId(), newsReply.getParentUserId(), pushMessageEnum.getType(),
                        JSON.toJSONString(message));
            }
        }

    }

    public void pushPostLikeMessage(ForumPostVo forumPostVo, Long userId, int isOld, int flowerNum) {
        //根据是否精华帖获得赞赏鲜花的数量
        UserBean userEntity = userIntegrationService.selectUserFromCache(userId);
        if (null != userEntity && null != forumPostVo && !userId.equals(forumPostVo.getUserId())) {
            String title;
            PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.BELONG_POST_LIKE);
            Long msgId = nextId();
            if (isOld == 0) {
                String  s = "给你点赞了，送你" + flowerNum + "朵小红花";
                title = userEntity.getNickname() +  s;
                info.addExtend("action", s);
            } else {
                title = userEntity.getNickname() + "点赞了你的帖子";
                info.addExtend("action", "点赞了你的帖子");
            }
            info.setMsgId(msgId);
            //添加用户图标
            info.addExtend("msgId", msgId);
            info.addExtend("postId", forumPostVo.getId());
            info.addExtend("newsId", forumPostVo.getId());
            info.addExtend("title", title);
            info.addExtend("icon", userEntity.getHeadImg());
            info.addExtend("nickname", userEntity.getNickname());
            info.addExtend("type", ReplyTypeEnum.POST_REPLY.getCode());
            info.addExtend("postTitle", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
            info.addExtend("flowerNum", flowerNum);
            this.fillExtendUrlOrContent(forumPostVo,info);
            PushMessage message = PushMessage.build();
            message.setTitle(title);
            //点赞帖子不显示标题
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPushReceiveScope(PushReceiveScope.pushSignle(forumPostVo.getUserId()));
            message.setPayloadInfo(info);
            this.addMessage(message, forumPostVo.getUserId());
            logger.debug("用户[{}]点赞了用户[{}]的帖子，给用户[{}]发送推送,推送类型为[{}]， \n推送的数据：[{}]", userId,
                    forumPostVo.getUserId(), forumPostVo.getUserId(), PushMessageEnum.BELONG_POST_LIKE.getType(),
                    JSON.toJSONString(message));
        }
    }

    /**
     * 添加推送的扩展参数
     *
     * @param newsReply 评论信息
     * @param info      推送主体消息
     */
    private void addPushPayloadInfo(NewsReply newsReply, PushPayloadInfo info, ForumPostVo forumPostVo, News news) {
        //帖子
        if (newsReply.getType() == 3) {
            if (!CollectionUtils.isEmpty(forumPostVo.getPostImgList())) {
                info.addExtend("extendUrl", forumPostVo.getPostImgList().get(0).getImgUrl());
            } else {
                info.addExtend("extendContent", ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()));
            }
            //小视频
        } else if (newsReply.getType() == 2) {
            Video video = videoMapper.selectByPrimaryKey(newsReply.getNewsId());
            String videoImg = video.getVideoImg();

            if (StringUtils.isNotBlank(videoImg)) {
                info.addExtend("extendUrl", videoImg);
            } else {
                info.addExtend("extendContent", ExtendFactory.getContent(video.getTitle(), null));
            }
            //新闻
        } else if (newsReply.getType() == 1) {
            String imgUrl = news.getImgUrl();
            List<String> images = JSONObject.parseArray(imgUrl, String.class);
            if (!CollectionUtils.isEmpty(images)) {
                info.addExtend("extendUrl", images.get(0));
            } else {
                info.addExtend("extendContent", ExtendFactory.getContent(news.getTitle(), news.getContent()));
            }
        }

    }

    /**
     * 点赞评论推送
     *
     * @param newsReplyLikeParam
     * @param pushMessageEnum
     */
    public void pushNewLikeMessage(NewsReplyLikeParam newsReplyLikeParam, PushMessageEnum pushMessageEnum, NewsReply newsReply,ForumPostVo forumPostVo, News news) {
        if (newsReplyLikeParam.getUserId() != 0) {
            UserBean userEntity = userIntegrationService.selectUserFromCache(newsReplyLikeParam.getUserId());
            if (Objects.nonNull(userEntity)) {
                UserBean user = userEntity;
                String title = user.getNickname() + "点赞了你的评论";
                PushPayloadInfo info = PushPayloadInfo.build(pushMessageEnum);
                Long msgId = nextId();
                info.setMsgId(msgId);
                //添加用户图标
                info.addExtend("msgId", msgId);
                info.addExtend("icon", user.getHeadImg());
                info.addExtend("replyId", newsReplyLikeParam.getReplyId());
                info.addExtend("title", title);

                if (null != newsReplyLikeParam.getNewsId()) {
                    info.addExtend("newsId", newsReplyLikeParam.getNewsId());
                    newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReplyLikeParam.getReplyId(), newsReplyLikeParam.getNewsId());
                } else {
                    newsReply = newsReplyMapper.selectByPrimaryKey(newsReplyLikeParam.getReplyId());
                }
                // 需要添加新闻类型
                info.addExtend("type", newsReply.getType());
                info.addExtend("nickname", user.getNickname());
                info.addExtend("action", "点赞了你的评论");
                info.addExtend("replyContent", newsReply.getReplyContent());
                //添加扩展参数
                this.addPushPayloadInfo(newsReply, info, forumPostVo, news);
                PushMessage message = PushMessage.build();
                message.setTitle(title);
                if (newsReply.getDeleteFlag() == 0) {
                    message.setContent(newsReply.getReplyContent());
                } else {
                    message.setContent("该评论已删除!");
                }
                message.setType(TemplateTypeEnum.NOTIFCTION);
                message.setPushReceiveScope(PushReceiveScope.pushSignle(newsReply.getUserId()));
                message.setPayloadInfo(info);
                this.addMessage(message, newsReply.getUserId());
                logger.debug("用户[{}]点赞了用户[{}]的评论，给用户[{}]发送推送,推送类型为[{}]， \n推送的数据：[{}]", newsReplyLikeParam.getUserId(),
                        newsReply.getUserId(), newsReply.getUserId(), pushMessageEnum.getType(),
                        JSON.toJSONString(message));
            }
        }
    }

    public void pushPostAddRecommendPool(ForumPostVo forumPostVo, MixedRecommendPool recommendPool) {
        UserBean userEntity = userIntegrationService.selectUserFromCache(forumPostVo.getUserId());
        if (null != userEntity) {
            String title;
            String extendContent = ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField());
            if (Objects.nonNull(recommendPool.getOptimizationNotice()) && recommendPool.getOptimizationNotice() == 1) {
                title = "你发布的【" + extendContent + "】被小编优化后推到头条啦，奖励你60朵小红花，你的内容将会被更多人看到哦~";
            } else {
                title = "你发布的【" + extendContent + "】被小编推到头条啦，奖励你60朵小红花，你的内容将会被更多人看到哦~";
            }
            PushPayloadInfo info = PushPayloadInfo.build(PushMessageEnum.POST_ADD_RECOMMAND);
            Long msgId = nextId();
            info.setMsgId(msgId);
            //添加用户图标
            info.addExtend("msgId", msgId);
            info.addExtend("postId", forumPostVo.getId());
            PushMessage message = PushMessage.build();
            message.setTitle("");
            message.setContent(title);
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPushReceiveScope(PushReceiveScope.pushSignle(forumPostVo.getUserId()));
            message.setPayloadInfo(info);
            this.addMessage(message, forumPostVo.getUserId());
            logger.debug("帖子[{}]加入推荐库，给用户[{}]发送推送,推送类型为[{}]， \n推送的数据：[{}]", forumPostVo.getId(),
                    forumPostVo.getUserId(),PushMessageEnum.POST_ADD_RECOMMAND.getType(),
                    JSON.toJSONString(message));
        }

    }

}
