package com.bxm.localnews.news.model.dto;

import com.bxm.localnews.news.model.enums.ForumContentPlaceHolderEnum;
import com.google.common.collect.Maps;
import lombok.Getter;
import lombok.Setter;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;

import java.util.Map;
import java.util.function.Supplier;

import static java.util.Objects.nonNull;

/**
 * @author gonzo
 * @date 2020-09-24 18:14
 **/
public class ContentContext {

    private String content;

    @Getter
    private Document document;

    /**
     * 0: 文档模式
     * 1: 字符串模式
     */
    private int model = 1;

    public ContentContext(String content) {
        this.content = content;
        // 默认切换到文档模式
        switchContentToDocumentModel();
    }

    /**
     * 要替换的map
     * key: 替换的key 如 areaName、userName等等
     * value: 要替换成的值的方法 通过调用get() 获取
     */
    @Getter
    private Map<ForumContentPlaceHolderEnum, Supplier<?>> replaceMap = Maps.newHashMap();

    /**
     * 添加要替换的类型
     * @param holderType 占位符类型
     * @param value 要替换成的值
     * @return this
     */
    public ContentContext addReplaceType(ForumContentPlaceHolderEnum holderType, Supplier<?> value) {

        if (nonNull(holderType) && nonNull(value)) {
            replaceMap.put(holderType, value);
        }

        return this;
    }

    public <T> Supplier<T> getReplaceValue(ForumContentPlaceHolderEnum holderType) {
        return (Supplier<T>) replaceMap.get(holderType);
    }

    /**
     * 从文档格式切换为字符串内容格式
     * 内部会判断 如果已经是内容模式则不会进行切换
     * 尽量不要和{@link #switchContentToDocumentModel}相互切换频繁调用
     */
    public void switchDocumentToContentModel() {
        if (this.model != 1) {
            this.content = document.body().html().replace("\n", "");
            this.model = 1;
        }
    }

    /**
     * 从文档格式切换为字符串内容格式
     * 内部会判断 如果已经是文档模式则不会进行切换
     * 尽量不要和{@link #switchDocumentToContentModel}相互切换频繁调用
     */
    public void switchContentToDocumentModel() {
        if (this.model != 0) {
            this.document = Jsoup.parse(this.content);
            document.outputSettings().prettyPrint(true).indentAmount(1);
            this.model = 0;
        }
    }

    public String getContent() {
        // 如果最后一次是文档模式 则进行转换
        return model == 0 ? document.body().html().replace("\n", "") : content;
    }

    /**
     * 设置内容
     * 内部会判断当前上下文的模式
     * 只有在内容模式的情况下 才设置成功
     * @param content 内容
     */
    public void setContent(String content) {
        if (this.model == 1) {
            this.content = content;
        }
    }
}
