package com.bxm.localnews.news.service;

import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.dto.ReplenishmentDTO;
import com.bxm.localnews.news.param.ForumPostInformParam;
import com.bxm.localnews.news.param.ForumPostListQueryParam;
import com.bxm.localnews.news.param.ForumPostListUserQueryParam;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;

import java.util.List;

/**
 * 社区相关服务
 */
public interface ForumService {

    /**
     * 查询版块
     *
     * @param areaCode 地区编码
     * @param enable   查询版块范围：0全部 1已上架
     * @return 分页结果
     */
    List<ForumVo> getForumList(String areaCode, Integer enable);

    /**
     * 获取版块详情
     *
     * @param id
     * @return
     */
    ForumVo selectForumById(Long id);

    /**
     * 查询话题
     *
     * @param areaCode 地区编码
     * @param enable   查询话题范围：0全部 1已上架
     * @return 分页结果
     */
    List<TopicVo> getTopicList(String areaCode, Integer enable);

    /**
     * 获取话题详情
     *
     * @param id
     * @return
     */
    TopicVo selectTopicById(Long id);

    /**
     * 根据版块或话题获取帖子列表
     *
     * @param param
     * @return
     */
    PageWarper<ForumPostVo> getPostPage(ForumPostListQueryParam param);

    /**
     * 根据用户及类型获取帖子列表
     *
     * @param param
     * @return
     */
    PageWarper<ForumPostVo> getPostPageByUser(ForumPostListUserQueryParam param);

    /**
     * 根据id获取帖子详情
     *
     * @param id
     * @param userId
     * @param deviceId
     * @return
     */
    ForumPostVo getPostDetailById(Long id, Long userId, String deviceId);

    /**
     * 新增或编辑帖子
     *
     * @param forumBasicVo
     * @return
     */
    int createOrUpdatePost(ForumBasicVo forumBasicVo);

    /**
     * 用户需要在发帖之后的一小时内，看到自己的帖子是置顶的
     * 把用户发布的帖子加入到redis用户帖子置顶中，并调用定时任务在一小时后进行清除置顶
     * @param forumPostVo
     */
    void createPostUserTopTaskJob(ForumPostVo forumPostVo);

    /**
     * 删除帖子
     *
     * @param id
     * @return
     */
    Boolean delete(Long id);

    /**
     * 分享帖子，在分享帖子后可以获得金币，每天会有次数限制
     *
     * @param userId
     * @param newsId
     * @param areaCode
     * @return
     */
    NewsCompleTaskDTO doShare(Long userId, Long newsId, String areaCode);

    /**
     * 填充其余参数
     * @param forumPostList
     * @param userId
     */
    void completePostListInfo(List<ForumPostVo> forumPostList, Long userId);

    /**
     * 对某篇帖子进行点赞操作
     *
     * @param userId
     * @param postId
     * @param type
     * @param areaCode
     */
    void doLike(Long userId, Long postId, int type, String areaCode);

    /**
     * 举报某篇帖子
     *
     * @param forumPostInformParam
     */
    Message doInform(ForumPostInformParam forumPostInformParam);

    /**
     * 收藏某篇帖子
     *
     * @param userId
     * @param postId
     */
    void doCollect(Long userId, Long postId);

    /**
     * 计算参与人数
     */
    void calcParticipantsNum();

    /**
     * 根据帖子id获取与帖子具有相关性的新闻和帖子
     * @param postId
     * @param userId
     * @param size 数量
     * @param areaCode 地区编号
     * @return
     */
    List<News4Client> listPostDetailRecommend(Long postId, Long userId, Integer size,String areaCode);

    /**
     * 获得推荐库中的一个热门帖子或者新闻用于补签
     * @param userId
     * @param areaCode
     * @return
     */
    ReplenishmentDTO getHotPostOrNews(Long userId,String areaCode);

    /**
     * 对帖子的消费次数进行消费，对帖子的次数进行增加
     *
     * @param ids
     */
    void doConsumeRecommendedCount(List<Long> ids,Long userId);

    /**
     * 对帖子的点击次数进行消费，对帖子的次数进行增加
     *
     * @param postId
     */
    void doConsumeClickCount(Long postId,Long userId);

    /**
     * 对帖子的有效阅读次数进行消费，对帖子的次数进行增加
     *
     * @param postId
     */
    void doConsumeActiveViewCount(Long postId,Long userId);

    /**
     * 更新帖子中用户头像以及昵称
     * @param userId
     * @param nickname
     * @param headImg
     */
    void triggerUpdateInfo(Long userId, String nickname, String headImg);
}
