package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.dto.NewsDetailDTO;
import com.bxm.localnews.integration.AppVersionIntegrationService;
import com.bxm.localnews.integration.NewsRecommendIntegrationService;
import com.bxm.localnews.integration.NewsSearchIntegrationService;
import com.bxm.localnews.integration.NewsStatisticIntegrationService;
import com.bxm.localnews.news.config.BizConfigProperties;
import com.bxm.localnews.news.constant.AppConst;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.enums.NewsSharEnum;
import com.bxm.localnews.news.param.NewsParam;
import com.bxm.localnews.news.service.AssemblyService;
import com.bxm.localnews.news.service.NewsService;
import com.bxm.localnews.news.service.RecommendService;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Service("newsService")
@Slf4j
public class NewsServiceImpl extends BaseService implements NewsService {
    @Resource
    private NewsMapper newsMapper;

    @Resource
    private NewsRecordMapper newsRecordMapper;

    @Resource
    private NewsStatisticsMapper newsStatisticsMapper;

    @Resource
    private BizConfigProperties bizConfigProperties;

    @Resource
    private NewsSearchWordMapper newsSearchWordMapper;

    @Resource
    private NewsKindMapper newsKindMapper;

    @Resource
    private NewsTagMapper newsTagMapper;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private NewsRecommendIntegrationService newsRecommendIntegrationService;

    @Resource
    private NewsSearchIntegrationService newsSearchIntegrationService;

    @Resource
    private NewsStatisticIntegrationService newsStatisticIntegrationService;

    @Resource
    private NewsCollectMapper newsCollectMapper;

    @Resource
    private AssemblyService assemblyService;

    @Override
    public Json<NewsUrlVo> getUrl(Long newsId, Long userId, Byte type) {
        //判断是否是新闻
        if (isNews(type)) {
            News news = this.newsMapper.selectByPrimaryKey(newsId);
            if (null != news) {
                String url = getNewsUrlBegin(type, news.getType()) + "newsId=" + newsId + "&userId=" + userId + "&type=" + type;
                NewsUrlVo newsUrlVo = new NewsUrlVo(url);
                return ResultUtil.genSuccessResult(newsUrlVo);
            }
        }

        log.error("获得详情地址时新闻不存在-newsId:{},userId:{},type:{}", newsId, userId, type);
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "获得详情地址时新闻不存在");
    }

    private boolean isEssayShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.ESSAY;
    }

    private boolean isIconEssayShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.ICON_ESSAY;
    }

    private boolean isVideoShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.VIDEO;
    }

    private boolean isEssayDetail(Byte type) {
        return type == 1 || type == 3;
    }

    private boolean isNews(Byte type) {
        return type != 4;
    }

    /**
     * 组装新闻url
     */
    private String getNewsUrlBegin(byte type, byte newsType) {
        String url = "";
        if (isEssayShare(type, newsType)) {
            url = "shareNewsDetail.html?";
        }
        if (isIconEssayShare(type, newsType)) {
            url = "sharePicDetail.html?";
        }
        if (isVideoShare(type, newsType)) {
            url = "shareVideoDetail.html?";
        }
        if (isEssayDetail(type)) {
            url = "newsDetail.html?";
        }
        return url;
    }


    /**
     * 是否处于提包状态
     */
    private Boolean getPublishState(BasicParam basicParam) {
        return appVersionIntegrationService.getPublishState(basicParam);
    }

    @Override
    public Json<NewsDetailVO> execGetSyDetail(NewsParam newsParam) {
        long begin = System.currentTimeMillis();
        if (null == newsParam.getNewsId()) {
            return ResultUtil.genSuccessResult(new NewsDetailVO());
        }
        log.info("新闻详情es开始拉取 ");
        List<NewsVO> newsVOList = newsSearchIntegrationService.multipleGet(new Long[]{newsParam.getNewsId()});

        if (newsVOList.isEmpty()) {
            log.error("新闻详情es拉取失败:{}", newsParam);
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "获取详情时新闻缺失");
        }
        NewsVO newsVO = newsVOList.get(0);

        if (newsVO.getStatus() != 1) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "新闻已下架");
        }

        newsVO.setCollect(0);
        if (newsParam.getUserId() != null) {
            newsVO.setCollect(countCollect(newsParam.getUserId(), newsParam.getNewsId()));
        }
        newsVO.setLocal((byte) 1);
        if (StringUtils.isNotEmpty(newsVO.getAreaDetail())) {
            newsVO.setLocal((byte) 2);
        }
        newsVO.setLinkUrl(buildNewsDetailUrl(newsParam.getNewsId(), newsParam.getUserId()));
        newsVO.setShareUrl(buildNewsShareUrl(newsParam.getNewsId(), newsParam.getUserId()));
        NewsDetailVO newsDetailVO = new NewsDetailVO(newsVO);

//        if (newsParam.getUserId() != null && newsParam.getType() != NewsSharEnum.NEWS_FRIEND_READ.getCode()) {
//            //TODO [liujia]用户每一次阅读均进行记录,这里会影响金币的发放
//            NewsRecord record = new NewsRecord();
//            record.setNewsId(newsParam.getNewsId());
//            record.setAddTime(new Date());
//            record.setUserId(newsParam.getUserId());
//            this.newsRecordMapper.insertSelective(record);
//        }
        if (getPublishState(newsParam)) {
            newsDetailVO.getNews().setAuthor(StringUtils.EMPTY);
            newsDetailVO.getNews().setAuthorImg(StringUtils.EMPTY);
        }

        //将用户行为记录上报消息中心
        if (null !=newsParam.getUserId()) {
            newsStatisticIntegrationService.pushNewClickData(newsParam.getUserId(),newsParam.getNewsId());
            newsStatisticIntegrationService.pushViewData(newsParam.getUserId(),newsParam.getNewsId());
        }

        log.info("[execGetSyDetail]新闻详情es拉取完毕, param:[{}] ,content:[{}],耗时:[{}]", newsParam, JSONObject.toJSON(newsDetailVO),System.currentTimeMillis() - begin);
        return ResultUtil.genSuccessResult(newsDetailVO);
    }

    private int countCollect(Long userId, Long newsId) {
        NewsCollect newsCollect = newsCollectMapper.selectUserCollect(newsId, userId, (byte) 1);
        return null != newsCollect ? 1 : 0;
    }

    @Override
    public Json getBack(Long newsId, Long userId, Integer viewTime, String tabType, String advertShow) {
        NewsRecord nRecord = this.newsRecordMapper.getByIds(newsId, userId, null);
        Date d = new Date();
        if (nRecord != null) {
            nRecord.setUpdateTime(d);
            //阅读时间大于上次阅读时间则更新
            if (viewTime > nRecord.getCheckTime()) {
                nRecord.setCheckTime(viewTime);
            }
            this.newsRecordMapper.updateByPrimaryKeySelective(nRecord);
        } else {
            NewsRecord nr = new NewsRecord();
            nr.setId(nextSequence());
            nr.setNewsId(newsId);
            nr.setUserId(userId);
            nr.setUpdateTime(d);
            nr.setCheckTime(viewTime);
            nr.setGetGold(AppConst.UN_GOLD);
            nr.setAddTime(d);
            nr.setLastLocation(tabType);
            nr.setAdViewType(Byte.valueOf(advertShow));
            this.newsRecordMapper.insertSelective(nr);
        }
        return ResultUtil.genSuccessResult();
    }

    @Override
    public Json<List<News4Client>> queryRecommendNews(Long newsId, Long userId,Integer size) {
        if (newsId == null || userId == null) {
            log.error("新闻详情推荐列表接口----参数有误,newsId:{},userId:{}",newsId,userId);
        }

        if (size == null ) {
            size =5;
        }
        List<Long> ids = newsRecommendIntegrationService.recommendByNewsDetail(newsId, userId,size);

        List<Long> nonNullIds = ids.stream().filter(Objects::nonNull).collect(Collectors.toList());
        if (!nonNullIds.isEmpty() ){
            nonNullIds = nonNullIds.stream().limit(size).collect(Collectors.toList());
            Long[] s = nonNullIds.toArray(new Long[0]);
            List<News> list = newsSearchIntegrationService.listRecommendedNews(s);
            fillExtInfo(userId,list);
            List<News4Client> news4ClientList = list.stream().map(news -> new News4Client(news, null)).collect(Collectors.toList());
            return ResultUtil.genSuccessResult(news4ClientList);
        }else{
            return  ResultUtil.genSuccessResult(new ArrayList<>());
        }

    }

    @Override
    public Json<List<News4Client>> queryRecommendNews4ImgNews(Long newsId) {
        //组图末尾推荐的组图
        List<News> list = this.newsMapper.queryRecommendNewsList(newsId, 8, null);
        List<News4Client> news4ClientList = new ArrayList<>();
        for (News aList : list) {
            news4ClientList.add(new News4Client(aList, null));
        }
        return ResultUtil.genSuccessResult(news4ClientList);
    }

    @Override
    public Json<List<NewsSearchWord>> searchHotTitle() {
        return ResultUtil.genSuccessResult(this.newsSearchWordMapper.queryList());
    }

    @Override
    public Json<List<NewsKind>> searchKinds() {
        return ResultUtil.genSuccessResult(this.newsKindMapper.selectVideoKinds());
    }

    @Deprecated
    @Override
    public Json<FailVideo> getFailVideo(Long newsId) {
        //新闻视频无法播放的处理
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "无效视频");
    }

    @Override
    public Json addClick(Long newsId) {
        this.newsStatisticsMapper.searchRecommended(newsId);
        return ResultUtil.genSuccessResult();
    }

    @Override
    public void delNewsTag() {
        this.newsTagMapper.delByDate(DateUtils.addField(new Date(), Calendar.DATE, -2));
    }

    @Override
    public String buildNewsDetailUrl(Long newsId, Long userId) {
        if (userId==null) {
            return bizConfigProperties.getH5ServerHost() + "/newsDetail.html?" + "newsId=" + newsId + "&type=" + 1;
        }
        return bizConfigProperties.getH5ServerHost() + "/newsDetail.html?" + "newsId=" + newsId + "&userId=" + userId + "&type=" + 1;
    }

    @Override
    public String buildNewsShareUrl(Long newsId, Long userId) {
        if (userId==null) {
            return bizConfigProperties.getH5ServerHost() + "/shareNewsDetail.html?" + "newsId=" + newsId + "&type=" + 2;
        }
        return bizConfigProperties.getH5ServerHost() + "/shareNewsDetail.html?" + "newsId=" + newsId + "&userId=" + userId + "&type=" + 2;
    }

    @Override
    public void fillExtInfo(Long userId, List<News> result) {
        List<NewsCollect> newsCollect = newsCollectMapper.selectByIds(userId, (byte) 1);

        result.forEach(item -> {
            item.setLinkUrl(buildNewsDetailUrl(item.getId(), userId));
            item.setShareUrl(buildNewsShareUrl(item.getId(), userId));
            item.setCollect(0);
            if (CollectionUtils.isNotEmpty(newsCollect)) {
                for (NewsCollect collect : newsCollect) {
                    if (item.getId().equals(collect.getNewsId())) {
                        item.setCollect(1);
                        break;
                    }
                }
            }
            item.setLocal((byte) 1);
            if (StringUtils.isNotEmpty(item.getAreaDetail())) {
                item.setLocal((byte) 2);
            }
        });

    }

    @Override
    public Json<NewsDetailVO> getNewsDetailById(NewsParam newsParam) {
        Json<NewsDetailVO> data = execGetSyDetail(newsParam);
        NewsDetailVO result = data.getResult();
        NewsVO news = result.getNews();
        String content = assemblyService.assemblyContent(news.getContent());
        news.setContent(content);
        result.setNews(news);
        data.setResult(result);
        return data;
    }
}
