package com.bxm.localnews.news.service.impl;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import com.bxm.localnews.common.vo.Json;
import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.common.config.NewsProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.integration.feign.NewsRecommendFeignService;
import com.bxm.localnews.integration.feign.NewsSearchFeignService;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.ESNewsContentDTO;
import com.bxm.localnews.news.dto.VideoDto;
import com.bxm.localnews.news.enums.ShowLevelEnum;
import com.bxm.localnews.news.param.VideoQueryParam;
import com.bxm.localnews.news.service.RecommendService;
import com.bxm.localnews.news.service.VideoRecommendService;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.google.common.collect.Lists;


@Service("recommendService")
public class RecommendServiceImpl extends BaseService implements RecommendService {

    /**
     * 每页获取新闻数量上限
     */
    private static final int                     MAX_PAGE_SIZE     = 20;

    /**
     * 用户缓存警戒值，达到此值就进行用户缓存的追加
     */
    private static final int                     CACHE_ALARM_VALUE = MAX_PAGE_SIZE * 3;

    private NewsMapper                           newsMapper;

    private NewsRecommendedMapper                newsRecommendedMapper;

    private MarketingActivitiesMapper            marketingActivitiesMapper;

    private MarketingActivitiesRecommendedMapper marketingActivitiesRecommendedMapper;

    private AppVersionSupplyService              appVersionSupplyService;

    private NewsProperties                       newsProperties;

    private VideoRecommendService                videoRecommendService;

    private RedisListAdapter                     redisListAdapter;

    private RedisStringAdapter                   redisStringAdapter;

    private NewsPoolMapper                       newsPoolMapper;

    private NewsRecommendFeignService            newsRecommendFeignService;

    private NewsSearchFeignService               newsSearchFeignService;

    private BizConfigProperties                  bizConfigProperties;

    private DistributedLock distributedLock;

    @Autowired
    public RecommendServiceImpl(NewsMapper newsMapper, NewsRecommendedMapper newsRecommendedMapper,
                                NewsProperties newsProperties, AppVersionSupplyService appVersionSupplyService,
                                VideoRecommendService videoRecommendService, RedisListAdapter redisListAdapter,
                                RedisStringAdapter redisStringAdapter, NewsPoolMapper newsPoolMapper,
                                NewsRecommendFeignService newsRecommendFeignService,
                                NewsSearchFeignService newsSearchFeignService,
                                MarketingActivitiesMapper marketingActivitiesMapper,
                                MarketingActivitiesRecommendedMapper marketingActivitiesRecommendedMapper,
                                BizConfigProperties bizConfigProperties,
                                DistributedLock distributedLock){
        this.newsMapper = newsMapper;
        this.newsRecommendedMapper = newsRecommendedMapper;
        this.newsProperties = newsProperties;
        this.appVersionSupplyService = appVersionSupplyService;
        this.videoRecommendService = videoRecommendService;
        this.redisListAdapter = redisListAdapter;
        this.redisStringAdapter = redisStringAdapter;
        this.newsPoolMapper = newsPoolMapper;
        this.newsRecommendFeignService = newsRecommendFeignService;
        this.newsSearchFeignService = newsSearchFeignService;
        this.marketingActivitiesMapper = marketingActivitiesMapper;
        this.marketingActivitiesRecommendedMapper = marketingActivitiesRecommendedMapper;
        this.bizConfigProperties = bizConfigProperties;
        this.distributedLock = distributedLock;
    }

    @Override
    public Json<NewsMeta> execRecommend(NewsRecommendParam param) {
        int count = 0;
        return retry(param, count);
    }

    private Json<NewsMeta> retry(NewsRecommendParam param,int count){
        String requestId = String.valueOf(nextSequence());
        String userId = String.valueOf(param.getUserId());
        NewsMeta meta = new NewsMeta();
        List<Long> idList = new ArrayList<>();
        if (distributedLock.lock(userId, requestId)) {
            // 调用推荐系统提供的推荐服务
            ResponseEntity<List<Long>> responseEntity = newsRecommendFeignService.recommendList(param.getUserId(),
                    param.getKindId().intValue(),
                    param.getPagesize(),
                    param.getAreaCode(),
                    param.getCurPage());
            idList = responseEntity.getBody();
            logger.info("用户id:{},推荐引擎返回:{}",param.getUserId(), responseEntity.getBody());
            //解锁
            distributedLock.unlock(userId, requestId);
        }else {
            logger.info("新闻推荐请求过快，请休息一会!参数:{}",JSON.toJSONString(param));
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            // 休息 100ms
            retry(param, count++);
        }

        if (CollectionUtils.isEmpty(idList)) {
            return ResultUtil.genSuccessResult(meta);
        }
        // 获取新闻信息
        List<News> list = newsMapper.findNewsByIds(idList);

        Long[] ids = list.stream().mapToLong(News::getId).boxed().toArray(Long[]::new);
        // 获取新闻内容
        ResponseEntity<List<ESNewsContentDTO>> newsRspEntity = newsSearchFeignService.multipleGet(ids);
        List<ESNewsContentDTO> dtos = newsRspEntity.getBody();
        for (News news : list) {
            if (dtos != null && !dtos.isEmpty()) {
                Optional<ESNewsContentDTO> optional = dtos.stream().filter(dto -> dto.getId().longValue() == news.getId().longValue()).findFirst();
                news.setContent(optional.map(ESNewsContentDTO::getContent).orElse(null));
                news.setLinkUrl(getNewsDetailUrl(news.getId(),param.getUserId()));
            }
        }
        List<News> result = new ArrayList<>();
        // 获取营销活动
        List<News> activityNews = addMarketingActivities(param.getUserId(), param.getAreaCode());
        if (CollectionUtils.isNotEmpty(activityNews)) {
            result.addAll(activityNews);
            result = result.stream().sorted(Comparator.comparing(News::getShowLevel)).collect(Collectors.toList());
        }

        for (Long id : idList) {
            Optional<News> optionalNews = list.stream().filter(news -> news.getId().longValue() == id.longValue()).findFirst();
            optionalNews.ifPresent(result::add);
        }

        result = warp(result, param.getPagesize());

        List<News4Client> news4ClientList = result.stream().map(news -> new News4Client(news,
                null)).collect(Collectors.toList());
        // 添加小视频
        if (param.getKindId() == null || this.newsProperties.getRecommendKindId() == param.getKindId()) {
            addRecommendVideo(param, news4ClientList);
        }
        meta.setList(news4ClientList);
        meta.setNewsCount(meta.getList().size());
        // 隐藏敏感信息
        hideInfo(param, meta);
        logger.info("推荐完成：{}", meta);
        return ResultUtil.genSuccessResult(meta);
    }


    /**
     * 得到文章详情地址
     * @param newsId
     * @param userId
     * @return
     */
    private String getNewsDetailUrl(Long newsId,Long userId){
        return  bizConfigProperties.getH5ServerHost()+"/newsDetail.html?" + "newsId=" + newsId + "&userId=" + userId + "&type=" + 1;
    }

    /**
     * 推荐小视频，每2页显示一次小视频
     * 
     * @param param 推荐参数
     * @param news4ClientList 推荐返回结果元对象
     */
    private void addRecommendVideo(NewsRecommendParam param, List<News4Client> news4ClientList) {
        logger.info("NewsRecommendParam：{}", param);
        if (param.getCurPage() != 0 && param.getCurPage() % 2 == 0) {
            VideoQueryParam videoParam = new VideoQueryParam();
            videoParam.setUserId(param.getUserId());
            // 在新闻列表中间插入
            List<VideoDto> videoDtoList = videoRecommendService.execRecommend(videoParam);
            if (CollectionUtils.isNotEmpty(videoDtoList)) {
                int size = news4ClientList.size();
                news4ClientList.add(size / 2, new News4Client(videoDtoList));
            }
        }
    }

    /**
     * 插入营销活动
     * 
     * @param userId
     * @return
     */
    private List<News> addMarketingActivities(Long userId, String areaCode) {
        List<MarketingActivities> list = marketingActivitiesMapper.listForRecommend(userId, areaCode);
        List<News> newsList = new ArrayList<>();
        for (MarketingActivities activities : list) {
            News news = new News();
            news.setId(activities.getId());
            news.setTitle(activities.getTitle());
            news.setAuthor(activities.getAuthor());
            news.setContent(activities.getLinkUrl());
            news.setKindId(activities.getDeliveryChannel());
            news.setActivity((byte) 2);// 1否2是 这个奇葩数值兼容前人挖的坑
            news.setIssueTime(activities.getShowTime());
            if (activities.getCoverUrl() != null) {
                String[] coverUrls = StringUtils.split(activities.getCoverUrl(), ",");
                news.setImgNum(coverUrls.length);
                news.setImgUrl(JSON.toJSONString(coverUrls));
            }
            news.setHot((byte) 1);// 1否
            news.setTop((byte) 1);// 1否
            news.setType((byte) 1);// '新闻类型 1：文章 2：组图 3：视频',
            news.setShowLevel(activities.getShowLevel());
            news.setShowLevelDetail(activities.getShowLevelDetail());
            news.setDeliveryType(activities.getDeliveryType());
            news.setAreaDetail(activities.getAreaDetail());
            news.setLinkUrl(activities.getLinkUrl());
            newsList.add(news);
            marketingActivitiesRecommendedMapper.save(userId, activities.getId());
        }
        return newsList;
    }

    /**
     * 对标签逻辑进行包装
     * 
     * @param list
     * @return
     */
    private List<News> warp(List<News> list, Integer pageSize) {
        list.forEach(news->{
            news.setTop((byte) 1);
            news.setHot((byte) 1);});
        List<News> result = new ArrayList<>();
        News news;
        for (int i = 0; i < (pageSize > list.size() ? list.size() : pageSize); i++) {
            news = list.get(i);
            logger.info("title:{},getShowLevelDetail:{}", news.getTitle(), news.getShowLevelDetail());
            logger.info("news.getShowLevelDetail().indexOf(ShowLevelEnum.TOP.getCode().toString())>-1:{}",
                        news.getShowLevelDetail().indexOf(ShowLevelEnum.TOP.getCode().toString()) > -1);
            if (news.getShowLevelDetail() != null
                && news.getShowLevelDetail().indexOf(ShowLevelEnum.TOP.getCode().toString()) > -1) {
                news.setTop((byte) 2);
            }
            if (news.getShowLevelDetail() != null
                && news.getShowLevelDetail().indexOf(ShowLevelEnum.HOT.getCode().toString()) > -1) {
                news.setHot((byte) 2);
            }
            if (news.getShowLevelDetail() != null
                && news.getShowLevelDetail().indexOf(ShowLevelEnum.ACTIVITY.getCode().toString()) > -1) {
                news.setActivity((byte) 2);
            }
            if (news.getShowLevelDetail() != null
                && news.getShowLevelDetail().indexOf(ShowLevelEnum.LOCAL.getCode().toString()) > -1) {
                news.setLocal((byte) 2);
            }
            result.add(news);
        }
        return result;
    }



    /**
     * 处于提包状态时，隐藏部分敏感信息，用于上包 去除作者信息
     * 
     * @param param 推荐参数
     * @param meta 返回结果
     */
    private void hideInfo(NewsRecommendParam param, NewsMeta meta) {
        if (appVersionSupplyService.getPublishState(param)) {
            meta.getList().removeIf(client -> client.getVideos() != null);
            for (News4Client client : meta.getList()) {
                if (client.getNews() != null) {
                    client.getNews().setAuthor(null);
                    client.getNews().setAuthorImg(null);
                }
            }
        }
    }

    /**
     * 查询置顶新闻
     */
    @Override
    public Json<NewsMeta> getTopNewsList(NewsTopListParam param) {

        // 置顶新闻列表
        List<News> topNewsList = Lists.newArrayList();

        // 区域性性置顶新闻
        List<News> topNewsOfAreaList = null;
        KeyGenerator keyGeneratorOfArea = RedisConfig.TOP_NEW_OF_AREA_LIST.copy().appendKey(param.getAreaCode());
        String topNewsOfAreaListStr = redisStringAdapter.getString(keyGeneratorOfArea);
        if (StringUtils.isNotBlank(topNewsOfAreaListStr)) {
            topNewsOfAreaList = JSON.parseArray(topNewsOfAreaListStr, News.class);
//            if (topNewsOfAreaList != null && topNewsOfAreaList.size() > 0) {
//                topNewsList.addAll(topNewsOfAreaList);
//            }
        } else {
            topNewsOfAreaList = newsMapper.findTopNewsOfAreaList(param.getAreaCode());
            if (topNewsOfAreaList != null && topNewsOfAreaList.size() > 0) {
//                topNewsList.addAll(topNewsOfAreaList);
                redisStringAdapter.set(keyGeneratorOfArea, JSON.toJSONString(topNewsOfAreaList));
            }
        }


        // 全国性置顶新闻
        List<News> topNewsOfWholeCountryList = null;
        KeyGenerator keyGeneratorOfWholeCountry = RedisConfig.TOP_NEW_OF_WHOLE_COUNTRY_LIST;
        String topNewsOfWholeCountryListStr = redisStringAdapter.getString(keyGeneratorOfWholeCountry);
        if (StringUtils.isNotBlank(topNewsOfWholeCountryListStr)) {
            topNewsOfWholeCountryList = JSON.parseArray(topNewsOfWholeCountryListStr, News.class);
//            if (topNewsOfWholeCountryList != null && topNewsOfWholeCountryList.size() > 0) {
//                topNewsList.addAll(topNewsOfWholeCountryList);
//            }
        } else {
            topNewsOfWholeCountryList = newsMapper.findTopNewsOfWholeCountryList();
            if (topNewsOfWholeCountryList != null && topNewsOfWholeCountryList.size() > 0) {
//                topNewsList.addAll(topNewsOfWholeCountryList);
                redisStringAdapter.set(keyGeneratorOfWholeCountry, JSON.toJSONString(topNewsOfWholeCountryList));
            }
        }

        topNewsList = this.filterTopNews(topNewsOfAreaList,topNewsOfWholeCountryList);

        // 组装结构数据
        NewsMeta newsMeta = new NewsMeta();
        if (topNewsList.size() == 0) {
            return ResultUtil.genSuccessResult(newsMeta);
        }
        List<News4Client> news4ClientList = Lists.newArrayList();

        // 最多取3条置顶数据
        topNewsList = warp(topNewsList, 2);

        for (News news : topNewsList) {
            News4Client news4Client = new News4Client();
            news4Client.setNews(news);
            news4ClientList.add(news4Client);
        }
        newsMeta.setList(news4ClientList);
        newsMeta.setNewsCount(newsMeta.getList().size());
        // 隐藏敏感信息
        this.hideInfo(param, newsMeta);

        return ResultUtil.genSuccessResult(newsMeta);
    }

    /**
     * 填充置顶新闻
     * 置顶条数限制改为最多显示两条
     * 当本地置顶和全国置顶都存在时显示一条本地一条全国，最新编辑的优先展示
     * 当只有本地资讯时展示最多两条本地置顶，最新编辑的优先展示
     * 当只有全国资讯时展示最多两条全国置顶，最新编辑的优先展示
     * @param topNewsOfAreaList
     * @param topNewsOfWholeCountryList
     * @return
     */
    private List<News> filterTopNews(List<News> topNewsOfAreaList,List<News> topNewsOfWholeCountryList){
        List<News> topNewsList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(topNewsOfAreaList)&&CollectionUtils.isNotEmpty(topNewsOfWholeCountryList)) {
            topNewsList.add(topNewsOfAreaList.get(0));
            topNewsList.add(topNewsOfWholeCountryList.get(0));
            return topNewsList;
        } else if (CollectionUtils.isNotEmpty(topNewsOfAreaList)&&CollectionUtils.isEmpty(topNewsOfWholeCountryList)) {
            topNewsList.addAll(topNewsOfAreaList);
        }else if(CollectionUtils.isNotEmpty(topNewsOfWholeCountryList)&&CollectionUtils.isEmpty(topNewsOfAreaList)){
            topNewsList.addAll(topNewsOfWholeCountryList);
        }
        topNewsList = topNewsList.stream().limit(2).collect(Collectors.toList());
        return topNewsList;

    }

    /**
     * 处于提包状态时，隐藏部分敏感信息，用于上包 去除作者信息
     * 
     * @param param 置顶参数
     * @param meta 返回结果
     */
    private void hideInfo(NewsTopListParam param, NewsMeta meta) {
        if (appVersionSupplyService.getPublishState(param)) {
            meta.getList().removeIf(client -> client.getVideos() != null);
            for (News4Client client : meta.getList()) {
                if (client.getNews() != null) {
                    client.getNews().setAuthor(null);
                    client.getNews().setAuthorImg(null);
                }
            }
        }
    }
}
