package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.news.domain.VideoMapper;
import com.bxm.localnews.news.domain.VideoReplayLikeMapper;
import com.bxm.localnews.news.domain.VideoReplyMapper;
import com.bxm.localnews.news.dto.VideoReplyDto;
import com.bxm.localnews.news.param.VideoReplyLikeParam;
import com.bxm.localnews.news.param.VideoReplyListQueryParam;
import com.bxm.localnews.news.param.VideoReplyParam;
import com.bxm.localnews.news.service.VideoReplyFacadeService;
import com.bxm.localnews.news.service.VideoReplyService;
import com.bxm.localnews.news.vo.Video;
import com.bxm.localnews.news.vo.VideoReplayLike;
import com.bxm.localnews.news.vo.VideoReply;
import com.bxm.localnews.user.dto.UserBaseInfoWarper;
import com.bxm.localnews.user.service.UserService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;

@Service
public class VideoReplyServiceImpl extends BaseService implements VideoReplyService, VideoReplyFacadeService {

    @Autowired
    private VideoReplyMapper videoReplyMapper;

    @Autowired
    private UserService userService;

    @Autowired
    private RedisListAdapter redisListAdapter;

    @Autowired
    private VideoMapper videoMapper;

    @Autowired
    private VideoReplayLikeMapper videoReplayLikeMapper;

    @Override
    public PageWarper<VideoReplyDto> queryByPageSize(VideoReplyListQueryParam param) {
        List<VideoReplyDto> videoReplyDtoList = videoReplyMapper.queryByPageSize(param);
        for (VideoReplyDto videoReplyDto : videoReplyDtoList) {
            VideoReplyDto sourceReply = videoReplyDto.getSourceReply();
            if (null != sourceReply) {
                if (isLike(param.getUserId(), sourceReply.getId())) {
                    sourceReply.setIsLike(1);
                }
            }

            if (isLike(param.getUserId(), videoReplyDto.getId())) {
                videoReplyDto.setIsLike(1);
            }
        }
        return new PageWarper<>(videoReplyDtoList);
    }

    /**
     * 判断当前用户是否对此回复点赞
     *
     * @param userId
     * @param replyId
     * @return
     */
    private boolean isLike(Long userId, Long replyId) {
        VideoReplayLike videoReplayLike = videoReplayLikeMapper.selectByModel(new VideoReplayLike(replyId, userId));
        if (null != videoReplayLike) {
            return true;
        }
        return false;
    }

    @Override
    public Message replyVideo(VideoReplyParam videoReplyParam) {
        if (!checkReplyParam(videoReplyParam)) {
            return Message.build(false, "参数验证错误");
        }

        VideoReply videoReply = convertReply(videoReplyParam);
        videoReplyMapper.insertSelective(videoReply);
        // 更新新闻评论数
        Video video = videoMapper.selectByPrimaryKey(videoReplyParam.getVideoId());
        if (null != video) {
            Video videoUp = new Video();
            videoUp.setId(videoReplyParam.getVideoId());
            videoUp.setCommentCount(video.getCommentCount() + 1);
            videoMapper.updateByPrimaryKeySelective(videoUp);
        }
        return Message.build(true);
    }

    /**
     * 评论回复时验证参数
     *
     * @param videoReplyParam
     * @return
     */
    private boolean checkReplyParam(VideoReplyParam videoReplyParam) {
        if (null == videoReplyParam.getUserId()
                || null == videoReplyParam.getComment() || null == videoReplyParam.getVideoId()) {
            return false;
        }
        return true;
    }

    /**
     * 转换param为实体
     *
     * @param param
     * @return
     */
    private VideoReply convertReply(VideoReplyParam param) {
        Long userId = param.getUserId();
        UserBaseInfoWarper userBaseInfoWarper = userService.getUserBaseInfo(userId);

        VideoReply videoReply = new VideoReply();
        videoReply.setVideoId(param.getVideoId());
        videoReply.setComment(param.getComment());
        videoReply.setUserId(userId);
        videoReply.setUserName(userBaseInfoWarper.getUserName());
        videoReply.setUserImg(userBaseInfoWarper.getUserImg());
        videoReply.setAddTime(new Date());
        videoReply.setReplyId(param.getReplyId());
        videoReply.setLikeCount(0);
        videoReply.setIsDeleted((byte) 0);
        videoReply.setId(nextSequence());

        return videoReply;
    }

    @Override
    public Message produceVideoReplyLike(VideoReplyLikeParam videoReplyLikeParam) {
        if (!checkLikeParam(videoReplyLikeParam)) {
            return Message.build(false, "参数验证错误");
        }
        KeyGenerator keyGenerator = RedisConfig.VIDEO_QUEUE.copy().setKey("videoReplyLikeQueue");

        redisListAdapter.leftPush(keyGenerator, videoReplyLikeParam);
        this.callAsyncConsume();
        return Message.build(true);

    }

    private void callAsyncConsume() {
        VideoReplyService videoReplyService = SpringContextHolder.getBean(VideoReplyService.class);
        videoReplyService.videoReplyLikeConsume();
    }

    @Async
    @Override
    public void videoReplyLikeConsume() {
        KeyGenerator keyGenerator = RedisConfig.VIDEO_QUEUE.copy().setKey("videoReplyLikeQueue");

        VideoReplyLikeParam videoReplyLikeWarper = redisListAdapter.rightPop(keyGenerator, VideoReplyLikeParam.class);
        if (null != videoReplyLikeWarper) {
            logger.info("小视频回复点赞开始消费...");
            int type = videoReplyLikeWarper.getType();
            VideoReply videoReply = videoReplyMapper.selectByPrimaryKey(videoReplyLikeWarper.getReplyId());
            if (videoReply != null) {
                int count = saveReplyLiKeRecord(videoReply.getId(), videoReplyLikeWarper.getUserId(), type);

                if (count > 0) {
                    logger.debug("评论实体:{}", JSONObject.toJSON(videoReply));
                    int likeCount = generatorLikeCount(type, videoReply.getLikeCount());
                    logger.debug("点赞数:{}", likeCount);
                    VideoReply videoReplyUp = new VideoReply();
                    videoReplyUp.setId(videoReplyLikeWarper.getReplyId());
                    videoReplyUp.setLikeCount(likeCount);
                    videoReplyMapper.updateByPrimaryKeySelective(videoReplyUp);
                }
            }
            videoReplyLikeConsume();
        }
    }

    /**
     * 得到更新之后的点赞数
     *
     * @param type
     * @param likeCount
     * @return
     */
    private int generatorLikeCount(int type, int likeCount) {
        if (0 == type) {
            if (likeCount > 0) {
                likeCount -= 1;
            }
        } else {
            likeCount += 1;
        }
        return likeCount;
    }

    private int saveReplyLiKeRecord(Long replyId, Long userId, int type) {
        int count = 0;
        VideoReplayLike videoReplayLike = videoReplayLikeMapper.selectByModel(new VideoReplayLike(replyId, userId));
        if (type == 1) {
            if (null == videoReplayLike) {
                VideoReplayLike videoReplayLikeNew = new VideoReplayLike();
                videoReplayLikeNew.setAddTime(new Date());
                videoReplayLikeNew.setId(nextSequence());
                videoReplayLikeNew.setReplyId(replyId);
                videoReplayLikeNew.setUserId(userId);
                count = videoReplayLikeMapper.insert(videoReplayLikeNew);
            }
        } else {
            if (null != videoReplayLike) {
                count = videoReplayLikeMapper.deleteByPrimaryKey(videoReplayLike.getId());
            }
        }
        return count;
    }


    @Override
    public void triggerUpdateInfo(Long userId, String nickname, String headImg) {
        VideoReply videoReply = new VideoReply();
        videoReply.setUserId(userId);
        videoReply.setUserImg(headImg);
        videoReply.setUserName(nickname);
        videoReplyMapper.updateByUserIdSelective(videoReply);
    }

    /**
     * 评论点赞时验证参数
     *
     * @param videoReplyLikeParam
     * @return
     */
    private boolean checkLikeParam(VideoReplyLikeParam videoReplyLikeParam) {
        if (null == videoReplyLikeParam.getReplyId() || null == videoReplyLikeParam.getUserId()) {
            return false;
        }
        return true;
    }

}
