package com.bxm.localnews.news.convert.impl;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import com.bxm.localnews.news.convert.Converter;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.util.AreaCodeUtils;
import com.bxm.localnews.news.util.UploadUtils;
import com.bxm.localnews.news.vo.*;

import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.service.BaseService;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class ForumBasicConvert extends BaseService implements Converter<ForumBasicVo, ForumPostVo> {

    private final ForumTopicMapper forumTopicMapper;

    private final UploadUtils uploadUtils;

    @Override
    public ForumPostVo convert(ForumBasicVo data) {
        ForumPostVo forumPostVo = new ForumPostVo();
        //emoji表情编码替换
        data.setTextField(EmojiCodeParser.replaceSoftbankEmoji(data.getTextField()));
        BeanUtils.copyProperties(data, forumPostVo);
        //新人一键发布的帖子无需审核
        if (Objects.nonNull(data.getIsNewReport()) && data.getIsNewReport() == 1) {
            forumPostVo.setStatus(6);
        } else {
            forumPostVo.setStatus(2);
        }

        if (StringUtils.isBlank(forumPostVo.getLocation())) {
            forumPostVo.setLocation(null);
        }

        forumPostVo.setDeliveryType(0);
        if (StringUtils.isNotBlank(forumPostVo.getAreaCode())) {
            forumPostVo.setDeliveryType(1);
            logger.info("");
            forumPostVo.setAreaCode(AreaCodeUtils.completeAreaCode(forumPostVo.getAreaCode()));
        }

        if (CollectionUtils.isNotEmpty(data.getTopicIdList())) {
            TopicVo topic = forumTopicMapper.selectTopicById(data.getTopicIdList().get(0), null);
            if (topic != null) {
                data.setForumContent(topic.getContent());
            }
        }

        //组装content
        StringBuilder content = new StringBuilder("<p>");
        if (StringUtils.isNotBlank(forumPostVo.getTextField())) {
            content.append(forumPostVo.getTextField().replaceAll("\\n", "</p><p>"));
        }
        content.append("</p>");
        if (StringUtils.isNotBlank(forumPostVo.getExtraContent())) {
            content.append("<p>");
            content.append(forumPostVo.getExtraContent());
            content.append("</p>");
        }
        if (CollectionUtils.isNotEmpty(forumPostVo.getPostImgList())) {
            dealPostImgList(forumPostVo, content);
        }
        forumPostVo.setContent(content.toString());
        if (StringUtils.isBlank(data.getLocation())) {
            forumPostVo.setLocation(null);
        }

        return forumPostVo;
    }

    private void dealPostImgList(ForumPostVo forumPostVo, StringBuilder content) {
        forumPostVo.getPostImgList().forEach(e -> {
            if (StringUtils.isBlank(e.getType())) {
                e.setType("IMG");
            }
        });
        //保证视频在图片的前面
        List<PostImgVo> postVideoList = forumPostVo.getPostImgList().stream().filter(e -> "VIDEO".equals(e.getType())).collect(Collectors.toList());
        List<PostImgVo> postImgList = forumPostVo.getPostImgList().stream().filter(e -> "IMG".equals(e.getType())).collect(Collectors.toList());

        for (PostImgVo postImgVo : postVideoList) {
            if (StringUtils.isNotBlank(postImgVo.getVideoUrl())) {
                content.append("<p>&nbsp;<video controls=\"controls\" src=\"");
                content.append(postImgVo.getVideoUrl());
                content.append("\" poster=\"");
                content.append(postImgVo.getImgUrl());
                content.append("\" size=\"");
                content.append(postImgVo.getSize() == null ? 0 : postImgVo.getSize());
                content.append("\" duration=\"");
                content.append(postImgVo.getDuration() == null ? 0 : postImgVo.getDuration());
                content.append("\"></video>&nbsp;</p>");
            }
        }

        for (PostImgVo postImgVo : postImgList) {
            String imgUrl = postImgVo.getImgUrl();
            //这里主要是为了客户端图片没有上传oss而是传了第三方图片，会过期，所以要在这里判断[TMD]
            if (!imgUrl.contains("wstong.com")) {
                imgUrl = uploadUtils.uploadImgToOssBySrc(imgUrl);
            }
            if (StringUtils.isNotBlank(postImgVo.getImgUrl())) {
                content.append("<p><img src=\"");
                content.append(imgUrl);
                content.append("\"></img></p>");
            }
        }

        postVideoList.addAll(postImgList);
        forumPostVo.setPostImgList(postVideoList);
    }

}
