package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.integration.AppVersionIntegrationService;
import com.bxm.localnews.news.config.BizConfigProperties;
import com.bxm.localnews.news.constant.NewsTemplateConst;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.service.AssemblyService;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.tools.FileUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

@Slf4j
@Service
public class AssemblyServiceImpl implements AssemblyService {

    private final AliyunOSSService aliyunOSSService;

    private final AppVersionIntegrationService appVersionIntegrationService;

    private final BizConfigProperties bizConfigProperties;

    @Autowired
    public AssemblyServiceImpl(AliyunOSSService aliyunOSSService,
                               AppVersionIntegrationService appVersionIntegrationService,
                               BizConfigProperties bizConfigProperties) {
        this.aliyunOSSService = aliyunOSSService;
        this.appVersionIntegrationService = appVersionIntegrationService;
        this.bizConfigProperties = bizConfigProperties;
    }


    @Override
    public String assemblyContent(String target, BasicParam basicParam, String ip, byte type, Long userId, String areaCode,Long newsId) {
        String newsTemplate = NewsTemplateConst.getNewsTemplate();
        if (StringUtils.isBlank(newsTemplate)) {
            newsTemplate = aliyunOSSService.getOssContentByPath(NewsTemplateConst.getPath());
            NewsTemplateConst.setNewsTemplate(newsTemplate);
        }
        if (StringUtils.isNotBlank(target)) {
            target = fixContent(target, basicParam, ip, type, userId, areaCode, newsId);
            return newsTemplate.replace("{body}", target);
        }
        return newsTemplate;
    }

    /**
     * 修正内容
     * 内容中的图片添加水印和压缩参数
     * 删除video上下文中的nbsp
     * @param content html内容
     * @return 处理过后的html内容
     */
    @Override
    public String fixContent(String content, BasicParam basicParam, String ip, byte type, Long userId, String areaCode,Long newsId) {

        content = appletAdapter(content);

        //提取HTTP地址
        Document document = Jsoup.parse(content);
        document.outputSettings().prettyPrint(true).indentAmount(1);

        //增加水印
        this.addWatermark(document);

        //删除video中的空白标识
        this.deleteNbspFromVideo(document);

        //对视频进行处理(将video标签换成img标签)
        if (appVersionIntegrationService.needProcessContentVideo(basicParam)) {
            this.processVideo2Img(document,basicParam);
        }

        //处理链接
        this.processLink(document,basicParam,ip,type,userId,areaCode,newsId);

        return document.body().html().replace("\n", "");
    }

    /**
     * 处理超链接点击事件埋点
     * @param document
     * @param basicParam
     * @param ip
     * @param type
     * @param userId
     * @param areaCode
     * @param newsId
     */
    private void processLink(Document document, BasicParam basicParam, String ip, byte type, Long userId, String areaCode, Long newsId) {
        Elements elements = document.select("a");
        if (!elements.isEmpty()) {
            for (Element element : elements) {
                String text = element.text();
                String ev;
                if (ReplyTypeEnum.POST_REPLY.getCode() == type) {
                    ev = "1.50.500.523." + newsId;
                } else {
                    ev = "1.50.1700.1705." + newsId;
                }
                String url = bizConfigProperties.getBuryPointHost() + "/dot/s.gif?ch=2&e=1015&p=news&s_type=1&a=" + areaCode + "&dtype=" + basicParam.getPlatform() + "&ev=" + ev + "&ip=" + ip + "&uid=" + userId + "&v=" + basicParam.getCurVer()+"&nlink="+text;
                element.attr("onclick", "return maidian('" + text + "','" + url + "')");
            }
        }
    }

    /**
     * 处理内容中的视频
     * 视频video标签需转为img标签
     * 并在其中设置跳转协议地址
     * @param document
     */
    private void processVideo2Img(Document document,BasicParam basicParam) {
        Elements elements = document.select("video");
        if (!elements.isEmpty()) {
            for (Element element : elements) {
                String src = element.attr("src");
                if (StringUtils.isBlank(src)) {
                    continue;
                }

                String poster = element.attr("poster");
                if (StringUtils.isBlank(poster)) {
                    continue;
                }
                poster = buildImgCover(poster);
                String size = element.attr("size") == null ? "0" : element.attr("size");
                String duration = element.attr("duration") == null ? "0" : element.attr("duration");

                Element videoImgElement = document.createElement("img");
                videoImgElement.attr("src", poster);
                videoImgElement.attr("poster", poster);
                videoImgElement.attr("size", size);
                videoImgElement.attr("duration", duration);
                String url = "";
                String posterUrl = "";
                try {
                    url = URLEncoder.encode(src, "UTF-8");
                    posterUrl = URLEncoder.encode(poster, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    log.error("url编码出错：[{}]", e.getMessage());
                }

                //包裹一层容器，方便外层模板进行样式处理
                Element divElement = document.createElement("div");
                divElement.addClass("videoCopy");

                if (appVersionIntegrationService.needProcessContentVideoPoster(basicParam)) {
                    divElement.attr("onclick", "window.location.href='tt://playVideo?url=" + url + "&poster=" + posterUrl + "'");
                }else{
                    divElement.attr("onclick", "window.location.href='tt://playVideo?url=" + url+ "'");
                }
                Element waterMarkElement = document.createElement("img");
                waterMarkElement.attr("class", "videoBubble");
                waterMarkElement.attr("src", "https://m.wstong.com/icon_video_play_new.png");
                waterMarkElement.attr("alt", "");
                divElement.appendChild(waterMarkElement);
                divElement.appendChild(videoImgElement);

                element.replaceWith(divElement);
            }
        }
    }

    /**
     * 适配小程序，临时处理方案，将活动描述文案进行修改。用于和实际活动匹配
     * 文案原本存在于编辑寄语
     * @param content 内容部分
     * @return 处理结果
     */
    private String appletAdapter(String content) {
        return StringUtils.replace(content, "你将获得10朵小红花", "最多获得300朵小红花");
    }

    /**
     * 内容中的图片添加水印和压缩参数
     * @param document
     */
    private void addWatermark(Document document) {
        // 图片添加水印
        Elements elements = document.select("img");
        if (null != elements) {
            for (Element element : elements) {
                String src = element.attr("src");
                if (StringUtils.isBlank(src)) {
                    src = element.attr("data-src");
                    if (StringUtils.isBlank(src)) {
                        continue;
                    }
                }
                String url = src;
                int lastIndex = src.lastIndexOf("?");
                if (-1 != lastIndex) {
                    url = src.substring(0, lastIndex);
                }
                String extend = FileUtils.getFilextension(url);

                if ("gif".equalsIgnoreCase(extend)) {
                    url += "?x-oss-process=style/mcompress";
                } else {
                    url += "?x-oss-process=style/wcompress";
                }

                element.attr("src", url);
                element.attr("class", "preview");
            }
        }
    }

    @Override
    public String deleteNbspFromVideo(Document document) {
        Elements elements = document.select("video");
        if (!elements.isEmpty()) {
            for (Element element : elements) {
                Element parent = element.parent();
                parent.html(parent.html().replace("&nbsp;", ""));
            }
        }
        return document.body().html();
    }

    @Override
    public String buildVideoCover(String imgUrl) {
        int lastIndex = imgUrl.lastIndexOf("?");
        if (-1 != lastIndex) {
            imgUrl = imgUrl.substring(0, lastIndex);
        }
        imgUrl += "?x-oss-process=style/vthumbnail";
        return imgUrl;
    }

    @Override
    public String buildImgCover(String imgUrl) {
        int lastIndex = imgUrl.lastIndexOf("?");
        if (-1 != lastIndex) {
            imgUrl = imgUrl.substring(0, lastIndex);
        }
        imgUrl += "?x-oss-process=style/thumbnail";
        return imgUrl;
    }
}
