package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.dto.BaseForumPostDTO;
import com.bxm.localnews.news.dto.ForumPostBriefInfoDto;
import com.bxm.localnews.news.dto.ImgDTO;
import com.bxm.localnews.news.dto.InteractRankInfo;
import com.bxm.localnews.news.enums.DisplayAreaEnum;
import com.bxm.localnews.news.enums.PostTypeEnum;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.ForumPostFillContext;
import com.bxm.localnews.news.service.ForumPostFacadeService;
import com.bxm.localnews.news.service.ForumPostService;
import com.bxm.localnews.news.service.ForumPostStatisticService;
import com.bxm.localnews.news.service.ForumTopicService;
import com.bxm.localnews.news.util.OssTransCodeUtils;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Service
public class ForumPostFacadeServiceImpl implements ForumPostFacadeService {

    private final ForumPostService forumPostService;

    private final ForumPostStatisticService forumPostStatisticService;

    private final ForumPostMapper forumPostMapper;

    private final RedisSetAdapter redisSetAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final UserProperties userProperties;

    private final OssTransCodeUtils ossTransCodeUtils;

    private final ForumTopicMapper forumTopicMapper;

    private final ForumTopicService forumTopicService;

    private final NewsReplyMapper newsReplyMapper;

    public ForumPostFacadeServiceImpl(ForumPostService forumPostService, ForumPostStatisticService forumPostStatisticService,
                                      ForumPostMapper forumPostMapper, RedisSetAdapter redisSetAdapter,
                                      RedisHashMapAdapter redisHashMapAdapter, UserProperties userProperties,
                                      OssTransCodeUtils ossTransCodeUtils, ForumTopicMapper forumTopicMapper,
                                      ForumTopicService forumTopicService,NewsReplyMapper newsReplyMapper) {
        this.forumPostService = forumPostService;
        this.forumPostStatisticService = forumPostStatisticService;
        this.forumPostMapper = forumPostMapper;
        this.redisSetAdapter = redisSetAdapter;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.userProperties = userProperties;
        this.ossTransCodeUtils = ossTransCodeUtils;
        this.forumTopicMapper = forumTopicMapper;
        this.forumTopicService = forumTopicService;
        this.newsReplyMapper = newsReplyMapper;
    }

    @Override
    public ForumPostBriefInfoDto getBriefInfo(Long postId, Long userId, String areaCode, String ip) {
        ForumPostVo forumPostVo = forumPostService.getForumPostDetail(postId, userId, null, new BasicParam(), areaCode, ip);
        ForumPostBriefInfoDto briefInfoDto = new ForumPostBriefInfoDto();

        if (null != forumPostVo) {
            PostImgVo imgInfo = forumPostVo.getShareImg();
            if (null != imgInfo) {
                briefInfoDto.setConvertImgUrl(imgInfo.getImgUrl());
            }
            briefInfoDto.setHasVote(forumPostVo.getVoteDetailDTO() != null);
            briefInfoDto.setId(postId);
            briefInfoDto.setTitle(getTitle(forumPostVo));
        }

        return briefInfoDto;
    }

    @Override
    public ForumPostBriefInfoDto getBriefInfo(Long postId) {
        ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(postId);

        return ForumPostBriefInfoDto.builder()
                .id(forumPostVo.getId())
                .title(getTitle(forumPostVo))
                .hasVote(forumPostVo.getVoteDetailDTO() != null)
                .build();
    }

    private String getTitle(ForumPostVo forumPostVo) {
        String title = forumPostVo.getTitle();
        if (StringUtils.isBlank(title) || PostTypeEnum.NOTE.getCode() == forumPostVo.getPostType()) {
            title = StringUtils.substring(forumPostVo.getTextField(), 0, 50);
        }
        return title;
    }

    @Override
    public Integer getPublishPostNum(Long userId) {
        return forumPostMapper.selectPublishPostNumByUserId(userId);
    }

    @Override
    public List<ForumPostClickCountVo> getRecentPosts(Integer clickCountLimit) {
        return forumPostMapper.getRecentPosts(clickCountLimit, DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -1));
    }

    @Override
    public List<ForumPostClickCountVo> getRecentRecommendPosts(Integer clickCountLimit) {
        return forumPostMapper.getRecentRecommendPosts(clickCountLimit, DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, -120));
    }

    @Override
    public void batchAddClick(List<ForumPostClickCountVo> forumPostClickList) {
        if (CollectionUtils.isNotEmpty(forumPostClickList)) {
            MybatisBatchBuilder.create(ForumPostMapper.class, forumPostClickList).run(ForumPostMapper::batchAddClick);
        }
    }

    @Async
    @Override
    public void transcodePostVideo(Long id, List<PostImgVo> postImgList) {
        if (CollectionUtils.isEmpty(postImgList)) {
            ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(id);
            if (forumPostVo != null) {
                postImgList = forumPostVo.getPostImgList();
            }
        }

        if (CollectionUtils.isNotEmpty(postImgList)) {
            postImgList.forEach(e -> {
                if ("VIDEO".equals(e.getType())) {
                    ossTransCodeUtils.transCodeByOriginUrl(e.getVideoUrl(), id);
                }
            });
        }
    }

    @Override
    public void batchUpdatePostVideo(List<TranscodeMap> transcodeMapList) {
        if (CollectionUtils.isNotEmpty(transcodeMapList)) {
            MybatisBatchBuilder.create(ForumPostMapper.class, transcodeMapList).run(ForumPostMapper::batchUpdatePostVideo);
        }
    }

    @Override
    public BaseForumPostDTO getForumPostByUserId(Long userId, Long targetUserId) {
        KeyGenerator recentKey = RedisConfig.RECENT_DEVELOPMENTS.copy().appendKey(String.valueOf(userId));
        //每次取目标用户的第一条帖子
        ForumPostVo forumPostVo = forumPostMapper.selectPostByUserIdAndWeek(targetUserId);
        BaseForumPostDTO result = null;
        //若用户的最新的第一条帖子已被获取,则返回空
        boolean flag = Objects.nonNull(forumPostVo) && !redisSetAdapter.exists(recentKey, forumPostVo.getId());
        if (flag) {
            List<PostImgVo> coverList = forumPostStatisticService.getCoverFromPost(
                    forumPostVo.getCoverList(), forumPostVo.getPostImgList(), forumPostVo.getCoverSelect());
            result = BaseForumPostDTO.builder()
                    .id(forumPostVo.getId())
                    //分享图片为空时取默认分享图
                    .imgUrl(CollectionUtils.isEmpty(coverList) ? null : coverList.get(0).getImgUrl())
                    .textField(forumPostVo.getTextField())
                    .title(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()))
                    .publishTime(forumPostVo.getPublishTime())
                    .build();
            redisSetAdapter.add(recentKey, forumPostVo.getId());
            //缓存7天后过期
            redisSetAdapter.expire(recentKey, DateUtils.WEEK_MILLISECOND / 1000);
        }
        return result;
    }

    @Override
    public List<UserImgVo> batchGetUserImg(List<Long> userIdList, Long currentUserId) {
        List<UserImgVo> userImgVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userImgVoList;
        }

        boolean hasCurrent = false;
        if (currentUserId != null && userIdList.contains(currentUserId)) {
            userIdList.remove(currentUserId);
            hasCurrent = true;
        }

        if (CollectionUtils.isNotEmpty(userIdList)) {
            //先取缓存
            Set<String> userIds = userIdList.stream().map(Object::toString).collect(Collectors.toSet());
            List<String> userForumImgCache = redisHashMapAdapter.multiGet(RedisConfig.USER_FORUM_IMG, userIds, String.class);

            //处理缓存数据
            if (CollectionUtils.isNotEmpty(userForumImgCache)) {
                userForumImgCache.forEach(userCache -> {
                    UserImgVo userImgVo = JSON.parseObject(userCache, UserImgVo.class);
                    if (null != userImgVo) {
                        userIdList.remove(userImgVo.getUserId());
                        userImgVoList.add(userImgVo);
                    }
                });
            }

            //若缓存中无法取到所有用户图片信息
            if (CollectionUtils.isNotEmpty(userIdList)) {
                userImgVoList.addAll(batchLoadUserImgVoToRedis(userIdList));
            }
        }

        //当前用户需返回仅自己可见内容
        if (hasCurrent) {
            List<ForumPostVo> forumPostVoList = forumPostMapper.batchGetUserPosts(
                    Collections.singletonList(currentUserId), userProperties.getAdvertTopicId(), 1);
            UserImgVo userImgVo = convertPostToUserImg(forumPostVoList, currentUserId);
            if (userImgVo != null) {
                userImgVoList.add(userImgVo);
            }
        }

        return userImgVoList;
    }

    @Override
    public void cacheUserPostImg(Date startTime) {
        //获取有更新的帖子用户
        List<Long> userIdList = forumPostMapper.getUpdatedUser(startTime);
        //缓存数据
        batchLoadUserImgVoToRedis(userIdList);
    }

    @Override
    public TopicFacadeVO getTopicById(Long id) {
        TopicVo topicVo = forumTopicService.getTopicDetail(id ,null, null);

        if (Objects.isNull(topicVo)) {
            return null;
        }
        return convert(topicVo);
    }

    private TopicFacadeVO convert(TopicVo topicVo) {
        TopicFacadeVO facadeVO = new TopicFacadeVO();
        facadeVO.setId(topicVo.getId());
        facadeVO.setTitle(topicVo.getTitle());
        facadeVO.setDescp(topicVo.getDescp());
        facadeVO.setImg(topicVo.getImg());
        facadeVO.setParticipantsNum(topicVo.getParticipantsNum());
        facadeVO.setAppletShareUrl(topicVo.getAppletShareUrl());
        facadeVO.setShareUrl(topicVo.getShareUrl());
        facadeVO.setLeadPostContent(topicVo.getLeadPostContent());
        facadeVO.setRecommendContent(topicVo.getRecommendContent());
        facadeVO.setJoinCode(topicVo.getJoinCode());
        facadeVO.setJoinHeadImgList(topicVo.getJoinHeadImgList());
        facadeVO.setContent(topicVo.getContent());
        return facadeVO;
    }

    private List<UserImgVo> batchLoadUserImgVoToRedis(List<Long> userIdList) {
        List<UserImgVo> userImgVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userImgVoList;
        }

        //获取用户帖子
        List<ForumPostVo> forumPostVoList = forumPostMapper.batchGetUserPosts(userIdList, userProperties.getAdvertTopicId(), 0);
        if (CollectionUtils.isEmpty(forumPostVoList)) {
            return userImgVoList;
        }

        //映射用户帖子图片
        Map<Long, List<ForumPostVo>> userForumPostMap = forumPostVoList.stream().collect(Collectors.groupingBy(ForumPostVo::getUserId));
        Map<String, String> userImgVoMap = new HashMap<>();
        userIdList.forEach(userId -> {
            UserImgVo userImgVo = convertPostToUserImg(userForumPostMap.get(userId), userId);
            //即使为空，也要覆盖缓存
            userImgVoMap.put(userId.toString(), JSON.toJSONString(userImgVo));
            if (null != userImgVo) {
                userImgVoList.add(userImgVo);
            }
        });

        //存入缓存
        if (userImgVoMap.size() > 0) {
            redisHashMapAdapter.putAll(RedisConfig.USER_FORUM_IMG, userImgVoMap);
        }

        return userImgVoList;
    }

    private UserImgVo convertPostToUserImg(List<ForumPostVo> forumPostVos, Long userId) {
        if (CollectionUtils.isEmpty(forumPostVos)) {
            return null;
        }

        // 只获取4个
        int size = 0;
        List<ImgDTO> imgDtoList = new ArrayList<>();
        for (ForumPostVo forumPostVo : forumPostVos) {
            List<PostImgVo> postImgVoList = forumPostVo.getPostImgList();
            if (CollectionUtils.isNotEmpty(postImgVoList)) {
                for (PostImgVo postImgVo : postImgVoList) {
                    if ("IMG".equals(postImgVo.getType())) {
                        ImgDTO imgDto = new ImgDTO();
                        imgDto.setId(forumPostVo.getId());
                        imgDto.setCategory("POST");
                        imgDto.setType(postImgVo.getType());
                        imgDto.setImgUrl(postImgVo.getImgUrl());
                        imgDtoList.add(imgDto);
                        size++;
                        if (size >= 4) {
                            break;
                        }
                    }
                }
            }
            if (size >= 4) {
                break;
            }
        }

        UserImgVo userImgVo = new UserImgVo();
        userImgVo.setUserId(userId);
        userImgVo.setImgDtoList(imgDtoList);
        return userImgVo;
    }
    @Override
    public ForumPostVo getForumPostById(Long id) {
        return forumPostMapper.selectBasicPostById(id);
    }

    @Override
    public List<ForumRankVo> forumRank(String areaCode,List<Long> userIds) {
        //总榜单
        List<ForumRankVo> list = forumPostMapper.getForumRankInfo(areaCode,userIds);

        //填充榜单内用户评论
        list.forEach(item-> item.setPostTitleInfo(this.fill(item.getUserId(),areaCode)));

        return list.size()>0 ? list : new ArrayList<>();
    }

    private List<ForumPostTitleInfoVo> fill(Long userId,String areaCode){
        List<ForumPostVo> list = forumPostMapper.selectPostTitletByUserId(userId);
        ForumPostFillContext context = ForumPostFillContext.builder()
                .data(list)
                .fillTitle(true)
                .build();

        forumPostService.fillExtInfo(context);
        return list.stream().map(item->{
            ForumPostTitleInfoVo forumPostTitleInfoVo = new ForumPostTitleInfoVo();
            forumPostTitleInfoVo.setId(item.getId());
            forumPostTitleInfoVo.setTitle(item.getTitle());
            return forumPostTitleInfoVo;
        }).collect(Collectors.toList());
    }

    @Override
    public ForumPostRankForUserVo forumRankForUser(Long userId) {
        return forumPostMapper.getForumRankInfoForUser(userId);
    }

    @Override
    public List<InteractRankInfo> getInteractRankInfo(String areaCode, int limitNum, List<Long> userIds) {
        return newsReplyMapper.selectInteractRankInfo(areaCode,limitNum,userIds);
    }

    @Override
    public Integer replyNumForUser(Long userId) {
        return newsReplyMapper.getReplyNumById(userId);
    }

}
