package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumTopicMapper;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.dto.PostTopicDTO;
import com.bxm.localnews.news.dto.RelationDTO;
import com.bxm.localnews.news.enums.UrlCategoryEnum;
import com.bxm.localnews.news.enums.UrlPositionEnum;
import com.bxm.localnews.news.enums.UrlTypeEnum;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.service.ForumTopicService;
import com.bxm.localnews.news.vo.ForumPostVo;
import com.bxm.localnews.news.vo.PostTopicVO;
import com.bxm.localnews.news.vo.TopicVo;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.RandomUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class ForumTopicServiceImpl extends BaseService implements ForumTopicService {
    private static final String AREA_PLACE_HOLDER = "{areaname}";
    private static final String TOPIC_PLACE_HOLDER = "{topicname}";

    @Resource
    private ForumTopicMapper forumTopicMapper;

    @Resource
    private NewsReplyMapper newsReplyMapper;

    @Resource
    private IUrlFactory iUrlFactory;

    @Resource
    private ForumProperties forumProperties;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Override
    public List<PostTopicVO> listTopicByPostIds(List<Long> postIds, Long userId) {
        List<PostTopicVO> postTopicVoList = new ArrayList<>();

        List<RelationDTO> relationList = forumTopicMapper.getPostTopicRelation(postIds);
        if (CollectionUtils.isNotEmpty(relationList)) {
            for (RelationDTO relation : relationList) {
                PostTopicVO postTopic = new PostTopicVO();
                postTopic.setPostId(relation.getAid());

                if (null != relation.getBid()) {
                    TopicVo topicVo = getTopicDetail(relation.getBid(), userId, null);
                    if (null != topicVo) {
                        BeanUtils.copyProperties(topicVo, postTopic);
                    }
                }

                postTopicVoList.add(postTopic);
            }
        }

        return postTopicVoList;
    }

    @Override
    public List<TopicVo> listTopicByPostId(Long postId, Long userId) {
        List<TopicVo> topicVoList = new ArrayList<>();
        List<RelationDTO> relationList = forumTopicMapper.getPostTopicRelation(Collections.singletonList(postId));

        if (CollectionUtils.isNotEmpty(relationList)) {
            for (RelationDTO relation : relationList) {
                if (null != relation.getBid()) {
                    TopicVo topicVo = getTopicDetail(relation.getBid(), userId, null);
                    if (null != topicVo) {
                        topicVoList.add(topicVo);
                    }
                }
            }
        }

        return topicVoList;
    }

    @Override
    public void fillTopicShareInfo(TopicVo topicVo, Long userId) {
        if (topicVo != null) {
            topicVo.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.TOPIC, topicVo.getId(), userId));
            topicVo.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.FORUM, UrlTypeEnum.TOPIC, topicVo.getId(), userId));
        }
    }

    @Override
    public List<TopicVo> listTopic(String areaCode, Integer size, Long userId, Integer postPage) {
        List<TopicVo> topicVoList = getTopicList(areaCode);

        if (CollectionUtils.isEmpty(topicVoList)) {
            return topicVoList;
        }

        // 发帖页面屏蔽小纸条话题
        if (postPage != null && 1 == postPage) {
            Long noteTopicId = forumProperties.getNoteTopicId();
            topicVoList = topicVoList.stream().filter(e -> !noteTopicId.equals(e.getId())).collect(Collectors.toList());
        }

        if (CollectionUtils.isEmpty(topicVoList)) {
            return topicVoList;
        }

        //如果数量传了2，则进行随机取
        if (size != null && size == 2) {
            topicVoList = generateRandomTopic(topicVoList);
        }

        topicVoList.forEach(e -> fillTopicShareInfo(e, userId));
        return topicVoList;
    }

    @Override
    public List<TopicVo> getTopicList(String areaCode) {
        String areaCodeKey = areaCode;
        //若不传areaCode则取全国
        if (StringUtils.isBlank(areaCode)) {
            areaCodeKey = "0";
        }

        List<TopicVo> topicVoList = redisHashMapAdapter.get(RedisConfig.TOPIC, areaCodeKey, new TypeReference<List<TopicVo>>() {
        });
        if (CollectionUtils.isNotEmpty(topicVoList)) {
            return topicVoList;
        }

        topicVoList = forumTopicMapper.getTopicList(areaCode, 1);
        if (CollectionUtils.isNotEmpty(topicVoList)) {
            //计算参与者头像
            calcJoinHeadImgList(topicVoList);

            //推荐语地区占位符替换
            String areaName = "全国";
            if (StringUtils.isNotBlank(areaCode)) {
                LocationDTO locationDTO = locationIntegrationService.getLocationByGeocode(areaCode);
                if (locationDTO != null) {
                    areaName = locationDTO.getName();
                }
            }
            for (TopicVo topicVo : topicVoList) {
                String recommendContent = topicVo.getRecommendContent();
                if (StringUtils.isBlank(recommendContent)) {
                    recommendContent = forumProperties.getDefaultRecommendContent();
                }

                recommendContent = recommendContent.replace(AREA_PLACE_HOLDER, areaName);
                topicVo.setRecommendContent(recommendContent.replace(TOPIC_PLACE_HOLDER, topicVo.getTitle()));
            }

            redisHashMapAdapter.put(RedisConfig.TOPIC, areaCodeKey, topicVoList);
        }

        return topicVoList;
    }

    @Override
    public TopicVo getTopicDetail(Long id, Long userId, String areaCode) {
        TopicVo topicVo = getTopicById(id, areaCode);
        fillTopicShareInfo(topicVo, userId);
        return topicVo;
    }

    @Override
    public void updateTopic(ForumPostVo forumPostVo) {
        logger.warn("帖子更新话题，id:[{}],话题列表:[{}]", forumPostVo.getId(), JSONObject.toJSON(forumPostVo.getTopicList()));
        if (null == forumPostVo.getId() || CollectionUtils.isEmpty(forumPostVo.getTopicIdList())) {
            return;
        }

        this.forumTopicMapper.batchDeleteTopic(forumPostVo.getId());

        if (CollectionUtils.isNotEmpty(forumPostVo.getTopicIdList())) {
            List<PostTopicDTO> list = new ArrayList<>();
            forumPostVo.getTopicIdList().forEach(topicId -> list.add(getPostTopic(forumPostVo.getId(), topicId)));
            this.forumTopicMapper.batchInsertTopic(list);
        }
    }

    private PostTopicDTO getPostTopic(Long postId, Long topicId) {
        PostTopicDTO postTopicDTO = new PostTopicDTO();
        postTopicDTO.setId(nextSequence());
        postTopicDTO.setPostId(postId);
        postTopicDTO.setTopicId(topicId);
        return postTopicDTO;
    }

    private TopicVo getTopicById(Long id, String areaCode) {
        List<TopicVo> topicVoList = getTopicList(areaCode);

        if (CollectionUtils.isNotEmpty(topicVoList)) {
            for (TopicVo topicVo : topicVoList) {
                if (topicVo.getId().equals(id)) {
                    return topicVo;
                }
            }
        }

        return forumTopicMapper.selectTopicById(id, areaCode);
    }

    private void calcJoinHeadImgList(List<TopicVo> topicVoList) {
        for (TopicVo topicVo : topicVoList) {
            //先获取话题下发帖人头像
            List<String> postHeadImgList = forumTopicMapper.getPostHeadImgList(topicVo.getId());
            //获取不到发帖人头像，则说明该话题无人参与，直接跳过
            if (CollectionUtils.isEmpty(postHeadImgList)) {
                continue;
            }
            Set<String> joinHeadImgSet = new HashSet<>(postHeadImgList);

            if (joinHeadImgSet.size() < 5) {
                //数量不够，则再获取话题下评论人头像，由于评论分表，所以需要挨个查
                List<Long> postIdList = forumTopicMapper.getPostIdListByTopicId(topicVo.getId());
                boolean isNotEnough = false;

                if (CollectionUtils.isNotEmpty(postIdList)) {
                    for (Long postId : postIdList) {
                        List<String> replyHeadImgList = newsReplyMapper.getReplyHeadImgList(postId);
                        if (CollectionUtils.isNotEmpty(replyHeadImgList)) {
                            joinHeadImgSet.addAll(replyHeadImgList);

                            if (joinHeadImgSet.size() >= 5) {
                                break;
                            }
                        }
                    }

                    if (joinHeadImgSet.size() < 5) {
                        isNotEnough = true;
                    }
                }

                if (isNotEnough) {
                    //数量仍然不够，则再取话题下点赞人
                    List<String> likeHeadImgList = forumTopicMapper.getLikeHeadImgList(topicVo.getId());
                    if (CollectionUtils.isNotEmpty(likeHeadImgList)) {
                        joinHeadImgSet.addAll(likeHeadImgList);
                    }
                }
            }

            topicVo.setJoinHeadImgList(joinHeadImgSet.stream().limit(5).collect(Collectors.toList()));
        }

    }

    /**
     * 随机取两条话题
     * 扯淡的需求
     *
     * @param originList 原始话题列表
     */
    private List<TopicVo> generateRandomTopic(List<TopicVo> originList) {
        if (originList.size() <= 2) {
            return originList;
        }

        List<TopicVo> randomTopicVoList = originList.stream().limit(6).collect(Collectors.toList());

        logger.debug("随机取两条话题:[{}]", originList.size());
        if (CollectionUtils.isNotEmpty(originList)) {
            int random = RandomUtils.nextInt(0, originList.size());
            logger.debug("随机取数1:[{}]", random);
            randomTopicVoList.add(originList.get(random));
            originList.remove(random);
            if (CollectionUtils.isNotEmpty(originList)) {
                int random2 = RandomUtils.nextInt(0, originList.size());
                logger.debug("随机取数2:[{}]", random2);
                randomTopicVoList.add(originList.get(random2));
            }
        }

        logger.debug("获得话题:[{}]", JSONObject.toJSONString(randomTopicVoList));
        return randomTopicVoList;
    }

}
