package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.news.Relation.MerchantRelationService;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.convert.impl.NewsReplyConver;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.*;
import com.bxm.localnews.news.enums.NewsConstant;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.enums.UrlCategoryEnum;
import com.bxm.localnews.news.enums.UrlTypeEnum;
import com.bxm.localnews.news.event.CommentActionEvent;
import com.bxm.localnews.news.event.UserActionEvent;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.NewsReplyAddParam;
import com.bxm.localnews.news.param.NewsReplyLikeParam;
import com.bxm.localnews.news.param.NewsReplyParam;
import com.bxm.localnews.news.param.UserReplyParam;
import com.bxm.localnews.news.service.*;
import com.bxm.localnews.news.thread.PostReplyThread;
import com.bxm.localnews.news.vo.*;
import com.bxm.localnews.vo.VirtualUserInfo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.CharUtil;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.eventbus.EventBus;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static java.util.Comparator.comparing;

/**
 * Created by Administrator on 2018/2/23 0023.
 */
@Service
@Transactional(rollbackFor = Exception.class)
public class NewNewsReplyServiceImpl extends BaseService implements NewNewsReplyService {

    @Resource
    private NewsReplyMapper newsReplyMapper;

    @Resource
    private NewsReplyLikeMapper newsReplyLikeMapper;

    @Resource
    private RedisListAdapter redisListAdapter;

    @Resource
    private UserReplyMapper userReplyMapper;

    @Resource
    private VideoMapper videoMapper;

    @Resource
    private NewsMapper newsMapper;

    @Resource
    private NewsCollectMapper newsCollectMapper;

    @Resource
    private VideoLikeMapper videoLikeMapper;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private IUrlFactory iUrlFactory;

    @Autowired
    private VideoRecommendService videoRecommendService;

    @Autowired
    private NewsTotalMapper newsTotalMapper;

    @Autowired
    private ForumPostMapper forumPostMapper;

    @Autowired
    private BizLogIntegrationService bizLogIntegrationService;

    @Autowired
    private NewsSearchIntegrationService newsSearchIntegrationService;

    @Autowired
    private MessageService messageService;

    @Autowired
    private MissionIntegrationService missionIntegrationService;

    @Autowired
    private NewsReplyConver newsReplyConver;

    @Autowired
    private LocationIntegrationService locationIntegrationService;

    @Autowired
    private AppVersionIntegrationService appVersionIntegrationService;

    @Autowired
    private SensitiveWordService sensitiveWordService;

    @Autowired
    private EventBus userActionEventBus;

    @Autowired
    private RedisSetAdapter redisSetAdapter;

    @Autowired
    private AsyncTaskExecutor asyncTaskExecutor;

    @Autowired
    private MerchantRelationService merchantRelationService;

    @Override
    public PageWarper<NewsReplyDTO> queryReplyList(NewsReplyParam newsReplyParam) {
        //分页查找level为0的评论
        PageWarper<NewsReplyDTO> newsReplyDTOPageWarper = new PageWarper<>(this.newsReplyMapper.queryByPageSize(newsReplyParam));
        List<NewsReplyDTO> list = newsReplyDTOPageWarper.getList();
        if (CollectionUtils.isNotEmpty(list)) {
            List<Long> rootIds = list.stream().map(NewsReplyDTO::getId).collect(Collectors.toList());
            //查找所有的根评论的子评论
            List<NewsReplyMirrorDTO> allReplies = this.newsReplyMapper.selectSubReplyLists(rootIds, newsReplyParam.getNewsId());
            //若是帖子，则需要加上评论人是否楼主的判断
            if (ReplyTypeEnum.POST_REPLY.getCode() == list.get(0).getType()) {
                Long userId = forumPostMapper.getUserIdByPostId(newsReplyParam.getNewsId());
                if (null != userId) {
                    list.forEach(e -> {
                        e.setUserIsAuthor(userId.equals(e.getUserId()));
                        e.setParentUserIsAuthor(userId.equals(e.getParentUserId()));
                    });
                    allReplies.forEach(e -> {
                        e.setUserIsAuthor(userId.equals(e.getUserId()));
                        e.setParentUserIsAuthor(userId.equals(e.getParentUserId()));
                    });
                }
            }
            //查询该用户在该条新闻下的所有评论的额点赞记录
            List<Long> replyLikes = newsReplyMapper.selectReplyLike(newsReplyParam.getNewsId(), newsReplyParam.getUserId(), null);

            if (CollectionUtils.isNotEmpty(list)) {
                for (NewsReplyDTO vo : newsReplyDTOPageWarper.getList()) {
                    //兼容老版本客户端
                    if (vo.getDeleteFlag() == 2) {
                        vo.setDeleteFlag((byte) 1);
                        vo.setReplyContent("该评论已删除!");
                    }
                    //用户是否点赞
                    vo.setIsLike(checkIsLike(replyLikes, vo.getId()));
                    vo.setReplyTime(DateUtils.timeConvertString(vo.getAddTime()));
                    //查询该评论下的子评论
                    List<NewsReplyMirrorDTO> replies = allReplies.stream()
                            .filter(e -> e.getRootId().equals(vo.getId()))
                            .sorted(comparing(NewsReplyMirrorDTO::getAddTime))
                            .collect((Collectors.toList()));
                    if (CollectionUtils.isNotEmpty(replies)) {
                        replies.forEach(replay -> {
                            //用户是否点赞
                            dealSubMirrorDTO(replay, newsReplyParam.getUserId(), replyLikes);
                        });
                        vo.setList(replies);
                    }
                }
            }
        }

        //增加vip标志
        newsReplyDTOPageWarper.getList().forEach(item -> {
            item.setIsVip(merchantRelationService.isVip(item.getUserId()));
        });

        return newsReplyDTOPageWarper;
    }

    private void dealSubMirrorDTO(NewsReplyMirrorDTO replay, Long userId, List<Long> replyLikes) {
        replay.setIsLike(checkIsLike(replyLikes, replay.getId()));
        //隐藏父级回复，详细需求问强哥
        if (replay.getLevel().equals(NewsConstant.REPLY_LEVEL_1)) {
            replay.setParentHeadImg(null);
            replay.setParentUserId(null);
            replay.setParentUserNickname(null);
        }
        replay.setReplyTime(DateUtils.timeConvertString(replay.getAddTime()));

    }

    private int checkIsLike(List<Long> replyLikes, Long replyId) {
        return replyLikes.contains(replyId) ? 1 : 0;
    }

    /**
     * 判断当前用户是否对此回复点赞
     *
     * @param userId
     * @param replyId
     * @return
     */
    private boolean isLike(Long userId, Long replyId) {
        NewsReplayLike newsReplayLike = newsReplyLikeMapper.selectByModel(new NewsReplayLike(replyId, userId));
        return null != newsReplayLike;
    }

    @Override
    public Json<ReplyDTO> doReply(NewsReplyAddParam newsReplyAddParam, BasicParam basicParam) {
        logger.debug("[doReply]发表评论，参数:[{}]", toJSONString(newsReplyAddParam));
        if (PlatformEnum.APPLET.getCode() == basicParam.getPlatform()) {
            if (appVersionIntegrationService.hasSensitiveWord(newsReplyAddParam.getReplyContent())) {
                return ResultUtil.genFailedResult("您发表的评论中有敏感内容，请修改后再发");
            }
        } else {
            if (appVersionIntegrationService.needValidSensitive(basicParam)
                    && sensitiveWordService.contains(newsReplyAddParam.getReplyContent())) {
                return ResultUtil.genFailedResult("您发表的评论中有敏感内容，请修改后再发");
            }
        }

        NewsReply newsReply = newsReplyConver.convert(newsReplyAddParam);
        this.newsReplyMapper.insertSelective(newsReply);
        //若是帖子，则需要加上评论人是否楼主的判断
        if (ReplyTypeEnum.POST_REPLY.getCode() == newsReply.getType()) {
            Long userId = forumPostMapper.getUserIdByPostId(newsReply.getNewsId());
            if (null != userId) {
                newsReply.setUserIsAuthor(userId.equals(newsReply.getUserId()));
                newsReply.setParentUserIsAuthor(userId.equals(newsReply.getParentUserId()));
            }
        }
        //保存至用户评论中
        UserReply userReply = new UserReply();
        BeanUtils.copyProperties(newsReply, userReply);
        logger.debug("用户userId=[{}]发表评论,userReply参数:[{}]", userReply.getUserId(), toJSONString(userReply));

        this.userReplyMapper.insertSelective(userReply);
        //兼容评论时间
        newsReply.setReplyTime(DateUtils.timeConvertString(newsReply.getAddTime()));
        //异步执行
        NewsReply param = new NewsReply();
        BeanUtils.copyProperties(newsReply, param);
        //计算帖子,新闻,小视频的评论数,及帖子的评论概览
        this.updateComment(newsReply);

        //增加用户表评论数统计
        userIntegrationService.addPostReplyNum(newsReplyAddParam.getUserId(), 3);

        //异步调用推送
        NewNewsReplyService newNewsReplyService = SpringContextHolder.getBean(this.getClass());
        newNewsReplyService.processWhenReply(param, newsReplyAddParam, basicParam.getPlatform(), newsReplyAddParam.getAreaCode());
        //隐藏父级评论
        if (newsReply.getLevel().equals(NewsConstant.REPLY_LEVEL_1)
                && basicParam.getPlatform() != PlatformEnum.APPLET.getCode()) {
            newsReply.setParentUserId(null);
            newsReply.setParentUserNickname(null);
        }
        ReplyDTO result = new ReplyDTO();
        BeanUtils.copyProperties(newsReply, result);
        //评论新闻并且超过3字
        if (ReplyTypeEnum.VIDEO_REPLY.getCode() != newsReply.getType() && CharUtil.isCharSatisfied(newsReply.getReplyContent(), 3)) {
            //调用完成任务接口
            NewsCompleTaskDTO taskDTO = missionIntegrationService.compleTask(newsReply.getUserId(),
                    "TASK_COMMENT_NEWS",
                    newsReply.getId().toString(),
                    "评论奖励");
            logger.info("user_id ={}完成评论任务,奖励{}多红花;", newsReply.getUserId(), taskDTO.getGoldNum());
            result.setGoldNum(Objects.isNull(taskDTO.getGoldNum()) ? 0 : taskDTO.getGoldNum());
        }
        // 将评论数据存入缓存用来发送定时回复提醒任务
        if (ReplyTypeEnum.POST_REPLY.getCode() == newsReplyAddParam.getType()) {
            this.saveForumpostReplyRecord(newsReplyAddParam.getUserId(), newsReplyAddParam.getNewsId(), new Date());
        }
        return Json.build(result);
    }

    /**
     * 当评论类型是帖子时,保存帖子的评论记录到缓存
     *
     * @param userId
     * @param postId
     */
    @Override
    public void saveForumpostReplyRecord(Long userId, Long postId, Date sendTime) {
        Date expireTime = this.getExpireTime();
        KeyGenerator postKey = RedisConfig.FORUM_REPLY_POSTID.copy();
        redisSetAdapter.add(postKey, postId);
        redisListAdapter.expire(postKey, expireTime);
        ForumReplyInfo info = ForumReplyInfo.builder()
                .userId(userId)
                .replyTime(sendTime)
                .build();
        KeyGenerator infoKey = RedisConfig.FORUM_REPLY_INFO.copy().appendKey(postId);
        redisListAdapter.leftPush(infoKey, info);
        redisListAdapter.expire(infoKey, expireTime);
    }

    /**
     * 获取缓存统计帖子评论的缓存过期时间的过期时间
     *
     * @return
     */
    private Date getExpireTime() {
        //当前时间
        Calendar current = Calendar.getInstance();
        int hour = current.get(Calendar.HOUR_OF_DAY);
        //过期时间
        Calendar expired = Calendar.getInstance();
        expired.set(Calendar.MINUTE, 2);
        expired.set(Calendar.SECOND, 0);
        if (hour < 9) {
            expired.set(Calendar.HOUR_OF_DAY, 9);
        } else if (hour < 12) {
            expired.set(Calendar.HOUR_OF_DAY, 12);
        } else if (hour < 18) {
            expired.set(Calendar.HOUR_OF_DAY, 18);
        } else if (hour < 21) {
            expired.set(Calendar.HOUR_OF_DAY, 21);
        } else {
            expired.add(Calendar.DAY_OF_MONTH, 1);
            expired.set(Calendar.HOUR_OF_DAY, 9);
        }
        return expired.getTime();
    }

    @Async
    @Override
    public void processWhenReply(NewsReply newsReply, NewsReplyAddParam newsReplyAddParam, Integer platform, String areaCode) {
        //1.新闻不是根级评论时将根级评论互动值+3
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        if (Objects.nonNull(newsReply.getRootId()) && newsReply.getRootId() != 0) {
            NewsReply rootReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getRootId(), newsReply.getNewsId());
            Integer interactiveCount = rootReply.getInteractiveCount();
            interactiveCount = (Objects.isNull(interactiveCount) || interactiveCount == 0) ? 3 : interactiveCount + 3;
            rootReply.setInteractiveCount(interactiveCount);
            newsReplyMapper.updateByPrimaryKeySelective(rootReply);
            UserReply userRootReply = new UserReply();
            BeanUtils.copyProperties(rootReply, userRootReply);
            userReplyMapper.updateinteractiveCountById(userRootReply);
        }
        //2.推送消息，自己给自己评论不推送信息
        if (Objects.nonNull(newsReplyAddParam.getParentUserId()) &&
                !(newsReplyAddParam.getParentUserId().equals(newsReply.getUserId()))) {
            logger.debug("封装的消息格式为：[{}]", toJSONString(newsReply));
            if (newsReply.getType() == ReplyTypeEnum.NEWS_REPLY.getCode() || newsReply.getType() == ReplyTypeEnum.VIDEO_REPLY.getCode()) {
                NewsService newsService = SpringContextHolder.getBean(NewsService.class);
                News news = newsSearchIntegrationService.multipleGet(new Long[]{newsReply.getNewsId()}).get(0);
                newsService.judgeEnablePlaceholder(news, location);
                newsService.exchangeDetailNews(news);
                messageService.pushNewReplyMessage(newsReply, PushMessageEnum.NEWS_REPLY, null, news);
            } else if (newsReply.getType() == ReplyTypeEnum.POST_REPLY.getCode()) {
                ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(newsReply.getNewsId());
                SpringContextHolder.getBean(ForumPostStatisticService.class).judgeEnablePlaceholder(forumPostVo, location);
                SpringContextHolder.getBean(ForumPostStatisticService.class).exchangeDetailPost(forumPostVo);
                messageService.pushNewReplyMessage(newsReply, PushMessageEnum.POST_REPLY, forumPostVo, null);
            }
        }
        //3.回复帖子的推送
        if (null == newsReplyAddParam.getParentUserId() && newsReply.getType() == ReplyTypeEnum.POST_REPLY.getCode()) {
            ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(newsReply.getNewsId());
            SpringContextHolder.getBean(ForumPostStatisticService.class).judgeEnablePlaceholder(forumPostVo, location);
            SpringContextHolder.getBean(ForumPostStatisticService.class).exchangeDetailPost(forumPostVo);
            messageService.pushPostReplyMessage(newsReply, forumPostVo);
        }
        //4.记录事件埋点
        if (StringUtils.isNotBlank(newsReplyAddParam.getAreaCode())) {
            bizLogIntegrationService.replySuccessed(newsReplyAddParam.getUserId(), newsReplyAddParam.getNewsId(),
                    new Integer(newsReplyAddParam.getType()), newsReplyAddParam.getAreaCode(), platform);
        }

        //5.上报评论信息到用户推荐数据源服务
        UserActionEvent userActionEvent = CommentActionEvent.of()
                .setLastCommentTime(new Date())
                .setTargetId(newsReplyAddParam.getNewsId())
                .setUserId(newsReplyAddParam.getUserId());

        userActionEventBus.post(userActionEvent);
    }

    @Override
    public void updateComment(NewsReply newsReply) {
        int count = newsReplyMapper.selectCountByNewsId(newsReply.getNewsId());
        if (ReplyTypeEnum.NEWS_REPLY.getCode() == newsReply.getType()) {
            newsTotalMapper.updateNewsComments(newsReply.getNewsId(), count);
        } else if (ReplyTypeEnum.VIDEO_REPLY.getCode() == newsReply.getType()) {
            Video video = new Video();
            video.setId(newsReply.getNewsId());
            video.setCommentCount((long) count);
            videoMapper.updateByPrimaryKeySelective(video);
        } else if (ReplyTypeEnum.POST_REPLY.getCode() == newsReply.getType()) {
            //维护帖子评论概览及帖子的评论数
            ForumPostStatistic forumPostStatistic = ForumPostStatistic.buildComments(newsReply.getNewsId(), count);
            List<NewsReply> replyList = newsReplyMapper.selectPostReplyByNewsIdOrderByAddTime(newsReply.getNewsId());
            if (CollectionUtils.isNotEmpty(replyList)) {
                forumPostStatistic.setReplyInfo(toJSONString(replyList));
            }
            forumPostMapper.updateReplyInfoAndCommentByPostId(forumPostStatistic);
        }
    }

    @Override
    public PageWarper<MyReplysVO> selectMyReplys(UserReplyParam newsReplyParam) {
        PageWarper<MyReplysVO> myReplysVOPageWarper = new PageWarper<>(this.userReplyMapper.selectMyReplys(newsReplyParam));
        List<MyReplysVO> myReplysVOList = myReplysVOPageWarper.getList();
        Long[] newsIds = myReplysVOList.stream().filter(e -> e.getType() == ReplyTypeEnum.NEWS_REPLY.getCode()).mapToLong(MyReplysVO::getNewsId).boxed().toArray(Long[]::new);
        List<NewsVO> newsList = newsSearchIntegrationService.multipleGet(newsIds);
        LocationDTO location = locationIntegrationService.getLocationByGeocode(newsReplyParam.getAreaCode());
        for (MyReplysVO myReplysVO : myReplysVOList) {
            Date addTime = myReplysVO.getAddTime();
            String replyTime = DateUtils.timeConvertString(addTime);
            myReplysVO.setReplyTime(replyTime);
            Byte type = myReplysVO.getType();
            if (type == ReplyTypeEnum.NEWS_REPLY.getCode()) {
                Optional<NewsVO> newsVoOptional = newsList.stream().filter(e -> e.getId().equals(myReplysVO.getNewsId())).findFirst();
                if (newsVoOptional.isPresent()) {
                    NewsVO news = newsVoOptional.get();
                    NewsService newsService = SpringContextHolder.getBean(NewsService.class);
                    //查看新闻是否开启占位符
                    newsService.judgeEnablePlaceholder(news, location);
                    //封面图处理
                    newsService.exchangeDetailNews(news);
                    NewsCollect newsCollect = newsCollectMapper.selectUserCollect(news.getId(), newsReplyParam.getUserId(), (byte) 1);
                    ReplyNewsDTO replyNewsDTO = getNewsDto(news, newsReplyParam.getUserId());
                    if (null != newsCollect) {
                        replyNewsDTO.setCollect(1);
                    } else {
                        replyNewsDTO.setCollect(0);
                    }
                    myReplysVO.setReplyNewsDto(replyNewsDTO);
                    myReplysVO.setTitle(news.getTitle());
                    myReplysVO.setNewsImgUrl(news.getImgUrl());
                }
            } else if (type == ReplyTypeEnum.VIDEO_REPLY.getCode()) {
                Video video = videoMapper.selectByPrimaryKey(myReplysVO.getNewsId());
                if (null != video) {
                    ReplyVideoDTO replyVideoDTO = getVideoDto(video, newsReplyParam.getUserId());
                    replyVideoDTO.setVideoImg("[\"" + video.getVideoImg() + "\"]");
                    myReplysVO.setReplyVideoDto(replyVideoDTO);
                    myReplysVO.setTitle(video.getTitle());
                    myReplysVO.setNewsImgUrl("[\"" + video.getVideoImg() + "\"]");
                }
            } else if (type == ReplyTypeEnum.POST_REPLY.getCode()) {
                ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(myReplysVO.getNewsId());
                if (null != forumPostVo) {
                    ForumPostStatisticService forumPostStatisticService = SpringContextHolder.getBean(ForumPostStatisticService.class);
                    //查看帖子是否开启占位符
                    forumPostStatisticService.judgeEnablePlaceholder(forumPostVo, location);
                    //封面图处理
                    forumPostStatisticService.exchangeDetailPost(forumPostVo);
                    ReplyPostDTO replyPostDTO = getForumPostReplyDto(forumPostVo);
                    myReplysVO.setReplyPostDto(replyPostDTO);
                    myReplysVO.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
                }
            }
        }
        return myReplysVOPageWarper;
    }

    @Override
    public NewsReplyDTO delMyReply(Long replyId, Long userId, Long newsId, Byte isUserDelete) {
        NewsReply newsReply = null;
        if (null != newsId && 0 != newsId) {
            newsReply = this.newsReplyMapper.selectByPrimaryKeyAndNewsId(replyId, newsId);
        }
        if (Objects.isNull(newsReply)) {
            newsReply = this.newsReplyMapper.selectByPrimaryKey(replyId);
        }
        //删除评论逻辑
        newsReply.setIsUserDelete(isUserDelete);
        this.replyRelationship(newsReply);
        NewsReplyDTO newsReplyDTO = new NewsReplyDTO();
        BeanUtils.copyProperties(newsReply, newsReplyDTO);
        //评论状态置为删除
        newsReplyDTO.setDeleteFlag((byte) 1);
        newsReplyDTO.setReplyTime(DateUtils.timeConvertString(newsReplyDTO.getAddTime()));
        newsReplyDTO.setReplyContent("该评论已删除！");
        newsReplyDTO.setIsLike(isLike(userId, replyId) ? 1 : 0);
        if (newsReply.getRootId() == 0) {
            List<NewsReplyMirrorDTO> replyMirrorDTOS = newsReplyMapper.selectSubReplyList(replyId, newsReply.getNewsId());
            List<Long> replyLike = newsReplyMapper.selectReplyLike(newsReply.getNewsId(), userId, replyId);
            replyMirrorDTOS.forEach(e -> {
                dealSubMirrorDTO(e, userId, replyLike);
            });
            newsReplyDTO.setList(replyMirrorDTOS);
        }
        if (newsReply.getLevel() == 1) {
            newsReplyDTO.setParentUserId(null);
            newsReplyDTO.setParentUserNickname(null);
        }
        //用户表评论数统计-1
        userIntegrationService.addPostReplyNum(userId, 4);
        //维护新闻,帖子,小视频评论数
        this.updateComment(newsReply);
        return newsReplyDTO;
    }

    /**
     * 评论删除时处理评论与回复
     *
     * @param newsReply
     */
    private void replyRelationship(NewsReply newsReply) {
        //当删除的评论时根评论时
        this.userReplyMapper.deleteByPrimaryKey(newsReply.getId(), newsReply.getUserId(), newsReply.getIsUserDelete());
        if (newsReply.getRootId() == 0) {
            List<NewsReplyMirrorDTO> list = newsReplyMapper.selectSubReplyList(newsReply.getId(), newsReply.getNewsId());
            if (CollectionUtils.isNotEmpty(list)) {
                newsReply.setDeleteFlag((byte) 2);
                newsReplyMapper.updateByPrimaryKeySelective(newsReply);
            } else {
                this.newsReplyMapper.deleteByPrimaryKey(newsReply.getId(), newsReply.getNewsId(), newsReply.getIsUserDelete());
            }
        } else {
            //删除评论不为根评论时
            this.newsReplyMapper.deleteByPrimaryKey(newsReply.getId(), newsReply.getNewsId(), newsReply.getIsUserDelete());
            List<NewsReplyMirrorDTO> list = newsReplyMapper.selectSubReplyList(newsReply.getRootId(), newsReply.getNewsId());
            NewsReply rootReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getRootId(), newsReply.getNewsId());
            if (rootReply.getDeleteFlag() != 0) {
                if (CollectionUtils.isNotEmpty(list)) {
                    NewsReply record = new NewsReply();
                    record.setId(newsReply.getRootId());
                    record.setNewsId(newsReply.getNewsId());
                    record.setDeleteFlag((byte) 2);
                    newsReplyMapper.updateByPrimaryKeySelective(record);
                }
            }

        }
    }


    @Async
    @Override
    public void doTriggerUpdateInfo(Long userId, String nickname, String headImg) {
        //更新自己的评论
        List<NewsReply> newsReplyList = newsReplyMapper.selectByUser(userId);
        for (NewsReply newsReply : newsReplyList) {
            NewsReply newsReplyUp = new NewsReply();
            newsReplyUp.setId(newsReply.getId());
            newsReplyUp.setUserId(userId);
            newsReplyUp.setNewsId(newsReply.getNewsId());
            newsReplyUp.setUserNickname(nickname);
            newsReplyUp.setHeadImg(headImg);

            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(newsReplyUp, userReply);

            newsReplyMapper.updateUserInfo(newsReplyUp);
            userReplyMapper.updateUserInfo(userReply);
        }
        //更新父级的评论
        List<NewsReply> parentNewsReplyList = newsReplyMapper.selectByParentUser(userId);
        for (NewsReply newsReply : parentNewsReplyList) {
            NewsReply newsReplyUp = new NewsReply();
            newsReplyUp.setId(newsReply.getId());
            newsReplyUp.setUserId(userId);
            newsReplyUp.setNewsId(newsReply.getNewsId());
            newsReplyUp.setParentUserNickname(nickname);
            newsReplyUp.setParentHeadImg(headImg);

            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(newsReplyUp, userReply);

            newsReplyMapper.updateParentUserInfo(newsReplyUp);
            userReplyMapper.updateParentUserInfo(userReply);
        }

    }

    @Override
    public Message doProduceNewsReplyLike(NewsReplyLikeParam newsReplyLikeParam) {
        if (!checkLikeParam(newsReplyLikeParam)) {
            return Message.build(false, "参数验证错误");
        }
        logger.info("点赞的参数是:{}", toJSONString(newsReplyLikeParam));
        KeyGenerator keyGenerator = RedisConfig.NEWS_QUEUE.copy().setKey("newsReplyLikeQueue");

        redisListAdapter.leftPush(keyGenerator, newsReplyLikeParam);
        this.callAsyncConsume();
        return Message.build(true);
    }

    private void callAsyncConsume() {
        NewNewsReplyService newsReplyService = SpringContextHolder.getBean(NewNewsReplyService.class);
        newsReplyService.doNewsReplyLikeConsume();
    }

    @Async
    @Override
    public void doNewsReplyLikeConsume() {
        KeyGenerator keyGenerator = RedisConfig.NEWS_QUEUE.copy().setKey("newsReplyLikeQueue");

        NewsReplyLikeParam newsReplyLikeWarper = redisListAdapter.rightPop(keyGenerator, NewsReplyLikeParam.class);
        if (null != newsReplyLikeWarper) {
            logger.info("新闻回复点赞开始消费...");
            int type = newsReplyLikeWarper.getType();
            NewsReply newsReply;
            if (null != newsReplyLikeWarper.getNewsId() && 0 != newsReplyLikeWarper.getNewsId()) {
                newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReplyLikeWarper.getReplyId(), newsReplyLikeWarper.getNewsId());
            } else {
                newsReply = newsReplyMapper.selectByPrimaryKey(newsReplyLikeWarper.getReplyId());
            }
            if (newsReply != null) {
                int count = saveReplyLiKeRecord(newsReply.getId(), newsReplyLikeWarper.getUserId(), type);
                if (count > 0 && type == 1) {
                    //1.增加评论点赞记录
                    logger.debug("评论实体:{}", JSONObject.toJSON(newsReply));
                    int likeCount = generatorLikeCount(type, newsReply.getLikeCount());
                    logger.debug("点赞数:{}", likeCount);
                    NewsReply newsReplyUp = new NewsReply();
                    newsReplyUp.setId(newsReplyLikeWarper.getReplyId());
                    newsReplyUp.setLikeCount(likeCount);
                    newsReplyUp.setUserId(newsReply.getUserId());
                    newsReplyUp.setNewsId(newsReply.getNewsId());
                    //根评论才计算互动值
                    if (newsReply.getRootId() == 0) {
                        newsReplyUp.setInteractiveCount(newsReply.getInteractiveCount() + 1);
                    }
                    newsReplyMapper.updateByPrimaryKeySelective(newsReplyUp);
                    UserReply userReply = new UserReply();
                    BeanUtils.copyProperties(newsReplyUp, userReply);
                    userReplyMapper.updateinteractiveCountById(userReply);

                    //2.评论点赞推送
                    if ((!newsReplyLikeWarper.getUserId().equals(newsReply.getUserId())) && newsReplyLikeWarper.getType() == 1) {
                        LocationDTO location = locationIntegrationService.getLocationByGeocode(newsReplyLikeWarper.getAreaCode());
                        if (newsReply.getType() == 1 || newsReply.getType() == 2) {
                            NewsService newsService = SpringContextHolder.getBean(NewsService.class);
                            News news = newsSearchIntegrationService.multipleGet(new Long[]{newsReply.getNewsId()}).get(0);
                            newsService.judgeEnablePlaceholder(news, location);
                            newsService.exchangeDetailNews(news);
                            messageService.pushNewLikeMessage(newsReplyLikeWarper, PushMessageEnum.NEWS_LIKE, newsReply, null, news);
                        } else if (newsReply.getType() == 3) {
                            ForumPostVo postVo = forumPostMapper.selectByPrimaryKey(newsReply.getNewsId());
                            SpringContextHolder.getBean(ForumPostStatisticService.class).judgeEnablePlaceholder(postVo, location);
                            SpringContextHolder.getBean(ForumPostStatisticService.class).exchangeDetailPost(postVo);
                            messageService.pushNewLikeMessage(newsReplyLikeWarper, PushMessageEnum.POST_LIKE, newsReply, postVo, null);
                        }
                    }

                    //调用增加用户点赞数接口
                    userIntegrationService.updateUserLikeNumByUserId(newsReply.getUserId());


                }
            }
            doNewsReplyLikeConsume();
        }
    }

    @Override
    public NewsReplyDetailDTO getNewsReplyDetailDTO(Long replyId, Long userId, Long newsId, String areaCode) {
        NewsReply newsReply;
        if (null != newsId && 0 != newsId) {
            newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(replyId, newsId);
        } else {
            newsReply = newsReplyMapper.selectByPrimaryKey(replyId);
        }
        if (Objects.isNull(newsReply) || Objects.isNull(newsReply.getRootId())) {
            return null;
        }
        //判断评论是否是跟评论,不是则找出根评论
        if (newsReply.getRootId() != 0) {
            if (null != newsId && 0 != newsId) {
                newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getRootId(), newsId);
            } else {
                newsReply = newsReplyMapper.selectByPrimaryKey(newsReply.getRootId());
            }
        }
        NewsReplyDetailDTO newsReplyDetailDTO = new NewsReplyDetailDTO();
        if (null != newsReply) {
            BeanUtils.copyProperties(newsReply, newsReplyDetailDTO);
            if (isLike(userId, newsReplyDetailDTO.getId())) {
                newsReplyDetailDTO.setIsLike(1);
            }
            newsReplyDetailDTO.setReplyTime(DateUtils.timeConvertString(newsReplyDetailDTO.getAddTime()));
            //添加评论的新闻.帖子.或视频的详细实体
            this.addSuperiorObject(newsReply, userId, newsReplyDetailDTO, areaCode);
            //查询该评论下的回复
            List<NewsReplyMirrorDTO> replies = this.newsReplyMapper.
                    selectSubReplyList(newsReply.getId(), newsReply.getNewsId());
            if (CollectionUtils.isNotEmpty(replies)) {
                replies.forEach(replay -> {
                    List<Long> replyLike = newsReplyMapper.selectReplyLike(replay.getNewsId(), userId, replyId);
                    this.dealSubMirrorDTO(replay, userId, replyLike);
                });
            }
            newsReplyDetailDTO.setList(replies);
        }
        return newsReplyDetailDTO;
    }

    /**
     * 添加评论新闻.帖子.或视频的详细实体
     *
     * @param newsReply
     * @param userId
     * @param newsReplyDetailDTO
     * @return
     */
    private NewsReplyDetailDTO addSuperiorObject(NewsReply newsReply, Long userId, NewsReplyDetailDTO newsReplyDetailDTO, String areaCode) {
        Byte type = newsReply.getType();
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        if (type == ReplyTypeEnum.NEWS_REPLY.getCode()) {
            News news = newsSearchIntegrationService.multipleGet(new Long[]{newsReply.getNewsId()}).get(0);
            if (null != news) {
                SpringContextHolder.getBean(NewsService.class).judgeEnablePlaceholder(news, location);
                ReplyNewsDTO replyNewsDTO = getNewsDto(news, userId);
                newsReplyDetailDTO.setReplyNewsDto(replyNewsDTO);
                newsReplyDetailDTO.setTitle(news.getTitle());
            }
        } else if (type == ReplyTypeEnum.VIDEO_REPLY.getCode()) {
            Video video = videoMapper.selectByPrimaryKey(newsReply.getNewsId());
            if (null != video) {
                ReplyVideoDTO replyVideoDTO = getVideoDto(video, userId);
                newsReplyDetailDTO.setReplyVideoDto(replyVideoDTO);
                newsReplyDetailDTO.setTitle(video.getTitle());
            }
        } else if (type == ReplyTypeEnum.POST_REPLY.getCode()) {
            ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(newsReply.getNewsId());
            SpringContextHolder.getBean(ForumPostStatisticService.class).judgeEnablePlaceholder(forumPostVo, location);
            if (null != forumPostVo) {
                ReplyPostDTO replyPostDTO = getForumPostReplyDto(forumPostVo);
                newsReplyDetailDTO.setReplyPostDto(replyPostDTO);
                newsReplyDetailDTO.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
            }
        }
        return newsReplyDetailDTO;
    }

    private ReplyPostDTO getForumPostReplyDto(ForumPostVo forumPostVo) {
        ReplyPostDTO replyPostDTO = new ReplyPostDTO();
        replyPostDTO.setId(forumPostVo.getId());
        replyPostDTO.setStatus((byte) forumPostVo.getStatus().intValue());
        replyPostDTO.setTitle(forumPostVo.getTitle());
        replyPostDTO.setPostImgList(forumPostVo.getPostImgList());

        return replyPostDTO;
    }

    private ReplyNewsDTO getNewsDto(News news, Long userId) {
        ReplyNewsDTO replyNewsDTO = new ReplyNewsDTO();
        replyNewsDTO.setId(news.getId());
        replyNewsDTO.setImgUrl(news.getImgUrl());
        replyNewsDTO.setLinkUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.DETAIL, news.getId(), userId));
        replyNewsDTO.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.SHARE, news.getId(), userId));
        replyNewsDTO.setComments(news.getComments());
        replyNewsDTO.setStatus(news.getStatus());
        return replyNewsDTO;
    }

    private ReplyVideoDTO getVideoDto(Video video, Long userId) {
        ReplyVideoDTO replyVideoDTO = new ReplyVideoDTO();
        replyVideoDTO.setAuthorImg(video.getAuthorImg());
        replyVideoDTO.setAuthor(video.getAuthorName());
        replyVideoDTO.setId(video.getId());

        VideoLike videoLike = videoLikeMapper.selectByModel(new VideoLike(video.getId(), userId));
        if (null != videoLike) {
            replyVideoDTO.setPraise(true);
        } else {
            replyVideoDTO.setPraise(false);
        }

        replyVideoDTO.setLikeCount(Math.toIntExact(video.getLikeCount()));
        replyVideoDTO.setVideoUrl(videoRecommendService.videoAddressConversion(video.getVideoUrl()));
        replyVideoDTO.setVideoImg(video.getVideoImg());
        replyVideoDTO.setCommentCount(Math.toIntExact(video.getCommentCount()));
        replyVideoDTO.setStatus(video.getStatus());
        return replyVideoDTO;
    }

    /**
     * 得到更新之后的点赞数
     *
     * @param type
     * @param likeCount
     * @return
     */
    private int generatorLikeCount(int type, int likeCount) {
        if (0 == type) {
            if (likeCount > 0) {
                likeCount -= 1;
            }
        } else {
            likeCount += 1;
        }
        return likeCount;
    }

    private int saveReplyLiKeRecord(Long replyId, Long userId, int type) {
        int count = 0;
        NewsReplayLike newsReplayLike = newsReplyLikeMapper.selectByModel(new NewsReplayLike(replyId, userId));
        if (type == 1) {
            if (null == newsReplayLike) {
                NewsReplayLike newsReplayLikeNew = new NewsReplayLike();
                newsReplayLikeNew.setAddTime(new Date());
                newsReplayLikeNew.setId(nextSequence());
                newsReplayLikeNew.setReplyId(replyId);
                newsReplayLikeNew.setUserId(userId);
                count = newsReplyLikeMapper.insert(newsReplayLikeNew);

            }
        } else {
            if (null != newsReplayLike) {
                count = newsReplyLikeMapper.deleteByPrimaryKey(newsReplayLike.getId());
            }
        }
        return count;
    }


    /**
     * 评论点赞时验证参数
     *
     * @param newsReplyLikeParam
     * @return
     */
    private boolean checkLikeParam(NewsReplyLikeParam newsReplyLikeParam) {
        return null != newsReplyLikeParam.getReplyId() && null != newsReplyLikeParam.getUserId();
    }


    @Override
    public Boolean doDealHistoryReply() {
        //删除新闻评论表中user_id = 0,parent_user_id = 0 数据
        newsReplyMapper.deleteNewsReply();
        List<VirtualUserInfo> userList = userIntegrationService.getVirtualUserList(1000);
        for (int i = 0; i < 10; i++) {
            String tableName = "t_news_reply_" + i;
            List<NewsReply> replyList = newsReplyMapper.selectByUserIdIsZero(tableName);
            dealReplyList(replyList, userList);
        }
        return Boolean.TRUE;
    }

    @Override
    public void statisticalPostReply() {
        //获得一段时间内,所有的帖子评论数量
        KeyGenerator postIdKey = RedisConfig.FORUM_REPLY_POSTID.copy();
        try {
            //开启线程的数量
            int threadNum = 10;
            Set<Long> postIds = redisSetAdapter.getAllMembers(postIdKey, new TypeReference<Long>() {
            });
            CountDownLatch countDownLatch = new CountDownLatch(threadNum);
            for (int i = 0; i < threadNum; i++) {
                final int j = i;
                List<Long> collect = postIds.stream()
                        .filter(e -> e % threadNum == j)
                        .collect(Collectors.toList());
                asyncTaskExecutor.execute(new PostReplyThread(collect, countDownLatch));
            }
            countDownLatch.await();
        } catch (InterruptedException e) {
            logger.error("线程等待异常", e);
        } finally {
            redisSetAdapter.remove(postIdKey);
        }
    }

    private void dealReplyList(List<NewsReply> replyList, List<VirtualUserInfo> userList) {
        replyList.forEach(e -> {
            int random = ThreadLocalRandom.current().nextInt(0, userList.size());
            VirtualUserInfo userInfo = userList.get(random);
            e.setUserId(userInfo.getId());
            e.setHeadImg(userInfo.getHeadImg());
            e.setUserNickname(userInfo.getNickname());
            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(e, userReply);
            newsReplyMapper.updateReplyUserInfo(e);
            userReplyMapper.insertSelective(userReply);

        });
    }


}
