package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.common.vo.MPage;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.BizLogIntegrationService;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.NewsSearchIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.NewsCollectMapper;
import com.bxm.localnews.news.dto.MyCollectDTO;
import com.bxm.localnews.news.dto.NewsCollectDTO;
import com.bxm.localnews.news.event.CollectActionEvent;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.NewsCollectPageParam;
import com.bxm.localnews.news.service.ForumPostStatisticService;
import com.bxm.localnews.news.service.NewsCollectService;
import com.bxm.localnews.news.service.NewsService;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.Page;
import com.google.common.eventbus.EventBus;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Created by Administrator on 2018/2/23 0023.
 */
@Service
public class NewsCollectServiceImpl extends BaseService implements NewsCollectService {

    @Resource
    private NewsCollectMapper newsCollectMapper;

    @Resource
    private EventBus userActionEventBus;

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private NewsSearchIntegrationService newsSearchIntegrationService;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private BizLogIntegrationService bizLogIntegrationService;

    @Resource
    private ForumProperties forumProperties;


    @Override
    public List<NewsCollectDTO> queryList(Long userId, MPage page) {
        List<NewsCollectDTO> resultList = newsCollectMapper.queryList(userId, (byte) 1, page);
        //过滤为空的新闻
        return resultList.stream().filter(Objects::nonNull).filter(e -> StringUtils.isNotBlank(e.getTitle())).collect(Collectors.toList());
    }

    @Override
    public Json collectNews(NewsCollect newsCollect, Integer platform, String areaCode) {
        NewsCollect collect = this.newsCollectMapper.selectUserCollect(
                newsCollect.getNewsId(),
                newsCollect.getUserId(),
                newsCollect.getType());

        if (!ObjectUtils.isEmpty(collect)) {
            this.newsCollectMapper.deleteByModel(collect);
            if (newsCollect.getType() == 1) {

                userActionEventBus.post(CollectActionEvent.of()
                        .setTargetId(newsCollect.getNewsId())
                        .setState(Boolean.FALSE)
                        .setUserId(newsCollect.getUserId()));
            } else if (newsCollect.getType() == 3) {
                //更新收藏数量
                ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
                forumPostStatistic.setCollectCount(-1);
                forumPostStatistic.setId(newsCollect.getNewsId());
                this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);
            }
        } else {
            newsCollect.setId(nextSequence());
            this.newsCollectMapper.insertSelective(newsCollect);

            if (newsCollect.getType() == 1) {
                userActionEventBus.post(CollectActionEvent.of()
                        .setTargetId(newsCollect.getNewsId())
                        .setState(Boolean.TRUE)
                        .setUserId(newsCollect.getUserId()));
            } else if (newsCollect.getType() == 3) {
                //更新收藏数量
                ForumPostStatistic forumPostStatistic = new ForumPostStatistic();
                forumPostStatistic.setCollectCount(1);
                forumPostStatistic.setId(newsCollect.getNewsId());
                this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);
            }

            if (Objects.nonNull(platform)) {
                bizLogIntegrationService.forumCollectSuccessed(newsCollect.getUserId(), newsCollect.getNewsId(), areaCode, Integer.valueOf(newsCollect.getType()), platform);
            }
        }
        return ResultUtil.genSuccessResult();
    }

    @Override
    public PageWarper<MyCollectDTO> listCollect(NewsCollectPageParam newsCollectPageParam) {
        PageWarper<NewsCollect> newsCollectPageWarper
                = new PageWarper<>(newsCollectMapper.queryByPageSize(newsCollectPageParam));

        List<NewsCollect> newsCollectList = newsCollectPageWarper.getList();
        //根据type得到帖子的id列表
        List<Long> postIdList = newsCollectList.stream()
                .filter(mix -> mix.getType() == 3)
                .map(NewsCollect::getNewsId).collect(Collectors.toList());

        //根据type筛选获得新闻id列表获得新闻列表内容，进行包装
        List<News> list = newsSearchIntegrationService.listRecommendedNews(newsCollectList.stream()
                .filter(mix -> mix.getType() == 1)
                .map(NewsCollect::getNewsId).toArray(Long[]::new));

        //根据id列表获得帖子列表内容，进行包装
        List<ForumPostVo> forumPostVoList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(postIdList)) {
            forumPostVoList = forumPostMapper.listPostByIdsInHome(postIdList);
        }

        //取得地区
        LocationDTO location = locationIntegrationService.getLocationByGeocode(newsCollectPageParam.getAreaCode());

        //分别将新闻和帖子组装成收藏的返回实体
        Map<Long, MyCollectDTO> myNewsCollectDtoList = list.stream()
                .map(e -> this.newsToMyCollectDTO(e, location))
                .collect(Collectors.toMap(MyCollectDTO::getId, Function.identity()));

        Map<Long, MyCollectDTO> myForumCollectDtoList = forumPostVoList.stream()
                .map(e -> this.forumToMyCollectDTO(e, location))
                .collect(Collectors.toMap(MyCollectDTO::getId, Function.identity()));


        List<MyCollectDTO> myCollectDtoList = new ArrayList<>();
        //塞入分页结果体中
        for (NewsCollect newsCollect : newsCollectList) {
            if (newsCollect.getType() == 1) {
                MyCollectDTO myCollectDtoOptional = myNewsCollectDtoList.get(newsCollect.getNewsId());
                if (null != myCollectDtoOptional) {
                    myCollectDtoOptional.setCollectTime(DateUtils.timeConvertString(newsCollect.getAddTime()));
                    myCollectDtoList.add(myCollectDtoOptional);
                }
            } else if (newsCollect.getType() == 3) {
                MyCollectDTO myCollectDtoOptional = myForumCollectDtoList.get(newsCollect.getNewsId());
                if (null != myCollectDtoOptional) {
                    myCollectDtoOptional.setCollectTime(DateUtils.timeConvertString(newsCollect.getAddTime()));
                    myCollectDtoList.add(myCollectDtoOptional);
                }
            }
        }

        //分页结果
        Page page = new Page(newsCollectPageWarper.getPageNum(), newsCollectPageWarper.getPageSize());
        page.setTotal(newsCollectPageWarper.getTotal());
        page.addAll(myCollectDtoList);

        return new PageWarper<>(page);

    }

    /**
     * 组装帖子收藏返回实体（对小纸条做额外处理）
     *
     * @param forumPostVo
     * @param location
     * @return
     */
    private MyCollectDTO forumToMyCollectDTO(ForumPostVo forumPostVo, LocationDTO location) {
        //判断帖子是否开启占位符
        ForumPostStatisticService forumPostStatisticService = SpringContextHolder.getBean(ForumPostStatisticService.class);
        forumPostStatisticService.judgeEnablePlaceholder(forumPostVo, location);

        MyCollectDTO myCollectDTO = new MyCollectDTO();
        myCollectDTO.setId(forumPostVo.getId());

        //如果是帖子
        if (1 == forumPostVo.getPostType()) {
            myCollectDTO.setImgList(forumPostStatisticService.getDetailFromPost(forumPostVo.getPostImgList()).stream().map(PostImgVo::getImgUrl).collect(Collectors.toList()));
            myCollectDTO.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
            myCollectDTO.setType((byte) 3);
            //如果是小纸条，则需额外做处理
        } else if (2 == forumPostVo.getPostType()) {
            myCollectDTO.setImgList(Collections.singletonList(forumProperties.getNoteShareImg()));
            myCollectDTO.setTitle(forumPostVo.getTextField());
            myCollectDTO.setType((byte) 4);
        }

        return myCollectDTO;
    }

    /**
     * 组装新闻收藏返回实体
     *
     * @param news
     * @param location
     * @return
     */
    private MyCollectDTO newsToMyCollectDTO(News news, LocationDTO location) {
        //判断新闻是否开启占位符
        NewsService newsService = SpringContextHolder.getBean(NewsService.class);
        newsService.judgeEnablePlaceholder(news, location);

        MyCollectDTO myCollectDTO = new MyCollectDTO();
        myCollectDTO.setImgList(SpringContextHolder.getBean(NewsService.class).getDetailFromNews(news.getImgUrl(), news.getVideoList()).stream().map(PostImgVo::getImgUrl).collect(Collectors.toList()));
        myCollectDTO.setId(news.getId());
        myCollectDTO.setTitle(news.getTitle());
        myCollectDTO.setType((byte) 1);
        return myCollectDTO;
    }

}
