package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.integration.LocationIntegrationService;
import com.bxm.localnews.integration.MissionIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.enums.NewsShareEnum;
import com.bxm.localnews.news.event.ShareActionEvent;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.ExecGoldCalParam;
import com.bxm.localnews.news.param.ExecGoldParam;
import com.bxm.localnews.news.service.ForumPostStatisticService;
import com.bxm.localnews.news.service.NewsReadRewardService;
import com.bxm.localnews.news.service.NewsService;
import com.bxm.localnews.news.strategy.GoldenStrategyContext;
import com.bxm.localnews.news.vo.*;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.google.common.eventbus.EventBus;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.util.Date;

import static com.bxm.localnews.news.constant.AppConst.UN_GOLD;
import static com.bxm.localnews.news.constant.RedisConfig.USER_SHARE_NEWS_RECORD;

@Service
public class NewsReadRewardServiceImpl extends BaseService implements NewsReadRewardService {

    private UserIntegrationService userIntegrationService;

    private RedisStringAdapter redisStringAdapter;

    private NewsStatisticsMapper newsStatisticsMapper;

    private NewsCollectMapper newsCollectMapper;

    private NewsMapper newsMapper;

    private NewsShareMapper newsShareMapper;

    private GoldenStrategyContext goldenStrategyContext;

    private MissionIntegrationService missionIntegrationService;

    private EventBus userActionEventBus;

    private ForumPostMapper forumPostMapper;

    private ForumPostShareMapper forumPostShareMapper;

    private DistributedLock distributedLock;

    private LocationIntegrationService locationIntegrationService;

    @Autowired
    public NewsReadRewardServiceImpl(UserIntegrationService userIntegrationService,
                                     RedisStringAdapter redisStringAdapter,
                                     NewsStatisticsMapper newsStatisticsMapper,
                                     NewsCollectMapper newsCollectMapper,
                                     NewsMapper newsMapper, NewsShareMapper newsShareMapper,
                                     GoldenStrategyContext goldenStrategyContext,
                                     MissionIntegrationService missionIntegrationService,
                                     EventBus userActionEventBus,
                                     ForumPostMapper forumPostMapper,
                                     ForumPostShareMapper forumPostShareMapper,
                                     DistributedLock distributedLock,
                                     LocationIntegrationService locationIntegrationService) {
        this.userIntegrationService = userIntegrationService;
        this.redisStringAdapter = redisStringAdapter;
        this.newsStatisticsMapper = newsStatisticsMapper;
        this.newsCollectMapper = newsCollectMapper;
        this.newsMapper = newsMapper;
        this.newsShareMapper = newsShareMapper;
        this.goldenStrategyContext = goldenStrategyContext;
        this.missionIntegrationService = missionIntegrationService;
        this.userActionEventBus = userActionEventBus;
        this.forumPostMapper = forumPostMapper;
        this.forumPostShareMapper = forumPostShareMapper;
        this.distributedLock = distributedLock;
        this.locationIntegrationService = locationIntegrationService;
    }

    @Override
    public NewsCompleTaskDTO saveShareNews(Long newsId, Long userId, Byte type, String areaCode, int platform) {
        logger.info("用户分享新闻，参数->newsId:{},userId:{},type:{},platform:{}", newsId, userId, type, platform);
        UserBean user = this.userIntegrationService.selectUserFromCache(userId);

        //用户未正式注册前不能分享新闻获得金币
        if (user == null || user.getState() != 1) {
            return new NewsCompleTaskDTO();
        }

        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        if (NewsShareEnum.SIGN_SHARE.getCode() == type) {
            newsCompleTaskDTO = missionIntegrationService.signCompleTaskDTO(userId, "TASK_SIGN_SHARE", newsId.toString(), "");
        } else if (NewsShareEnum.SUPPLEMENT_SHARE.getCode() == type) {
            //补签分享不发送任务
        } else {
            newsCompleTaskDTO = completeNewsShareTask(userId, newsId, areaCode);
        }
        //分享成功获取金币记录
        NewsShare share = new NewsShare();
        share.setAddTime(new Date());
        share.setNewsId(newsId);
        share.setUserId(userId);
        share.setId(nextSequence());
        share.setReward(Math.toIntExact(newsCompleTaskDTO.getGoldNum()));
        share.setType(type);
        this.newsShareMapper.insertSelective(share);

        userActionEventBus.post(ShareActionEvent.of().setTargetId(newsId)
                .setUserId(userId));

        return newsCompleTaskDTO;

    }

    private NewsCompleTaskDTO completeNewsShareTask(Long userId, Long newsId, String areaCode) {
        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        KeyGenerator shareRecordKey = getShareRecordKey(userId, newsId);
        if (!redisStringAdapter.hasKey(shareRecordKey)) {
            //发放分享文章奖励
            News news = newsMapper.selectByPrimaryKey(newsId);
            NewsService newsService = SpringContextHolder.getBean(NewsService.class);
            LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
            newsService.judgeEnablePlaceholder(news, location);
            String content = "转发【" + ExtendFactory.getContent(news.getTitle(), news.getContent()) + "】";
            newsCompleTaskDTO = missionIntegrationService.compleTask(userId, "TASK_NEWS_SHARE", newsId.toString(), content);
            //记录是否分享过该新闻
            redisStringAdapter.set(shareRecordKey, "exist", DateUtils.getCurSeconds());
        }
        return newsCompleTaskDTO;
    }

    /**
     * 得到文章分享页的key
     *
     * @param userId
     * @param newsId
     * @return
     */
    private KeyGenerator getShareRecordKey(Long userId, Long newsId) {
        return USER_SHARE_NEWS_RECORD.copy().appendKey(userId.toString())
                .appendKey(DateUtils.formatDate(new Date()))
                .appendKey(newsId.toString());
    }


    @Override
    public NewsCompleTaskDTO saveShareForums(Long postId, Long userId, Byte type, String areaCode) {
        UserBean user = this.userIntegrationService.selectUserFromCache(userId);

        //用户未正式注册前不能分享新闻获得金币
        if (user == null || user.getState() != 1) {
            return new NewsCompleTaskDTO();
        }

        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        if (NewsShareEnum.SIGN_SHARE.getCode() == type) {
            newsCompleTaskDTO = missionIntegrationService.signCompleTaskDTO(userId, "TASK_SIGN_SHARE", postId.toString(), "");
        } else if (NewsShareEnum.SUPPLEMENT_SHARE.getCode() == type) {
            //补签分享不发送任务
        } else {
            newsCompleTaskDTO = completePostShareTask(userId, postId, areaCode);
        }

        //分享成功获取金币记录
        ForumPostShare forumPostShare = new ForumPostShare();
        forumPostShare.setAddTime(new Date());
        forumPostShare.setId(nextSequence());
        forumPostShare.setPostId(postId);
        forumPostShare.setReward(Math.toIntExact(newsCompleTaskDTO.getGoldNum()));
        forumPostShare.setUserId(userId);
        forumPostShare.setType(type);
        forumPostShareMapper.insertSelective(forumPostShare);

        //更新分享数量
        ForumPostStatistic forumPostStatistic = ForumPostStatistic.buildShares(postId, 1);
        this.forumPostMapper.updateStatisticByPrimaryKeySelective(forumPostStatistic);

        return newsCompleTaskDTO;
    }

    private NewsCompleTaskDTO completePostShareTask(Long userId, Long postId, String areaCode) {
        NewsCompleTaskDTO newsCompleTaskDTO = new NewsCompleTaskDTO();
        KeyGenerator shareRecordKey = getShareRecordKey(userId, postId);
        if (!redisStringAdapter.hasKey(shareRecordKey)) {
            //发放分享文章奖励
            ForumPostVo forumPostVo = forumPostMapper.selectByPrimaryKey(postId);
            LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
            SpringContextHolder.getBean(ForumPostStatisticService.class).judgeEnablePlaceholder(forumPostVo, location);
            String content = "转发【" + ExtendFactory.getPostContent(forumPostVo.getTitle(), forumPostVo.getTextField()) + "】成功";
            newsCompleTaskDTO = missionIntegrationService.compleTask(userId, "TASK_NEWS_SHARE", postId.toString(), content);
            //记录是否分享过该新闻
            redisStringAdapter.set(shareRecordKey, "exist", DateUtils.getCurSeconds());
        }
        return newsCompleTaskDTO;
    }

    @Override
    public Json<NewsGoldMeta> execGetGold(ExecGoldParam execGoldParam) {
        //首先判断有无此新闻
        if (null != execGoldParam.getUserId() && null != execGoldParam.getNewsId()) {
            News news = this.newsMapper.selectByPrimaryKey(execGoldParam.getNewsId());
            if (isNewsExsit(news, execGoldParam.getViewTime())) {

                //填充一些额外参数
                NewsGoldMeta newsGoldMeta = new NewsGoldMeta();
                ExecGoldCalParam execGoldCalParam = new ExecGoldCalParam();
                BeanUtils.copyProperties(execGoldParam, execGoldCalParam);
                execGoldCalParam.setNewsGoldMeta(newsGoldMeta);
                execGoldCalParam.setNews(news);

                //默认策略
                goldenStrategyContext.chooseCalulator(execGoldCalParam);

                return ResultUtil.genSuccessResult(execGoldCalParam.getNewsGoldMeta());
            }
        }
        //否则返回默认的，防止出现参数错误
        NewsGoldMeta newsGoldMeta = new NewsGoldMeta();
        newsGoldMeta.setGoldNum(0);
        newsGoldMeta.setTaskName("阅读新闻");
        newsGoldMeta.setGoldType(UN_GOLD);
        newsGoldMeta.setTotalGold(BigDecimal.ZERO);
        return ResultUtil.genSuccessResult(newsGoldMeta);
    }

    private boolean isNewsExsit(News news, Integer viewTime) {
        return news != null && viewTime != null;
    }
}
