package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.constant.PlatformEnum;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.dto.LocationDTO;
import com.bxm.localnews.dto.MixRecommendDTO;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.news.config.NewsProperties;
import com.bxm.localnews.news.config.UserProperties;
import com.bxm.localnews.news.constant.AppConst;
import com.bxm.localnews.news.convert.impl.ForumPostConvert;
import com.bxm.localnews.news.domain.*;
import com.bxm.localnews.news.dto.NewsCompleTaskDTO;
import com.bxm.localnews.news.dto.NewsLikeDTO;
import com.bxm.localnews.news.dto.PostLikeDto;
import com.bxm.localnews.news.enums.*;
import com.bxm.localnews.news.event.ViewActionEvent;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.factory.impl.ExtendFactory;
import com.bxm.localnews.news.param.NewsLikeParam;
import com.bxm.localnews.news.param.NewsParam;
import com.bxm.localnews.news.service.AssemblyService;
import com.bxm.localnews.news.service.ForumPostStatisticService;
import com.bxm.localnews.news.service.NewsService;
import com.bxm.localnews.news.service.RecommendService;
import com.bxm.localnews.news.vo.*;
import com.bxm.localnews.param.AccountGoldParam;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.HtmlContentUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Lists;
import com.google.common.eventbus.EventBus;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.interceptor.TransactionAspectSupport;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class NewsServiceImpl extends BaseService implements NewsService {

    @Resource
    private NewsMapper newsMapper;

    @Resource
    private NewsRecordMapper newsRecordMapper;

    @Resource
    private NewsStatisticsMapper newsStatisticsMapper;

    @Resource
    private NewsSearchWordMapper newsSearchWordMapper;

    @Resource
    private NewsKindMapper newsKindMapper;

    @Resource
    private AppVersionIntegrationService appVersionIntegrationService;

    @Resource
    private NewsRecommendIntegrationService newsRecommendIntegrationService;

    @Resource
    private NewsSearchIntegrationService newsSearchIntegrationService;

    @Resource
    private EventBus userActionEventBus;

    @Resource
    private NewsCollectMapper newsCollectMapper;

    @Resource
    private AssemblyService assemblyService;

    @Resource
    private MissionIntegrationService missionIntegrationService;

    @Resource
    private MessageService messageService;

    @Resource
    private IUrlFactory iUrlFactory;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private NewsLikeMapper newsLikeMapper;

    @Resource
    private UserProperties userProperties;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    @Resource
    private NewsProperties newsProperties;

    @Override
    public Json<NewsUrlVo> getUrlAddress(Long newsId, Long userId, Byte type) {
        //判断是否是新闻
        if (isNews(type)) {
            News news = this.newsMapper.selectByPrimaryKey(newsId);
            if (null != news) {
                String url = getNewsUrlBegin(type, news.getType()) + "newsId=" + newsId + "&userId=" + userId + "&type=" + type;
                NewsUrlVo newsUrlVo = new NewsUrlVo(url);
                return ResultUtil.genSuccessResult(newsUrlVo);
            }
        }

        logger.error("获得详情地址时新闻不存在-newsId:{},userId:{},type:{}", newsId, userId, type);
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "获得详情地址时新闻不存在");
    }

    private boolean isEssayShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.ESSAY;
    }

    private boolean isIconEssayShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.ICON_ESSAY;
    }

    private boolean isVideoShare(Byte type, Byte newsType) {
        return type == 2 && newsType == AppConst.NEWS.VIDEO;
    }

    private boolean isEssayDetail(Byte type) {
        return type == 1 || type == 3;
    }

    private boolean isNews(Byte type) {
        return type != 4;
    }

    /**
     * 组装新闻url
     */
    private String getNewsUrlBegin(byte type, byte newsType) {
        String url = "";
        if (isEssayShare(type, newsType)) {
            url = "shareNewsDetail.html?";
        }
        if (isIconEssayShare(type, newsType)) {
            url = "sharePicDetail.html?";
        }
        if (isVideoShare(type, newsType)) {
            url = "shareVideoDetail.html?";
        }
        if (isEssayDetail(type)) {
            url = "newsDetail.html?";
        }
        return url;
    }

    @Override
    public Json<NewsDetailVO> getNewsDetail(NewsParam newsParam, BasicParam basicParam) {
        long begin = System.currentTimeMillis();
        if (null == newsParam.getNewsId()) {
            return ResultUtil.genSuccessResult(new NewsDetailVO());
        }

        List<NewsVO> newsVoList = newsSearchIntegrationService.multipleGet(new Long[]{newsParam.getNewsId()});

        if (newsVoList.isEmpty()) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "获取详情时新闻缺失");
        }

        NewsVO newsVO = newsVoList.get(0);
        if (newsVO.getStatus() != 1) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "新闻已下架");
        }
        //填充额外信息
        this.fillDetailExtInfo(newsVO, newsParam, basicParam);

        NewsDetailVO newsDetailVO = new NewsDetailVO(newsVO);

        logger.debug("[getNewsDetail]新闻详情es拉取完毕, param:[{}] ,content:[{}],耗时:[{}]",
                newsParam, JSONObject.toJSONString(newsDetailVO), System.currentTimeMillis() - begin);

        //将用户行为记录上报消息中心
        if (null != newsParam.getUserId()) {
            userActionEventBus.post(ViewActionEvent.of()
                    .setTargetId(newsParam.getNewsId())
                    .setUserId(newsParam.getUserId()));

        }

        return ResultUtil.genSuccessResult(newsDetailVO);
    }

    /**
     * 填充详情数据
     *
     * @param newsVO
     * @param newsParam
     */
    private void fillDetailExtInfo(NewsVO newsVO, NewsParam newsParam, BasicParam basicParam) {
        //判断用户是否收藏该新闻
        //查询当前用户是否对该新闻点赞
        newsVO.setCollect(0);
        newsVO.setLiked((byte) 0);
        if (newsParam.getUserId() != null) {
            newsVO.setCollect(newsCollectMapper.selectUserCollect(newsParam.getNewsId(), newsParam.getUserId(), (byte) 1) != null ? 1 : 0);
            NewsLike newsLike = newsLikeMapper.selectByUserIdAndNewsId(newsParam.getUserId(), newsParam.getNewsId());
            if (Objects.nonNull(newsLike)) {
                newsVO.setLiked((byte) 1);
            }
        }

        LocationDTO location = locationIntegrationService.getLocationByGeocode(newsParam.getAreaCode());
        judgeEnablePlaceholder(newsVO, location);

        if (StringUtils.isNotBlank(newsVO.getEditorMessage())) {
            newsVO.setContent(newsVO.getContent() + newsVO.getEditorMessage());
        }

        //判断新闻是否是本地新闻
        newsVO.setLocal((byte) 1);
        if (StringUtils.isNotEmpty(newsVO.getAreaDetail())) {
            newsVO.setLocal((byte) 2);
        }

        //添加分享链接
        newsVO.setLinkUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.DETAIL, newsParam.getNewsId(), newsParam.getUserId()));
        newsVO.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.SHARE, newsParam.getNewsId(), newsParam.getUserId()));
        newsVO.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.NEWS, newsParam.getNewsId(), newsParam.getUserId()));

        //如过审则将敏感信息设置为空
        if (appVersionIntegrationService.getPublishState(newsParam)) {
            newsVO.setAuthor(StringUtils.EMPTY);
            newsVO.setAuthorImg(StringUtils.EMPTY);
        }

        //强制引导下载为空时给默认值 0
        if (Objects.isNull(newsVO.getIsBootDownload())) {
            newsVO.setIsBootDownload((byte) 0);
        }

        String shareDesc = HtmlContentUtils.getHtmlDesc(newsVO.getContent(), 50);
        newsVO.setShareDesc(shareDesc);

        this.exchangeDetailNews(newsVO);
    }


    @Override
    @Async
    public void completeTaskAndPush(NewsParam newsParam, NewsVO newsVO) {
        if (PlatformEnum.APPLET.getCode() == newsParam.getPlatform()
                && Objects.nonNull(newsParam.getShareUserId())
                && !newsParam.getShareUserId().equals(newsParam.getUserId())) {

            //完成任务，向biz服务调用任务接口
            String content = "转发【" + ExtendFactory.getContent(newsVO.getTitle(), newsVO.getContent()) + "】被阅读";
            NewsCompleTaskDTO completeTask = missionIntegrationService.compleTask(newsParam.getShareUserId(), TaskEnum.TASK_FIRST_BROWSE.name(), newsParam.getUserId().toString(), content);

            if (Objects.nonNull(completeTask) && completeTask.getGoldNum() > 0) {
                //提取必要信息
                String extendContent = null;
                List<PostImgVo> imgList = getDetailFromNews(newsVO.getImgUrl(), newsVO.getVideoList());
                String extendUrl = CollectionUtils.isNotEmpty(imgList) ? imgList.get(0).getImgUrl() : null;
                if (StringUtils.isEmpty(extendUrl)) {
                    extendContent = ExtendFactory.getContent(newsVO.getTitle(), newsVO.getContent());
                }

                //推送
                messageService.addMessage(
                        newsParam.getShareUserId(),
                        newsParam.getUserId(),
                        completeTask,
                        newsParam.getNewsId(),
                        ReplyTypeEnum.NEWS_REPLY.getCode(),
                        extendUrl,
                        extendContent,
                        newsVO.getTitle());
            }
        }
    }

    @Override
    public Json doRecordNews(Long newsId, Long userId, Integer viewTime, String tabType, String advertShow) {
        NewsRecord nRecord = this.newsRecordMapper.getByIds(newsId, userId, null);
        Date d = new Date();
        if (nRecord != null) {
            nRecord.setUpdateTime(d);
            //阅读时间大于上次阅读时间则更新
            if (viewTime > nRecord.getCheckTime()) {
                nRecord.setCheckTime(viewTime);
            }
            this.newsRecordMapper.updateByPrimaryKeySelective(nRecord);
        } else {
            NewsRecord nr = new NewsRecord();
            nr.setId(nextSequence());
            nr.setNewsId(newsId);
            nr.setUserId(userId);
            nr.setUpdateTime(d);
            nr.setCheckTime(viewTime);
            nr.setGetGold(AppConst.UN_GOLD);
            nr.setAddTime(d);
            nr.setLastLocation(tabType);
            nr.setAdViewType(Byte.valueOf(advertShow));
            this.newsRecordMapper.insertSelective(nr);
        }
        return ResultUtil.genSuccessResult();
    }

    @Override
    public Json<List<News4Client>> listRecommendNews(Long newsId, Long userId, Integer size, String areaCode, Integer platform) {
        if (newsId == null || userId == null) {
            logger.error("新闻详情推荐列表接口----参数有误,newsId:{},userId:{}", newsId, userId);
        }
        if (size == null) {
            size = 5;
        }
        List<Long> ids = newsRecommendIntegrationService.recommendByNewsDetail(newsId, userId, size, platform);

        logger.debug("[listRecommendNews]新闻详情推荐：{}", JSONObject.toJSONString(ids));
        List<MixRecommendDTO> mixRecommendDTOList = ForumPostConvert.convertIds2MixRecommendDTO(ids);

        List<News4Client> news4Clients = SpringContextHolder.getBean(RecommendService.class).listNews4Client(mixRecommendDTOList, userId, areaCode, true);

        return ResultUtil.genSuccessResult(news4Clients);

    }

    @Override
    public Json<List<News4Client>> listRecommendNews4ImgNews(Long newsId) {
        List<News> list = this.newsMapper.queryRecommendNewsList(newsId, 8, null);
        List<News4Client> news4ClientList = list.stream().map(e -> new News4Client(e, null)).collect(Collectors.toList());
        return ResultUtil.genSuccessResult(news4ClientList);
    }

    @Override
    public Json<List<NewsSearchWord>> listHotTitle() {
        return ResultUtil.genSuccessResult(this.newsSearchWordMapper.queryList());
    }

    @Override
    public Json<List<NewsKind>> listNewsKinds() {
        return ResultUtil.genSuccessResult(this.newsKindMapper.selectVideoKinds());
    }

    @Deprecated
    @Override
    public Json<FailVideo> getFailVideo(Long newsId) {
        //新闻视频无法播放的处理
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "无效视频");
    }

    @Override
    public Json doSaveNewsClick(Long newsId) {
        this.newsStatisticsMapper.searchRecommended(newsId);
        return ResultUtil.genSuccessResult();
    }

    @Override
    public void fillExtInfo(Long userId, List<News> result, String areaCode) {
        List<Long> newsCollect = new ArrayList<>();
        if (userId != null) {
            newsCollect = newsCollectMapper.selectByIds(userId, (byte) 1);
        }

        List<Long> finalNewsCollect = newsCollect;
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);

        result.parallelStream().forEach(item -> {
            item.setLinkUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.DETAIL, item.getId(), userId));
            item.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.NEWS, UrlTypeEnum.SHARE, item.getId(), userId));
            item.setCollect(finalNewsCollect.stream().anyMatch(e -> e.equals(item.getId())) ? 1 : 0);
            item.setLocal((byte) (StringUtils.isNotEmpty(item.getAreaDetail()) ? 2 : 1));

            this.exchangeCoverNews(item);
            this.judgeEnablePlaceholder(item, location);
        });
    }

    @Override
    public void exchangeCoverNews(News news) {
        List<PostImgVo> coverFromNews = getCoverFromNews(news.getCoverList(), news.getImgUrl(), news.getVideoList(), news.getCoverSelect());
        news.setNewsImgList(coverFromNews);
        news.setShareImg(this.getShareImg(coverFromNews));
        news.setImgUrl(JSONObject.toJSONString(news.getNewsImgList().stream().map(PostImgVo::getImgUrl).collect(Collectors.toList())));
    }

    @Override
    public void exchangeDetailNews(News news) {
        List<PostImgVo> coverFromNews = getCoverFromNews(news.getCoverList(), news.getImgUrl(), news.getVideoList(), news.getCoverSelect());
        news.setShareImg(this.getShareImg(coverFromNews));
        news.setNewsImgList(getDetailFromNews(news.getImgUrl(), news.getVideoList()));
        news.setImgUrl(JSONObject.toJSONString(news.getNewsImgList().stream().map(PostImgVo::getImgUrl).collect(Collectors.toList())));
    }

    /**
     * 按次序取得分享图片
     *
     * @param coverList
     * @return
     */
    private PostImgVo getShareImg(List<PostImgVo> coverList) {
        PostImgVo postImgVo = null;
        if (CollectionUtils.isNotEmpty(coverList)) {
            Optional<PostImgVo> imgVoOptional = coverList.stream()
                    .filter(e -> StringUtils.isNotEmpty(e.getImgUrl()) && !e.getImgUrl().contains(".gif"))
                    .findFirst();
            if (imgVoOptional.isPresent()) {
                postImgVo = imgVoOptional.get();
            }
        }
        if (Objects.isNull(postImgVo)) {
            postImgVo = PostImgVo.buildImg(newsProperties.getDefaultShareCover());
        }

        return postImgVo;
    }

    @Override
    public List<PostImgVo> getCoverFromNews(String coverStr, String imgUrl, List<PostImgVo> videoList, Byte coverSelect) {
        List<PostImgVo> coverArray = new ArrayList<>();
        if (null == coverSelect) {
            coverSelect = 0;
        }
        //视频封面数量
        Integer vodeoSize = 1;
        //内容如偏数量
        Integer imageSize = 3;

        if (coverSelect == 0) {
            coverArray = getCoverByCoverStr(coverStr);
            if (CollectionUtils.isEmpty(coverArray)) {
                coverArray = getCoverByVideo(videoList, vodeoSize);
            }
            if (CollectionUtils.isEmpty(coverArray)) {
                coverArray = getCoverByImgUrl(imgUrl, imageSize);
            }
        } else if (coverSelect == 1) {
            coverArray = getCoverByVideo(videoList, vodeoSize);
            if (CollectionUtils.isEmpty(coverArray)) {
                coverArray = getCoverByImgUrl(imgUrl, imageSize);
            }
        } else if (coverSelect == 2) {
            vodeoSize = 3;
            coverArray = getCoverByVideo(videoList, vodeoSize);
            if (coverArray.size() < 3) {
                imageSize = vodeoSize - coverArray.size();
                List<PostImgVo> coverByImgUrl = getCoverByImgUrl(imgUrl, imageSize);
                coverArray.addAll(coverByImgUrl);
            }
        }

        return coverArray;
    }


    /**
     * 封面图转封面
     *
     * @param coverStr
     * @return
     */
    @Override
    public List<PostImgVo> getCoverByCoverStr(String coverStr) {
        if (StringUtils.isNotBlank(coverStr)) {
            List<String> coverList = JSONObject.parseArray(coverStr, String.class);
            List<PostImgVo> coverArray = coverList.stream()
                    .filter(StringUtils::isNotBlank)
                    .map(e -> assemblyService.buildImgCover(e)).map(PostImgVo::buildImg)
                    .limit(3)
                    .collect(Collectors.toList());
            return coverArray;
        }
        return Lists.newArrayList();
    }

    /**
     * 内容图转封面
     *
     * @param imgUrl
     * @return
     */
    private List<PostImgVo> getCoverByImgUrl(String imgUrl, Integer size) {
        if (StringUtils.isNotBlank(imgUrl)) {
            List<String> imgList = JSONArray.parseArray(imgUrl, String.class);
            List<PostImgVo> coverArray = imgList.stream().filter(StringUtils::isNotBlank)
                    .map(e -> assemblyService.buildImgCover(e))
                    .map(PostImgVo::buildImg)
                    .limit(size)
                    .collect(Collectors.toList());
            return coverArray;
        }
        return Lists.newArrayList();
    }

    /**
     * 视频图转封面
     *
     * @param videoList
     * @return
     */
    private List<PostImgVo> getCoverByVideo(List<PostImgVo> videoList, Integer size) {
        List<PostImgVo> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(videoList)) {
            for (PostImgVo e : videoList) {
                if (StringUtils.isNotBlank(e.getImgUrl())) {
                    e.setImgUrl(assemblyService.buildVideoCover(e.getImgUrl()));
                    result.add(e);
                }
                if (result.size() >= size) {
                    break;
                }
            }
        }
        return result;
    }


    @Override
    public List<PostImgVo> getDetailFromNews(String imgUrl, List<PostImgVo> videoList) {
        List<PostImgVo> coverArray = new ArrayList<>();
        //1.优先取视频
        if (CollectionUtils.isNotEmpty(videoList)) {
            videoList.stream().filter(e -> StringUtils.isNotEmpty(e.getImgUrl())).forEach(e -> e.setImgUrl(assemblyService.buildVideoCover(e.getImgUrl())));
            coverArray.addAll(videoList);
        }
        //2.其次取内容图片
        if (StringUtils.isNotEmpty(imgUrl)) {
            List<String> imgList = JSONArray.parseArray(imgUrl, String.class);
            coverArray.addAll(imgList.stream().filter(StringUtils::isNoneEmpty).map(e -> assemblyService.buildImgCover(e)).map(PostImgVo::buildImg).collect(Collectors.toList()));
        }
        return coverArray.stream().limit(9).collect(Collectors.toList());
    }

    @Override
    public Json<NewsDetailVO> getNewsDetailById(NewsParam newsParam, BasicParam basicParam, String ip) {
        //新闻id为空时返回空对象
        if (Objects.isNull(newsParam.getNewsId())) {
            return ResultUtil.genSuccessResult(new NewsDetailVO());
        }
        Json<NewsDetailVO> data = this.getNewsDetail(newsParam, basicParam);
        if (data.success()) {
            NewsDetailVO result = data.getResult();
            NewsVO news = result.getNews();
            String content = news.getContent();
            if (StringUtils.isNotEmpty(content)) {
                String htmlContent;
                //小程序和H5不需要html上下文
                if (PlatformEnum.APPLET.getCode() == basicParam.getPlatform()
                        || PlatformEnum.WEB.getCode() == basicParam.getPlatform()) {
                    //添加水印和图片压缩参数
                    htmlContent = assemblyService.fixContent(content, basicParam, ip, ReplyTypeEnum.NEWS_REPLY.getCode(), newsParam.getUserId(), newsParam.getAreaCode(), newsParam.getNewsId());
                } else {
                    htmlContent = assemblyService.assemblyContent(content, basicParam, ip, ReplyTypeEnum.NEWS_REPLY.getCode(), newsParam.getUserId(), newsParam.getAreaCode(), newsParam.getNewsId());
                }
                news.setContent(htmlContent);
            }

            result.setNews(news);
            data.setResult(result);

            //新闻分享,在在小程序中每日首次打开,分享人随机获得20-50的小红花
            //异步调用 用户首次打开小程序完成分享任务
            if (newsParam.getShareUserId() != null && newsParam.getShareUserId() != 0) {
                NewsService newsService = SpringContextHolder.getBean(NewsService.class);
                newsService.completeTaskAndPush(newsParam, news);
            }
        }
        return data;
    }

    @Override
    public PostLikeDto doNewsLike(NewsLikeParam newsLikeParam) {
        PostLikeDto postLikeDto;
        long id;
        NewsLike newsLike = newsLikeMapper.selectByUserIdAndNewsId(newsLikeParam.getUserId(), newsLikeParam.getNewsId());
        UserBean userBean = userIntegrationService.selectUserFromCache(newsLikeParam.getUserId());
        if (StringUtils.isBlank(userBean.getHeadImg()) || StringUtils.isBlank(userBean.getNickname())) {
            logger.info("帖子点赞用户头像或昵称为空,userInfo={}", JSON.toJSONString(userBean));
        }
        //判断帖子是否被用户点赞,或者为是取消点赞状态
        if (Objects.isNull(newsLike) || newsLike.getType() == 0) {
            //生成新闻点赞记录
            id = this.generateNewsLike(newsLike, (byte) 1, newsLikeParam.getNewsId(), newsLikeParam.getUserId(), userBean);
            News news = newsMapper.selectByPrimaryKey(newsLikeParam.getNewsId());
            postLikeDto = calculateFlower(news, newsLikeParam.getUserId(), id, newsLikeParam.getAreaCode());
        } else {
            postLikeDto = null;
        }
        return postLikeDto;
    }


    /**
     * 新闻点赞扣除小红花
     *
     * @param news
     * @param userId
     * @param id
     * @param areaCode
     * @return
     */
    private PostLikeDto calculateFlower(News news, Long userId, long id, String areaCode) {
        //新闻使用非精华帖点赞扣除的小红花数量
        ForumPostStatisticService forumPostStatisticService = SpringContextHolder.getBean(ForumPostStatisticService.class);
        final int postLikeConsume = forumPostStatisticService.getPostLikeConsume(0);

        //判断新闻是否开启占位符
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        judgeEnablePlaceholder(news, location);

        //点赞用户减少红花
        String consumeContent = "点赞【" + ExtendFactory.getContent(news.getTitle(), news.getContent()) + "】 消耗小红花";
        Boolean consume = userAccountIntegrationService.addGold(AccountGoldParam.buildPostLikeExpendParam(userId, postLikeConsume, id, consumeContent));
        PostLikeDto postLikeDto = new PostLikeDto();
        if (consume) {
            //红花扣除成功，填充返回实体
            postLikeDto.setAmount(postLikeConsume);
            postLikeDto.setCode(PostLikeDto.SUCCESS_CODE);
            forumPostStatisticService.fillPostLikeDto(postLikeDto, userId, userId, postLikeConsume);
        } else {
            //扣除红花失败,回滚数据
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            postLikeDto.setCode(PostLikeDto.FAILURE_CODE);
            postLikeDto.setShowFlower(PostLikeDto.FAILURE_CODE);
            postLikeDto.setCopyWriter("你的小红花余额不足，快转发文章赚小红花");
        }
        return postLikeDto;
    }

    /**
     * 生成新闻点赞记录
     */
    private Long generateNewsLike(NewsLike newsLike, Byte type, Long newsId, Long userId, UserBean userBean) {

        String updateHeadImg = StringUtils.isBlank(userBean.getHeadImg()) ? userProperties.getDefaultHeadImgUrl() : userBean.getHeadImg();
        String updateNickname = StringUtils.isBlank(userBean.getNickname()) ? null : userBean.getNickname();
        long id;
        if (Objects.isNull(newsLike)) {
            id = nextId();
            newsLike = NewsLike.buildNewsLike(newsId, userId);
            newsLike.setId(id);
            newsLike.setType(type);
            newsLike.setHeadImg(updateHeadImg);
            newsLike.setUserNickname(updateNickname);
            //插入点赞表
            newsLikeMapper.insertSelective(newsLike);
        } else {
            id = newsLike.getId();
            newsLike.setType(type);
            newsLike.setHeadImg(updateHeadImg);
            newsLike.setUserNickname(updateNickname);
            newsLikeMapper.updateByPrimaryKeySelective(newsLike);
        }
        return id;
    }

    @Override
    public List<NewsLikeDTO> listPostLikeByNewsId(Long newsId) {
        return newsLikeMapper.selectByNewsId(newsId);
    }

    @Override
    public News judgeEnablePlaceholder(News newsVo, LocationDTO location) {
        //开启占位符
        boolean flag = Objects.nonNull(newsVo) && Objects.nonNull(newsVo.getEnablePlaceholder()) && newsVo.getEnablePlaceholder() == 1;
        //内容是否为空
        boolean contentFlag = Objects.nonNull(newsVo) && StringUtils.isNotBlank(newsVo.getContent());
        //标题是否为空
        boolean titleFlag = Objects.nonNull(newsVo) && StringUtils.isNotBlank(newsVo.getTitle());
        //帖子标题是否为空
        if (flag) {
            String replacement = "本地";
            if (Objects.isNull(location) || StringUtils.isBlank(location.getName())) {
                replacement = "本地";
            } else {
                replacement = location.getName();
            }

            if (titleFlag) {
                newsVo.setTitle(newsVo.getTitle().replace("{areaname}", replacement));
            }
            if (contentFlag) {
                newsVo.setContent(newsVo.getContent().replace("{areaname}", replacement));
            }
        }
        return newsVo;
    }

    @Async
    @Override
    public void doTriggerUpdateNewsLikeInfo(Long userId, String nickname, String headImg) {
        if (StringUtils.isBlank(nickname) && StringUtils.isBlank(headImg)) {
            return;
        }
        newsLikeMapper.updateNickNameAndHeadImgByUserId(userId, nickname, headImg);
    }

}
