package com.bxm.localnews.news.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.integration.BizLogIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.domain.ForumPostMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.enums.*;
import com.bxm.localnews.news.event.NoteActionEvent;
import com.bxm.localnews.news.event.UserActionEvent;
import com.bxm.localnews.news.factory.IUrlFactory;
import com.bxm.localnews.news.param.NoteParam;
import com.bxm.localnews.news.service.*;
import com.bxm.localnews.news.util.AreaCodeUtils;
import com.bxm.localnews.news.vo.ForumPostVo;
import com.bxm.localnews.news.vo.NotePublishVo;
import com.bxm.localnews.news.vo.NoteVo;
import com.bxm.newidea.component.emoji.EmojiCodeParser;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.CharUtil;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.google.common.eventbus.EventBus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class NoteServiceImpl extends BaseService implements NoteService {

    @Resource
    private SensitiveWordService sensitiveWordService;

    @Resource
    private ForumPostService forumPostService;

    @Resource
    private ForumTopicService forumTopicService;

    @Resource
    private ForumPostMapper forumPostMapper;

    @Resource
    private MixedRecommendPoolMapper mixedRecommendPoolMapper;

    @Resource
    private ForumProperties forumProperties;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private EventBus userActionEventBus;

    @Resource
    private BizLogIntegrationService bizLogIntegrationService;

    @Resource
    private IUrlFactory iUrlFactory;

    @Override
    public Message create(NoteParam noteParam) {
        if (null == noteParam || null == noteParam.getUserId() || StringUtils.isEmpty(noteParam.getTextField())
                || null == noteParam.getValidDay()) {
            return Message.build(false, "缺少必要参数");
        }
        if (StringUtils.isBlank(noteParam.getTextField())) {
            return Message.build(false, "发布内容为空");
        }
        if (CharUtil.getWordCount(noteParam.getTextField()) > 50) {
            return Message.build(false, "发布内容过长");
        }
        if (sensitiveWordService.contains(noteParam.getTextField())) {
            return Message.build(false, "含有敏感词，请重新编辑");
        }
        // 3.4.0移出发布次数限制 如有需要查看git log找回

        ForumPostVo forumPostVo = new ForumPostVo();
        BeanUtils.copyProperties(noteParam, forumPostVo);
        forumPostVo.setPostType(PostTypeEnum.NOTE.getCode());
        forumPostVo.setStatus((int) PostStatusEnum.APPROVING.getCode());
        //emoji表情编码替换
        forumPostVo.setTextField(EmojiCodeParser.replaceSoftbankEmoji(noteParam.getTextField()));
        forumPostVo.setContent(noteParam.getTextField());
        forumPostVo.setDeliveryType(0);
        if (StringUtils.isNotBlank(forumPostVo.getAreaCode())) {
            forumPostVo.setDeliveryType(1);
            forumPostVo.setAreaCode(AreaCodeUtils.completeAreaCode(forumPostVo.getAreaCode()));
        }
        //默认小纸条话题
        forumPostVo.setTopicIdList(Collections.singletonList(forumProperties.getNoteTopicId()));
        forumPostVo.setValidDate(DateUtils.addField(new Date(), Calendar.DAY_OF_MONTH, forumPostVo.getValidDay()));

        forumPostService.saveForumPost(forumPostVo);
        forumTopicService.updateTopic(forumPostVo);

        //上报发小纸条信息到用户推荐数据源服务(异步)
        UserActionEvent userActionEvent = NoteActionEvent.of().setLastNoteId(forumPostVo.getId())
                .setLastNoteTime(new Date())
                .setUserId(noteParam.getUserId());

        userActionEventBus.post(userActionEvent);


        //更新用户发布数（异步）
        SpringContextHolder.getBean(ForumPostStatisticService.class).doAsyncAddPublishNum(forumPostVo);

        //增加小纸条每日发布数缓存
        redisStringAdapter.increment(getNoteDailyPublishNum());

        //删除小纸条缓存
        redisHashMapAdapter.remove(RedisConfig.USER_FORUM_NOTE, noteParam.getUserId() + "");

        //埋点上报
        bizLogIntegrationService.noteSucceed(noteParam, forumPostVo.getId(), forumPostVo.getUserId(), forumPostVo.getAreaCode());

        //返回小纸条实体
        NoteVo noteVo = new NoteVo();
        noteVo.setId(forumPostVo.getId());
        noteVo.setUserId(forumPostVo.getUserId());
        noteVo.setUserImg(forumPostVo.getUserImg());
        noteVo.setTextField(forumPostVo.getTextField());
        noteVo.setValidDay(forumPostVo.getValidDay());
        noteVo.setDisplayTime(forumPostVo.getDisplayDateTime());
        this.completeNoteInfo(noteVo, noteParam.getUserId());

        return Message.build(true).addParam("note", noteVo);
    }

    @Override
    public Message delete(Long noteId) {
        Long userId = forumPostMapper.getUserIdByPostId(noteId);
        ForumPostVo forumPostVo = new ForumPostVo();
        forumPostVo.setId(noteId);
        forumPostVo.setStatus((int) PostStatusEnum.USER_DELETE.getCode());
        //更新帖子信息表
        forumPostMapper.updateByPrimaryKeySelective(forumPostVo);
        //删除内容推荐库的内容
        mixedRecommendPoolMapper.deleteMixRecommandPoolById(noteId);
        //更新用户发布数
        userIntegrationService.addPostReplyNum(userId, 2);
        //清除用户小纸条缓存
        redisHashMapAdapter.remove(RedisConfig.USER_FORUM_NOTE, userId + "");
        return Message.build(true);
    }

    @Override
    public NotePublishVo getNotePublish(Long userId) {
        NotePublishVo notePublishVo = new NotePublishVo();
        notePublishVo.setPublishNum(getPublishNum(userId));
        // 引导文案
        notePublishVo.setGuidance(forumProperties.getNoteGuidance());
        //获取小纸条轮播列表
        notePublishVo.setCarouselList(forumPostMapper.getCarouselNoteList());

        return notePublishVo;
    }

    @Override
    public List<NoteVo> batchGetUserNote(List<Long> userIdList, Long currentUserId) {
        List<NoteVo> userNoteVoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return userNoteVoList;
        }

        boolean hasCurrent = false;
        if (currentUserId != null && userIdList.contains(currentUserId)) {
            userIdList.remove(currentUserId);
            hasCurrent = true;
        }

        if (CollectionUtils.isNotEmpty(userIdList)) {
            //先取缓存
            Set<String> userIds = userIdList.stream().map(Object::toString).collect(Collectors.toSet());
            List<String> userNoteCache = redisHashMapAdapter.multiGet(RedisConfig.USER_FORUM_NOTE, userIds, String.class);
            log.debug("从缓存中获取用户小纸条数据：" + JSON.toJSONString(userNoteCache));

            //处理缓存数据
            if (CollectionUtils.isNotEmpty(userNoteCache)) {
                userNoteCache.forEach(userCache -> {
                    List<NoteVo> noteVoList = JSON.parseArray(userCache, NoteVo.class);
                    if (CollectionUtils.isNotEmpty(noteVoList)) {
                        userIdList.remove(noteVoList.get(0).getUserId());
                        userNoteVoList.add(noteVoList.get(0));
                    }
                });
            }

            //若缓存中无法取到所有用户小纸条信息
            if (CollectionUtils.isNotEmpty(userIdList)) {
                userNoteVoList.addAll(batchLoadUserNoteVoToRedis(userIdList, true));
            }
        }

        //当前用户需返回仅自己可见内容
        if (hasCurrent) {
            List<NoteVo> noteVoList = forumPostMapper.batchGetUserNotes(Collections.singletonList(currentUserId), 1);
            if (CollectionUtils.isNotEmpty(noteVoList)) {
                userNoteVoList.add(noteVoList.get(0));
            }
        }

        userNoteVoList.forEach(userNote -> this.completeNoteInfo(userNote, currentUserId));
        log.debug("返回用户小纸条：" + JSON.toJSONString(userNoteVoList));
        return userNoteVoList;
    }

    @Override
    public void cacheUserNote(Date startTime) {
        //获取有更新的帖子用户
        List<Long> userIdList = forumPostMapper.getUpdatedUser(startTime);
        //缓存数据
        batchLoadUserNoteVoToRedis(userIdList, true);
    }

    @Override
    public List<NoteVo> getUserNoteList(Long userId) {
        if (null == userId) {
            return new ArrayList<>();
        }

        //先取缓存
        String userNoteCache = redisHashMapAdapter.get(RedisConfig.USER_FORUM_NOTE, userId.toString(), String.class);

        //处理缓存数据
        if (StringUtils.isNotBlank(userNoteCache)) {
            List<NoteVo> noteVoList = JSON.parseArray(userNoteCache, NoteVo.class);
            if (CollectionUtils.isNotEmpty(noteVoList)) {
                return noteVoList;
            }
        }

        //若缓存中无法取到用户图片信息
        return batchLoadUserNoteVoToRedis(Collections.singletonList(userId), false);
    }

    private List<NoteVo> batchLoadUserNoteVoToRedis(List<Long> userIdList, boolean getFirst) {
        List<NoteVo> resultList = new ArrayList<>();
        if (CollectionUtils.isEmpty(userIdList)) {
            return resultList;
        }

        //获取用户小纸条
        List<NoteVo> noteList = forumPostMapper.batchGetUserNotes(userIdList, 0);
        if (CollectionUtils.isEmpty(noteList)) {
            return resultList;
        }

        //映射用户小纸条
        Map<Long, List<NoteVo>> userNoteMap = noteList.stream().collect(Collectors.groupingBy(NoteVo::getUserId));
        Map<String, String> userNoteStrMap = new HashMap<>();
        userIdList.forEach(userId -> {
            List<NoteVo> noteVoList = userNoteMap.get(userId);
            //即使为空，也要覆盖缓存
            userNoteStrMap.put(userId.toString(), JSON.toJSONString(noteVoList));

            //根据是否每个用户一条获取结果
            if (CollectionUtils.isNotEmpty(noteVoList)) {
                if (getFirst) {
                    resultList.add(noteVoList.get(0));
                } else {
                    resultList.addAll(noteVoList);
                }
            }
        });

        //存入缓存
        if (userNoteMap.size() > 0) {
            redisHashMapAdapter.putAll(RedisConfig.USER_FORUM_NOTE, userNoteStrMap);
        }

        return resultList;
    }

    private NoteVo completeNoteInfo(NoteVo noteVo, Long userId) {
        //填充分享地址
        noteVo.setShareUrl(iUrlFactory.getAppUrl(UrlCategoryEnum.FORUM, UrlTypeEnum.SHARE, noteVo.getId(), userId));
        //填充小程序分享地址
        noteVo.setAppletShareUrl(iUrlFactory.getAppletUrl(UrlPositionEnum.POST, noteVo.getId(), userId));
        //填充分享图片
        noteVo.setShareImg(noteVo.getUserImg());

        return noteVo;
    }

    private Integer getPublishNum(Long userId) {
        KeyGenerator key = getUserNotePublishKey(userId);
        if (redisStringAdapter.hasKey(key)) {
            return redisStringAdapter.getInt(key);
        } else {
            //初始化
            Integer publishNum = forumProperties.getUserPublishNum();
            redisStringAdapter.set(key, publishNum);
            redisStringAdapter.expire(key, DateUtils.getCurSeconds());
            return publishNum;
        }
    }

    private KeyGenerator getUserNotePublishKey(Long userId) {
        return RedisConfig.USER_NOTE_PUBLISH.copy().appendKey(userId);
    }

    private KeyGenerator getNoteDailyPublishNum() {
        return RedisConfig.NOTE_DAILY_PUBLISH_NUM.copy().appendKey(DateUtils.formatDate(new Date()));
    }

}
