package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.dto.MixRecommendDTO;
import com.bxm.localnews.news.domain.ForumPostBriefInfoMapper;
import com.bxm.localnews.news.domain.MixedRecommendPoolMapper;
import com.bxm.localnews.news.dto.ShareContentDTO;
import com.bxm.localnews.news.enums.DisplayAreaEnum;
import com.bxm.localnews.news.param.ForumPostFillContext;
import com.bxm.localnews.news.service.ForumPostService;
import com.bxm.localnews.news.service.RecommendService;
import com.bxm.localnews.news.service.ShareRankService;
import com.bxm.localnews.news.vo.ForumPostVo;
import com.bxm.localnews.news.vo.MixedRecommendPool;
import com.bxm.localnews.news.vo.News4Client;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bxm.localnews.news.constant.RedisConfig.AREA_TOP_POST;

@Service
public class ShareRankServiceImpl implements ShareRankService {

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final ForumPostService forumPostService;

    private final ForumPostBriefInfoMapper forumPostBriefInfoMapper;

    private final MixedRecommendPoolMapper mixedRecommendPoolMapper;

    private final RecommendService recommendService;

    private final int cacheNum = 5;

    private TypeReference<List<ShareContentDTO>> typeReference = new TypeReference<List<ShareContentDTO>>() {
    };

    @Autowired
    public ShareRankServiceImpl(RedisHashMapAdapter redisHashMapAdapter,
                                ForumPostService forumPostService,
                                ForumPostBriefInfoMapper forumPostBriefInfoMapper,
                                MixedRecommendPoolMapper mixedRecommendPoolMapper,
                                RecommendService recommendService) {
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.forumPostService = forumPostService;
        this.forumPostBriefInfoMapper = forumPostBriefInfoMapper;
        this.mixedRecommendPoolMapper = mixedRecommendPoolMapper;
        this.recommendService = recommendService;
    }

    @Override
    public List<ShareContentDTO> getTopPost(String areaCode) {
        if (StringUtils.isBlank(areaCode)) {
            return Lists.newArrayList();
        }

        //根据地区编码从缓存获取阅读量靠前的数据
        //如果获取不到则进入降级处理，查询本地
        List<ShareContentDTO> cache = redisHashMapAdapter.get(AREA_TOP_POST, areaCode, typeReference);

        if (cache == null) {
            cache = putCache(areaCode);
        }
        return cache;
    }

    private List<ShareContentDTO> putCache(String areaCode) {
        Date fetchTime = DateUtils.addField(new Date(), Calendar.DAY_OF_YEAR, -2);
        int fixNum;

        // 优先从推荐库获取
        List<ShareContentDTO> cache = getRecommendList(areaCode);

        if (cache.size() < cacheNum) {
            fixNum = cacheNum - cache.size();
            //获取本地阅读量最高的5条帖子
            cache.addAll(getPostList(areaCode, fetchTime, fixNum));

            //本地帖子不足，通过全国新闻进行填充
            if (cache.size() < cacheNum) {
                fixNum = cacheNum - cache.size();
                cache.addAll(getGlobalPostList(areaCode, fixNum));
            }

        }

        redisHashMapAdapter.put(AREA_TOP_POST, areaCode, cache);
        return cache;
    }

    private List<ShareContentDTO> getRecommendList(String areaCode) {
        List<ShareContentDTO> result = Lists.newArrayList();
        List<MixedRecommendPool> recommendList = mixedRecommendPoolMapper.selectShareContent(areaCode, cacheNum);

        if (CollectionUtils.isNotEmpty(recommendList)) {
            List<MixRecommendDTO> recommendMapResult = recommendList.stream().map(record -> {
                MixRecommendDTO mixRecommendDTO = new MixRecommendDTO();
                mixRecommendDTO.setId(record.getId());
                mixRecommendDTO.setOrigin(record.getOrigin());
                return mixRecommendDTO;
            }).collect(Collectors.toList());

            List<News4Client> news4Clients = recommendService.listNews4Client(recommendMapResult, null, areaCode, true);

            for (News4Client news4Client : news4Clients) {
                ShareContentDTO shareContentDTO = ShareContentDTO.builder().build();
                if (news4Client.getPost() != null) {
                    shareContentDTO.setId(news4Client.getPost().getId());
                    shareContentDTO.setTitle(news4Client.getPost().getTitle());
                    shareContentDTO.setOrigin("1");
                    shareContentDTO.setReadNum(news4Client.getPost().getClickCount());
                    shareContentDTO.setShareNum(calShareNum(news4Client.getPost().getClickCount(), news4Client.getPost().getShareCount()));

                    if (news4Client.getPost().getShareImg() != null) {
                        shareContentDTO.setImgList(Lists.newArrayList(news4Client.getPost().getShareImg().getImgUrl()));
                    }

                } else if (news4Client.getNews() != null) {
                    int clickCount = NumberUtils.parseToInt(String.valueOf(news4Client.getNews().getViews()));
                    int shareCount = NumberUtils.parseToInt(String.valueOf(news4Client.getNews().getShares()));

                    shareContentDTO.setId(news4Client.getNews().getId());
                    shareContentDTO.setTitle(news4Client.getNews().getTitle());
                    shareContentDTO.setOrigin("2");
                    shareContentDTO.setReadNum(clickCount);
                    shareContentDTO.setShareNum(calShareNum(clickCount, shareCount));
                    if (news4Client.getNews().getShareImg() != null) {
                        shareContentDTO.setImgList(Lists.newArrayList(news4Client.getNews().getShareImg().getImgUrl()));
                    }
                }

                result.add(shareContentDTO);
            }
        }
        return result;
    }

    /**
     * 优先获取地方的热门新闻
     *
     * @param areaCode  地区编码
     * @param fetchTime 拉取时间
     * @return 帖子列表
     */
    private List<ShareContentDTO> getPostList(String areaCode, Date fetchTime, int fixNum) {
        List<ShareContentDTO> result = Lists.newArrayList();

        List<ForumPostVo> postList = forumPostBriefInfoMapper.getTopSharePostWithLimit(fetchTime, areaCode, fixNum);

        if (CollectionUtils.isNotEmpty(postList)) {
            fillPostExtend(areaCode, result, postList);
        }
        return result;
    }

    /**
     * 地方新闻不够，可能是未开通未运营的地区，拉取最近配置的全国性帖子
     *
     * @param areaCode 地区编码
     * @param limit    拉取数量
     * @return 转化后的分享内容信息
     */
    private List<ShareContentDTO> getGlobalPostList(String areaCode, int limit) {
        List<ShareContentDTO> result = Lists.newArrayList();

        List<ForumPostVo> postList = forumPostBriefInfoMapper.getGlobalPostWithLimit(limit);

        if (CollectionUtils.isNotEmpty(postList)) {
            fillPostExtend(areaCode, result, postList);
        }

        return result;
    }

    private void fillPostExtend(String areaCode, List<ShareContentDTO> result, List<ForumPostVo> postList) {
        ForumPostFillContext context = ForumPostFillContext.builder()
                .data(postList)
                .userId(null)
                .areaCode(areaCode)
                .fillTitle(true)
                .loadComment(false)
                .loadLikeStatus(false)
                .loadTopic(false)
                .loadCollectStatus(false)
                .displayArea(DisplayAreaEnum.INDEX)
                .build();
        forumPostService.fillExtInfo(context);

        postList.forEach(item -> result.add(convert(item)));
    }

    /**
     * 根据产品的需求计算分享数量  V3.2.0
     *
     * @param clickCount 点击次数（阅读数）
     * @param shareCount 分享数量
     */
    private int calShareNum(int clickCount, int shareCount) {
        int shareNum = clickCount / 100 + RandomUtils.nextInt(50, 100);

        if (shareNum < shareCount) {
            shareNum = shareCount;
        }
        return shareNum;
    }

    private ShareContentDTO convert(ForumPostVo post) {

        return ShareContentDTO.builder()
                .id(post.getId())
                .imgList(ImmutableList.of(post.getShareImg().getImgUrl()))
                .readNum(post.getClickCount())
                .shareNum(calShareNum(post.getClickCount(), post.getShareCount()))
                .title(post.getTitle())
                .origin("1")
                .build();
    }

    @Override
    public void refreshCache() {
        Set<String> keys = redisHashMapAdapter.keys(AREA_TOP_POST);
        if (null != keys) {
            keys.forEach(this::putCache);
        }
    }
}
