package com.bxm.localnews.news.task;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.news.service.ForumPostFacadeService;
import com.bxm.localnews.news.service.ForumPostStatisticService;
import com.bxm.localnews.news.vo.ForumPostClickCountVo;
import com.bxm.newidea.component.schedule.task.AbstractCronTask;
import com.xxl.job.core.biz.model.ReturnT;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

/**
 * 创建即发布的周期定时任务
 * 为了模拟帖子的阅读数
 */
@Component
@Slf4j
public class CalcPostClickCountTask extends AbstractCronTask<String> {
    private static final Integer BATCH_SIZE = 50;

    @Resource
    private ForumPostFacadeService forumPostFacadeService;

    @Override
    protected ReturnT<String> service(String param) {
        log.info(".....................................帖子阅读数相关计算 begin...................................");
        //调用新闻服务获取一天内阅读数小于6000的帖子
        List<ForumPostClickCountVo> forumPostClickCountVoList = forumPostFacadeService.getRecentPosts(6000);

        if (CollectionUtils.isNotEmpty(forumPostClickCountVoList)) {
            log.debug("获取到需要计算的帖子列表：" + JSON.toJSONString(forumPostClickCountVoList));
            List<Long> newForumPosts = new ArrayList<>();
            List<Long> oldPassedForumPosts = new ArrayList<>();
            List<Long> oldOtherForumPosts = new ArrayList<>();

            //根据1500的阅读数阈值及帖子状态分组
            for (ForumPostClickCountVo forumPostClickCountVo : forumPostClickCountVoList) {
                if (null == forumPostClickCountVo.getClickCount() || forumPostClickCountVo.getClickCount() < 1500) {
                    newForumPosts.add(forumPostClickCountVo.getId());
                } else if (forumPostClickCountVo.getStatus() != null) {
                    if (1 == forumPostClickCountVo.getStatus()) {
                        oldPassedForumPosts.add(forumPostClickCountVo.getId());
                    } else {
                        oldOtherForumPosts.add(forumPostClickCountVo.getId());
                    }
                }
            }

            addPostClickCount(newForumPosts, oldPassedForumPosts, oldOtherForumPosts);
        }

        log.info(".....................................帖子阅读数相关计算 end...................................");
        return ReturnT.SUCCESS;
    }

    private void addPostClickCount(List<Long> newForumPosts, List<Long> oldPassedForumPosts, List<Long> oldOtherForumPosts) {
        List<ForumPostClickCountVo> forumPostClickList = new ArrayList<>();

        //小于1500的在40-60范围内增加
        if (CollectionUtils.isNotEmpty(newForumPosts)) {
            newForumPosts.forEach(e -> {
                ForumPostClickCountVo forumPostClickCountVo = new ForumPostClickCountVo();
                forumPostClickCountVo.setId(e);
                forumPostClickCountVo.setClickCount(40 + new Random().nextInt(20));
                forumPostClickList.add(forumPostClickCountVo);
            });
        }

        //大于1500的审核通过帖子在5-20范围内增加
        if (CollectionUtils.isNotEmpty(oldPassedForumPosts)) {
            oldPassedForumPosts.forEach(e -> {
                ForumPostClickCountVo forumPostClickCountVo = new ForumPostClickCountVo();
                forumPostClickCountVo.setId(e);
                forumPostClickCountVo.setClickCount(5 + new Random().nextInt(15));
                forumPostClickList.add(forumPostClickCountVo);
            });
        }

        //大于1500的其他状态帖子在5-8范围内增加
        if (CollectionUtils.isNotEmpty(oldOtherForumPosts)) {
            oldOtherForumPosts.forEach(e -> {
                ForumPostClickCountVo forumPostClickCountVo = new ForumPostClickCountVo();
                forumPostClickCountVo.setId(e);
                forumPostClickCountVo.setClickCount(5 + new Random().nextInt(3));
                forumPostClickList.add(forumPostClickCountVo);
            });
        }

        if (CollectionUtils.isNotEmpty(forumPostClickList)) {
            int i = 0;
            while (i + BATCH_SIZE < forumPostClickList.size()) {
                forumPostFacadeService.batchAddClick(forumPostClickList.subList(i, i + BATCH_SIZE));
                i += BATCH_SIZE;
            }
            forumPostFacadeService.batchAddClick(forumPostClickList.subList(i, forumPostClickList.size()));
        }
    }

    @Override
    public String taskName() {
        return "calcPostClickCountTask";
    }

    @Override
    public String cron() {
        return "30 1/6 * * * ? ";
    }

    @Override
    public String description() {
        return "定时增加帖子阅读数任务";
    }

}
