package com.bxm.localnews.news.util;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.exceptions.ServerException;
import com.aliyuncs.mts.model.v20140618.SubmitJobsRequest;
import com.aliyuncs.mts.model.v20140618.SubmitJobsResponse;
import com.aliyuncs.profile.DefaultProfile;
import com.bxm.localnews.news.config.OssProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.vo.TranscodeMap;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import it.sauronsoftware.jave.Encoder;
import it.sauronsoftware.jave.MultimediaInfo;
import it.sauronsoftware.jave.VideoInfo;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * oss视频转码工具类
 *
 * @author stephen
 */
@Component
public class OssTransCodeUtils {
    private static final String MP4_SUFFIX = ".mp4";
    private static final String MP4_FOLDER = "/mp4/";
    private static final String H264 = "h264";

    @Resource
    private OssProperties ossProperties;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private RedisListAdapter redisListAdapter;

    /**
     * 根据原视频地址转码，并缓存原视频对应的转码后视频地址
     *
     * @param originUrl 视频地址
     * @param postId    帖子id
     * @return
     */
    public boolean transCodeByOriginUrl(String originUrl, Long postId) {
        // 若转码开关未打开，或者原视频地址为空，或者不包含指定前缀，则直接返回
        if (!ossProperties.getTranscodeSwitch() || StringUtils.isBlank(originUrl) || !originUrl.contains(ossProperties.getUrlPrefix())) {
            return false;
        }

        KeyGenerator videoTranscodeKey = RedisConfig.VIDEO_TRANSCODE.copy().appendKey(postId);
        // 若缓存中已存在，说明已经转码或者无需转码
        if (redisSetAdapter.exists(videoTranscodeKey, originUrl)) {
            return true;
        }

        String dealOriginUrl = originUrl.replace(ossProperties.getUrlPrefix(), "");
        int index = originUrl.lastIndexOf("/");
        if (index < 0) {
            return false;
        }
        String fileName = originUrl.substring(index + 1);
        // 下载原视频，判断是否h264编码
        boolean downloadResult = download(dealOriginUrl, fileName);
        if (!downloadResult) {
            return false;
        }
        File file = new File(fileName);
        try {
            Encoder encoder = new Encoder();
            MultimediaInfo info = encoder.getInfo(file);
            VideoInfo video = info.getVideo();
            String decoder = video.getDecoder();
            if (H264.equals(decoder)) {
                redisSetAdapter.add(videoTranscodeKey, originUrl);
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            file.delete();
        }

        // 生成转码目标地址
        index = originUrl.lastIndexOf(".");
        if (index < 0) {
            return false;
        }
        String fileSuffix = originUrl.substring(index + 1);
        String destUrl = dealOriginUrl.replace("/" + fileSuffix + "/", MP4_FOLDER).replace("." + fileSuffix, MP4_SUFFIX);

        // 开始转码，并缓存原视频地址和转码后地址
        boolean result = transCode(dealOriginUrl, destUrl);
        if (result) {
            destUrl = ossProperties.getUrlPrefix() + destUrl;
            TranscodeMap transcodeMap = new TranscodeMap();
            transcodeMap.setPostId(postId);
            transcodeMap.setOriginUrl(originUrl);
            transcodeMap.setDestUrl(destUrl);
            redisListAdapter.leftPush(RedisConfig.VIDEO_TRANSCODE_MAP.copy(), transcodeMap);
            redisSetAdapter.add(videoTranscodeKey, destUrl);
            return true;
        }

        return false;
    }

    /**
     * 提交oss转码任务
     *
     * @param ossInputObject
     * @param ossOutputObject
     * @return
     */
    private boolean transCode(String ossInputObject, String ossOutputObject) {
        IAcsClient client = getiAcsClient();

        // 创建API请求并设置参数
        SubmitJobsRequest request = new SubmitJobsRequest();
        // Input
        JSONObject input = new JSONObject();
        input.put("Location", ossProperties.getOssLocation());
        input.put("Bucket", ossProperties.getOssBucket());
        try {
            input.put("Object", URLEncoder.encode(ossInputObject, "utf-8"));
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return false;
        }
        request.setInput(input.toJSONString());
        // Output
        String outputObject;
        try {
            outputObject = URLEncoder.encode(ossOutputObject, "utf-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return false;
        }
        JSONObject output = new JSONObject();
        output.put("OutputObject", outputObject);
        // Ouput->TemplateId
        output.put("TemplateId", ossProperties.getTemplateId());
        JSONArray outputs = new JSONArray();
        outputs.add(output);
        request.setOutputs(outputs.toJSONString());
        request.setOutputBucket(ossProperties.getOssBucket());
        request.setOutputLocation(ossProperties.getOssLocation());
        // PipelineId
        request.setPipelineId(ossProperties.getPipelineId());
        // 发起请求并处理应答或异常
        SubmitJobsResponse response;
        try {
            response = client.getAcsResponse(request);
            if (response.getJobResultList().get(0).getSuccess()) {
                return true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return false;
    }

    /**
     * 创建DefaultAcsClient实例并初始化
     *
     * @return
     */
    private IAcsClient getiAcsClient() {
        DefaultProfile profile = DefaultProfile.getProfile(
                ossProperties.getMpsRegionId(),
                ossProperties.getAccessKeyId(),
                ossProperties.getAccessKeySecret());
        return new DefaultAcsClient(profile);
    }

    private Boolean download(String path, String filePath) {
        try {
            OSSClient ossClient = this.getOssClient();
            GetObjectRequest getObjectRequest = new GetObjectRequest(ossProperties.getOssBucket(), path);
            ossClient.getObject(getObjectRequest, new File(filePath));
            ossClient.shutdown();
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    private OSSClient getOssClient() {
        return new OSSClient(ossProperties.getEndpoint(), ossProperties.getAccessKeyId(), ossProperties.getAccessKeySecret());
    }
}
