package com.bxm.localnews.payment.controller;

import java.math.BigDecimal;

import javax.servlet.http.HttpServletRequest;

import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.payment.dto.OrderStatusDTO;
import com.bxm.localnews.payment.dto.PaymentOrderDTO;
import com.bxm.localnews.payment.facde.service.PayService;
import com.bxm.newidea.component.util.WebUtils;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

/**
 * @author zhaoyadong 2018/12/27 19:35
 * @desc
 */
@Api(tags = "6-01 支付相关接口", description = "包括支付的相关接口操作")
@RestController
@RequestMapping("/api/pay")
public class PayController {

    private PayService payService;

    @Autowired
    public PayController(PayService payService) {
        this.payService = payService;
    }

    @ApiOperation(value = "6-01-1 统一下单（购买VIP），并组装所需支付参数", notes = "h5支付调用的下单接口")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "Long", required = true),
            @ApiImplicitParam(name = "payType", value = "付款类型, 1:微信，2：支付宝", allowableValues = "1,2", required = true),
            @ApiImplicitParam(name = "returnUrl", value = "支付成功后跳转地址", required = true)
    })
    @PostMapping("/order")
    public Json<PaymentOrderDTO> payOrder(Long userId, Byte payType, String returnUrl) {
        PaymentOrderDTO paymentOrderDTO = payService.addUserOrder(userId, payType, returnUrl);
        if (paymentOrderDTO == null) {
            return ResultUtil.genFailedResult("你已成功激活VIP,请勿重复购买");
        }
        return ResultUtil.genSuccessResult(paymentOrderDTO);
    }

    @ApiOperation(value = "6-01-2 查询订单信息", notes = "支付后查询订单状态")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "paymentNum", value = "订单编号, 1待付款，2取消付款，3付款成功，4付款失败，5失败超时", dataType = "String", required = true)
    })
    @GetMapping("/order/status")
    public Json<OrderStatusDTO> queryOrder(String paymentNum) {
        return ResultUtil.genSuccessResult(payService.queryOrder(paymentNum));
    }

    @ApiOperation(value = "6-01-3 统一下单，并组装所需支付参数", notes = "支付调用的下单接口")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "Long", required = true),
            @ApiImplicitParam(name = "payType", value = "付款类型, 1:微信，2：支付宝", allowableValues = "1,2", required = true),
            @ApiImplicitParam(name = "amount", value = "支付金额(金额:元)", required = true),
            @ApiImplicitParam(name = "bizId", value = "业务id", required = false),
    })
    @PostMapping("/order/money")
    public Json<PaymentOrderDTO> payOrder(Long userId, Byte payType, @RequestParam("amount") BigDecimal amount,
                                          @RequestParam(value = "bizId", required = false) Long bizId,
                                          HttpServletRequest request) {
        String ip = WebUtils.getIpAddr(request);
        PaymentOrderDTO paymentOrderDTO = payService.createPayOrder(userId, payType, amount, bizId, ip);
        if (paymentOrderDTO == null) {
            return ResultUtil.genFailedResult("下单失败,请稍后尝试");
        }
        return ResultUtil.genSuccessResult(paymentOrderDTO);
    }

    @ApiOperation(value = "6-01-4 微信APP支付后，查询支付结果", notes = "查询支付结果(只适用于查询微信支付结果)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "paymentNum", value = "订单编号, 返回的状态值:1待付款，2取消付款，3付款成功，4付款失败，5失败超时",
                    dataType = "String", required = true)
    })
    @GetMapping("/order/wxStatus")
    public Json<OrderStatusDTO> queryWxOrderStatus(String paymentNum) {
        return ResultUtil.genSuccessResult(payService.queryOrderStatus(paymentNum, (byte) 1));
    }
}
