package com.bxm.localnews.payment.service.impl;

import java.util.Map;

import com.alibaba.fastjson.JSONObject;
import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.AlipayResponse;
import com.alipay.api.domain.AlipayTradeQueryModel;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.request.AlipayTradeQueryRequest;
import com.alipay.api.request.AlipayTradeWapPayRequest;
import com.alipay.api.response.AlipayTradeQueryResponse;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.payment.config.AlipayProperties;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.facde.service.PayService;
import com.bxm.localnews.payment.service.PayModeService;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * @author zhaoyadong 2019/1/2 11:18
 * @desc
 */
@Component

public class AliPayModeService extends BaseService implements PayModeService {

    private PayProperties payProperties;

    private BizConfigProperties bizConfigProperties;

    private AlipayProperties alipayProperties;

    private AlipayClient alipayClient;

    private PayService payService;

    @Autowired
    public AliPayModeService(PayProperties payProperties, BizConfigProperties bizConfigProperties,
                             AlipayProperties alipayProperties, AlipayClient alipayClient,
                             PayService payService) {
        this.payProperties = payProperties;
        this.bizConfigProperties = bizConfigProperties;
        this.alipayProperties = alipayProperties;
        this.alipayClient = alipayClient;
        this.payService = payService;
    }

    @Override
    public PayTypeEnum support() {
        return PayTypeEnum.ALI_PAY;
    }

    @Override
    public void create(PaymentOrderDetail paymentOrderDetail) {
        try {
            AlipayResponse alipayResponse = alipayClient.pageExecute(createWapPayRequest(paymentOrderDetail));
            paymentOrderDetail.setLink(alipayResponse.getBody());
            logger.info("支付宝预支付返回信息：{}", alipayResponse.getBody());
        } catch (AlipayApiException e) {
            logger.error("用户id{}，订单id{}，创建支付宝预支付信息失败", paymentOrderDetail.getUserId(), paymentOrderDetail.getPaymentNum());
            logger.error("支付宝创建预支付信息失败：{}", e);
        }
    }


    @Override
    public void query(PaymentOrder paymentOrder) {
        try {
            AlipayTradeQueryResponse alipayTradeQueryResponse = alipayClient.execute(createTradeQueryRequest(paymentOrder.getPaymentNum()));
            logger.info("支付宝查询订单返回的信息：{}", alipayTradeQueryResponse.getBody());
        } catch (AlipayApiException e) {
            logger.error("支付宝订单[{}]查询相关信息失败", paymentOrder.getPaymentNum());
            logger.error("支付宝查询订单信息失败:{}", e);
        }
    }

    @Override
    public String callBack(String data) {
        Map<String, String> map = JSONObject.parseObject(data, Map.class);
        String tradeStatus = map.get("tradeStatus");
        String paymentNum = map.get("outTradeNo");
        PaymentOrder paymentOrder = payService.getPaymentOrderByPaymentNo(paymentNum);
        if (paymentOrder == null) {
            logger.error("接收到的支付宝支付回调参数错误,支付订单[{}]不存在", paymentNum);
            return null;
        }

        if (!PaymentStatusEnum.WAIT.getType().equals(paymentOrder.getStatus())) {
            logger.error("支付订单[{}]已处理，状态为：[{}]", paymentNum, paymentOrder.getStatus());
            return null;
        }

        if (tradeStatus.equals("TRADE_FINISHED") || tradeStatus.equals("TRADE_SUCCESS")) {
            afterPaySucceed(map, paymentOrder);
        } else if (tradeStatus.equals("TRADE_CLOSED")) {
            afterPayClose(map, paymentOrder);
        }

        return null;
    }

    /**
     * 取消支付回调函数
     *
     * @param map          包含回调请求的相关信息
     * @param paymentOrder 需要更新的支付订单
     */
    private void afterPayClose(Map<String, String> map, PaymentOrder paymentOrder) {
        paymentOrder.setStatus(PaymentStatusEnum.UNDO.getType());
        paymentOrder.setFinishTime(DateUtils.parseDateNonStrict(map.get("notifyTime")));
        payService.modifyStatus(paymentOrder);
    }

    /**
     * 确认支付成功后的回调函数
     *
     * @param map          包含回调请求的相关信息
     * @param paymentOrder 需要更新的支付订单
     */
    private void afterPaySucceed(Map<String, String> map, PaymentOrder paymentOrder) {
        paymentOrder.setStatus(PaymentStatusEnum.SUCCEED.getType());
        paymentOrder.setFinishTime(DateUtils.parseDateNonStrict(map.get("notifyTime")));
        payService.modifyStatus(paymentOrder);
    }

    /**
     * 创建预支付信息
     *
     * @param paymentOrderDetail
     * @return
     */
    private AlipayTradeWapPayModel createWapPayModel(PaymentOrderDetail paymentOrderDetail) {
        logger.debug("payProperties:{}", payProperties);
        logger.debug("payProperties.vipPrice:{}", payProperties.getVipPrice());
        AlipayTradeWapPayModel model = new AlipayTradeWapPayModel();
        model.setOutTradeNo(paymentOrderDetail.getPaymentNum());
        model.setSubject(payProperties.getVipTitle());
        model.setTotalAmount(payProperties.getVipPrice().toString());
        model.setBody(payProperties.getVipDesc());
        model.setProductCode(alipayProperties.getProductCode());
        model.setTimeoutExpress(alipayProperties.getTimeoutExpress());
        return model;
    }

    /**
     * 创建支付请求
     *
     * @param paymentOrderDetail
     * @return
     */
    private AlipayTradeWapPayRequest createWapPayRequest(PaymentOrderDetail paymentOrderDetail) {
        AlipayTradeWapPayRequest alipayRequest = new AlipayTradeWapPayRequest();
        alipayRequest.setBizModel(createWapPayModel(paymentOrderDetail));
        alipayRequest.setNotifyUrl(alipayProperties.getNotifyUrl());
        alipayRequest.setReturnUrl(bizConfigProperties.getH5ServerHost() + paymentOrderDetail.getReturnUrl()
                + "&paymentNum=" + paymentOrderDetail.getPaymentNum());
        return alipayRequest;
    }

    /**
     * 查询订单信息
     *
     * @param paymentNum
     * @return
     */
    private AlipayTradeQueryModel createTradeQueryModel(String paymentNum) {
        AlipayTradeQueryModel model = new AlipayTradeQueryModel();
        model.setOutTradeNo(paymentNum);
        return model;
    }

    /**
     * 创建查询订单请求
     *
     * @param paymentNum
     * @return
     */
    private AlipayTradeQueryRequest createTradeQueryRequest(String paymentNum) {
        AlipayTradeQueryRequest alipayTradeQueryRequest = new AlipayTradeQueryRequest();
        alipayTradeQueryRequest.setBizModel(createTradeQueryModel(paymentNum));
        return alipayTradeQueryRequest;
    }
}
