package com.bxm.localnews.payment.service.impl;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

import com.bxm.localnews.activity.service.VipFacadeService;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.util.NidGeneratorUtil;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.config.ProfileUtil;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.domain.PaymentOrderMapper;
import com.bxm.localnews.payment.dto.OrderStatusDTO;
import com.bxm.localnews.payment.dto.PaymentOrderDTO;
import com.bxm.localnews.payment.facde.service.PayService;
import com.bxm.localnews.payment.param.PaymentOrderParam;
import com.bxm.localnews.payment.proxy.PayProxyService;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2019/1/2 14:05
 * @desc
 */
@Service

public class PayServiceImpl implements PayService {

    private PaymentOrderMapper paymentOrderMapper;

    private PayProxyService payProxyService;

    private SequenceCreater sequenceCreater;

    private RedisStringAdapter redisStringAdapter;

    private PayProperties payProperties;

    private VipFacadeService vipFacadeService;

    @Autowired
    public PayServiceImpl(PaymentOrderMapper paymentOrderMapper, PayProxyService payProxyService,
                          SequenceCreater sequenceCreater, RedisStringAdapter redisStringAdapter,
                          PayProperties payProperties, VipFacadeService vipFacadeService) {
        this.paymentOrderMapper = paymentOrderMapper;
        this.payProxyService = payProxyService;
        this.sequenceCreater = sequenceCreater;
        this.redisStringAdapter = redisStringAdapter;
        this.payProperties = payProperties;
        this.vipFacadeService = vipFacadeService;
    }

    @Override
    public PaymentOrderDTO addUserOrder(Long userId, Byte payType, String returnUrl) {
        if (vipFacadeService.checkUserActivatedVip(userId)) {
            return null;
        }
        //更新用户以前的订单信息
        updateUserPrePaymentOrder(userId);
        PaymentOrder paymentOrder = createOrder(userId, payType, null);

        //添加新的订单信息
        paymentOrderMapper.insertSelective(paymentOrder);
        redisStringAdapter.set(getOrderStatusKey(paymentOrder.getPaymentNum()), paymentOrder.getStatus(), 3600 * 24);
        //下单
        PaymentOrderDetail paymentOrderDetail = new PaymentOrderDetail();
        BeanUtils.copyProperties(paymentOrder, paymentOrderDetail);
        paymentOrderDetail.setReturnUrl(returnUrl);
        payProxyService.create(paymentOrderDetail);

        return new PaymentOrderDTO(paymentOrderDetail.getLink(), paymentOrder.getPaymentNum());
    }


    @Override
    public OrderStatusDTO queryOrder(String paymentNum) {
        return new OrderStatusDTO(paymentNum, getOrderStatus(paymentNum));
    }

    @Override
    public Message modifyStatus(PaymentOrder paymentOrder) {
        //vip信息是否已激活
        Long bizId = vipFacadeService.addUserVipByPay(paymentOrder.getUserId());
        if (bizId == null) {
            return Message.build(Boolean.FALSE);
        }
        paymentOrder.setBizId(bizId);

        if (paymentOrderMapper.updateByPrimaryKey(paymentOrder) > 0) {
            redisStringAdapter.set(getOrderStatusKey(paymentOrder.getPaymentNum()), paymentOrder.getStatus(), 3600 * 24);
        }

        return Message.build();
    }

    @Override
    public Message modifyPaymentOrderStatus(PaymentOrder paymentOrder) {
        if (paymentOrderMapper.updateByPrimaryKey(paymentOrder) > 0) {
            redisStringAdapter.set(getOrderStatusKey(paymentOrder.getPaymentNum()), paymentOrder.getStatus(), 3600 * 24);
        }

        return Message.build();
    }

    @Override
    public PaymentOrder getPaymentOrderByPaymentNo(String paymentNum) {
        PaymentOrderParam param = new PaymentOrderParam();
        param.setPaymentNum(paymentNum);
        List<PaymentOrder> paymentOrderList = paymentOrderMapper.selectUserOrder(param);
        if (CollectionUtils.isNotEmpty(paymentOrderList)) {
            return paymentOrderList.get(0);
        }
        return null;
    }

    @Override
    public PaymentOrderDTO createPayOrder(Long userId, Byte payType, BigDecimal amount, Long bizId, String ip) {
        updateUserPrePaymentOrder(userId);
        PaymentOrder paymentOrder = createOrder(userId, payType, amount);
        paymentOrder.setBizId(bizId);
        paymentOrder.setClientIp(ip);
        //添加新的订单信息
        paymentOrderMapper.insertSelective(paymentOrder);
        redisStringAdapter.set(getOrderStatusKey(paymentOrder.getPaymentNum()), paymentOrder.getStatus(), 3600 * 24);
        //下单
        PaymentOrderDetail paymentOrderDetail = new PaymentOrderDetail();
        BeanUtils.copyProperties(paymentOrder, paymentOrderDetail);
        payProxyService.create(paymentOrderDetail);
        return new PaymentOrderDTO(paymentOrderDetail.getLink(), paymentOrder.getPaymentNum());
    }

    @Override
    public OrderStatusDTO queryOrderStatus(String paymentNum, Byte payType) {
        PaymentOrder paymentOrder = getPaymentOrderByPaymentNo(paymentNum);
        //获取环境
        if (PaymentStatusEnum.WAIT.getType().equals(paymentOrder.getStatus())) {
            payProxyService.query(paymentOrder, PayTypeEnum.getPayTypeEnum(payType));
        }
        return new OrderStatusDTO(paymentNum, getOrderStatus(paymentNum));
    }

    /**
     * 创建订单
     */
    private PaymentOrder createOrder(Long userId, Byte payType, BigDecimal amount) {
        PaymentOrder paymentOrder = new PaymentOrder();
        paymentOrder.setId(sequenceCreater.nextLongId());
        String paymentNum = generatePaymentNum(payType);
        paymentOrder.setPaymentNum(paymentNum);
        Date now = new Date();
        paymentOrder.setStatus(PaymentStatusEnum.WAIT.getType());
        paymentOrder.setStartTime(now);
        if (amount == null) {
            paymentOrder.setAmount(payProperties.getVipPrice());
        } else {
            paymentOrder.setAmount(amount);
        }
        paymentOrder.setPayType(payType);
        paymentOrder.setUserId(userId);
        paymentOrder.setCreateTime(now);

        return paymentOrder;
    }

    /**
     * 生成订单号
     *
     * @return
     */
    private String generatePaymentNum(Byte payType) {
        String prefix = PayTypeEnum.getNameByType(payType);
        return NidGeneratorUtil.getOrderNo(prefix);
    }

    /**
     * 获取订单状态
     *
     * @param paymentNum
     * @return
     */
    private KeyGenerator getOrderStatusKey(String paymentNum) {

        return RedisConfig.WEIXIN_ORDER_STATUS.copy().appendKey(paymentNum);
    }

    /**
     * 获取缓存中的订单状态
     *
     * @param paymentNum
     * @return
     */
    private Byte getOrderStatus(String paymentNum) {
        String status = redisStringAdapter.getString(getOrderStatusKey(paymentNum));
        if (StringUtils.isEmpty(status)) {
            PaymentOrder paymentOrder = getPaymentOrderByPaymentNo(paymentNum);
            if (paymentOrder == null) {
                return null;
            }

            redisStringAdapter.set(getOrderStatusKey(paymentNum), paymentOrder.getStatus(), 3600 * 24);
            return paymentOrder.getStatus();

        }

        return Byte.valueOf(status);
    }

    /**
     * 更新用户上一笔待付款订单为取消付款
     *
     * @param userId
     */
    private void updateUserPrePaymentOrder(Long userId) {
        PaymentOrder paymentOrder = paymentOrderMapper.getUserPaymentOrderByStatus(userId, PaymentStatusEnum.WAIT.getType());
        if (paymentOrder == null) {
            return;
        }
        paymentOrder.setStatus(PaymentStatusEnum.UNDO.getType());
        paymentOrderMapper.updateByPrimaryKey(paymentOrder);
        redisStringAdapter.set(getOrderStatusKey(paymentOrder.getPaymentNum()), paymentOrder.getStatus(), 3600 * 24);
    }
}
