package com.bxm.localnews.payment.pay.alipay;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.AlipayResponse;
import com.alipay.api.domain.AlipayTradeQueryModel;
import com.alipay.api.domain.AlipayTradeRefundModel;
import com.alipay.api.domain.AlipayTradeWapPayModel;
import com.alipay.api.request.AlipayTradeQueryRequest;
import com.alipay.api.request.AlipayTradeRefundRequest;
import com.alipay.api.request.AlipayTradeWapPayRequest;
import com.alipay.api.response.AlipayTradeQueryResponse;
import com.alipay.api.response.AlipayTradeRefundResponse;
import com.bxm.localnews.common.config.BizConfigProperties;
import com.bxm.localnews.payment.config.AlipayProperties;
import com.bxm.localnews.payment.config.PayProperties;
import com.bxm.localnews.payment.constant.PayTypeEnum;
import com.bxm.localnews.payment.constant.PaymentStatusEnum;
import com.bxm.localnews.payment.param.PaymentRefundParam;
import com.bxm.localnews.payment.pay.PayModeService;
import com.bxm.localnews.payment.pay.PaymentOrderService;
import com.bxm.localnews.payment.pay.impl.AbstractPayModeService;
import com.bxm.localnews.payment.request.AlipayNotifyResult;
import com.bxm.localnews.payment.vo.PaymentOrder;
import com.bxm.localnews.payment.vo.PaymentOrderDetail;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.NumberUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;

import static java.math.RoundingMode.HALF_DOWN;

@Component
@Slf4j
public class AliPayModeServiceImpl extends AbstractPayModeService implements PayModeService {

    private PayProperties payProperties;

    private BizConfigProperties bizConfigProperties;

    private AlipayProperties alipayProperties;

    private AlipayClient alipayClient;

    private PaymentOrderService payService;

    private final static String SUCCESS_CODE = "10000";

    @Autowired
    public AliPayModeServiceImpl(PayProperties payProperties, BizConfigProperties bizConfigProperties,
                                 AlipayProperties alipayProperties, AlipayClient alipayClient,
                                 PaymentOrderService payService) {
        this.payProperties = payProperties;
        this.bizConfigProperties = bizConfigProperties;
        this.alipayProperties = alipayProperties;
        this.alipayClient = alipayClient;
        this.payService = payService;
    }

    @Override
    public PayTypeEnum support() {
        return PayTypeEnum.ALI_PAY;
    }

    @Override
    public void create(PaymentOrderDetail paymentOrderDetail) {
        try {
            AlipayResponse alipayResponse = alipayClient.pageExecute(createWapPayRequest(paymentOrderDetail));
            paymentOrderDetail.setLink(alipayResponse.getBody());
            logger.info("支付宝预支付返回信息：{}", alipayResponse.getBody());
        } catch (AlipayApiException e) {
            logger.error("用户id{}，订单id{}，创建支付宝预支付信息失败", paymentOrderDetail.getUserId(), paymentOrderDetail.getPaymentNum(), e);
        }
    }

    @Override
    public void query(PaymentOrder paymentOrder) {
        try {
            AlipayTradeQueryResponse alipayTradeQueryResponse = alipayClient.execute(createTradeQueryRequest(paymentOrder.getPaymentNum()));
            logger.info("支付宝查询订单返回的信息：{}", alipayTradeQueryResponse.getBody());
        } catch (AlipayApiException e) {
            logger.error("支付宝订单[{}]查询相关信息失败", paymentOrder.getPaymentNum(), e);
        }
    }

    @Override
    public String callBack(String data) {
        AlipayNotifyResult result = JSONObject.parseObject(data, AlipayNotifyResult.class);

        PaymentOrder paymentOrder = payService.getPaymentOrderByPaymentNo(result.getOutTradeNo());
        if (paymentOrder == null) {
            logger.error("接收到的支付宝支付回调参数错误,支付订单[{}]不存在", result.getOutTradeNo());
            return null;
        }

        // 除了支付成功的状态外，其他状态均更新为支付成功
        if (PaymentStatusEnum.SUCCEED.getType().equals(paymentOrder.getStatus())) {
            logger.error("支付订单[{}]重复处理，状态为：[{}]", result.getOutTradeNo(), paymentOrder.getStatus());
            return null;
        }

        if(!PaymentStatusEnum.WAIT.getType().equals(paymentOrder.getStatus())){
            logger.warn("订单[{}]当前状态不是待付款，此情况是异常状态，请注意业务流程和逻辑", paymentOrder.getPaymentNum());
        }

        paymentOrder.setTradeNo(result.getTradeNo());
        paymentOrder.setFinishTime(DateUtils.parseDateNonStrict(result.getNotifyTime()));

        if (StringUtils.equalsAny(result.getTradeStatus(), "TRADE_FINISHED", "TRADE_SUCCESS")) {
            paymentOrder.setStatus(PaymentStatusEnum.SUCCEED.getType());
            payService.modifyStatus(paymentOrder);
        } else if ("TRADE_CLOSED".equals(result.getTradeStatus())) {
            paymentOrder.setStatus(PaymentStatusEnum.UNDO.getType());
            payService.modifyStatus(paymentOrder);
        }

        return null;
    }

    /**
     * 创建预支付信息
     * @param paymentOrderDetail
     * @return
     */
    private AlipayTradeWapPayModel createWapPayModel(PaymentOrderDetail paymentOrderDetail) {
        logger.debug("payProperties:{}", payProperties);
        logger.debug("payProperties.vipPrice:{}", payProperties.getVipPrice());
        AlipayTradeWapPayModel model = new AlipayTradeWapPayModel();
        model.setOutTradeNo(paymentOrderDetail.getPaymentNum());
        model.setSubject(payProperties.getVipTitle());
        model.setTotalAmount(payProperties.getVipPrice().toString());
        model.setBody(payProperties.getVipDesc());
        model.setProductCode(alipayProperties.getProductCode());
        model.setTimeoutExpress(alipayProperties.getTimeoutExpress());
        return model;
    }

    /**
     * 创建支付请求
     * @param paymentOrderDetail
     * @return
     */
    private AlipayTradeWapPayRequest createWapPayRequest(PaymentOrderDetail paymentOrderDetail) {
        AlipayTradeWapPayRequest alipayRequest = new AlipayTradeWapPayRequest();
        alipayRequest.setBizModel(createWapPayModel(paymentOrderDetail));
        alipayRequest.setNotifyUrl(alipayProperties.getNotifyUrl());

        if (StringUtils.isNotBlank(paymentOrderDetail.getReturnUrl())) {
            StringBuilder returnUrl = new StringBuilder();

            if (!StringUtils.startsWith(paymentOrderDetail.getReturnUrl(), "http")) {
                returnUrl.append(bizConfigProperties.getH5ServerHost());
            }

            returnUrl.append(paymentOrderDetail.getReturnUrl());
            returnUrl.append("&paymentNum=");
            returnUrl.append("paymentOrderDetail.getPaymentNum()");

            alipayRequest.setReturnUrl(returnUrl.toString());
        }

        if (log.isDebugEnabled()) {
            log.debug("创建支付宝支付请求参数: {}", JSON.toJSONString(alipayRequest));
        }

        return alipayRequest;
    }

    /**
     * 查询订单信息
     * @param paymentNum
     * @return
     */
    private AlipayTradeQueryModel createTradeQueryModel(String paymentNum) {
        AlipayTradeQueryModel model = new AlipayTradeQueryModel();
        model.setOutTradeNo(paymentNum);
        return model;
    }

    /**
     * 创建查询订单请求
     * @param paymentNum
     * @return
     */
    private AlipayTradeQueryRequest createTradeQueryRequest(String paymentNum) {
        AlipayTradeQueryRequest alipayTradeQueryRequest = new AlipayTradeQueryRequest();
        alipayTradeQueryRequest.setBizModel(createTradeQueryModel(paymentNum));
        return alipayTradeQueryRequest;
    }

    @Override
    public Message submitRefund(PaymentRefundParam param) {
        log.debug("发起支付宝退款申请,请求参数：{}", param);

        BigDecimal totalFee = param.getPayOrder().getAmount();
        BigDecimal refundFee;
        String outRefundNo = buildRefundOrderNo(param);

        if (param.getFullAmount()) {
            refundFee = totalFee;
        } else {
            refundFee = param.getRefundFee();
        }

        AlipayTradeRefundModel model = new AlipayTradeRefundModel();
        //退款对应的支付交易号
        model.setTradeNo(param.getPayOrder().getTradeNo());
        //退款订单号（我方定义）,多次退款需要保证退款订单号一致
        model.setOutRequestNo(outRefundNo);
        //退款金额，支持两位小数
        model.setRefundAmount(NumberUtils.format(refundFee.setScale(2, HALF_DOWN).floatValue()));
        model.setRefundReason("订单退款");

        AlipayTradeRefundRequest tradeRefundRequest = new AlipayTradeRefundRequest();
        tradeRefundRequest.setBizModel(model);

        try {
            createRefundOrder(param, outRefundNo, refundFee);
            System.out.println(JSONObject.toJSONString(tradeRefundRequest));
            AlipayTradeRefundResponse response = alipayClient.execute(tradeRefundRequest);
            if (StringUtils.equalsAny(response.getCode(), SUCCESS_CODE)) {
                log.info("支付申请退款成功，请求参数：{}，返回结果：{}", param, JSON.toJSONString(response));

                changeRefundOrderStatus(outRefundNo,
                        response.getTradeNo(),
                        JSON.toJSONString(response),
                        PaymentStatusEnum.REFUND);
            } else {
                log.error("支付宝退款申请失败，请求参数：{}，请求结果：{}", param, JSON.toJSONString(response));

                changeRefundOrderStatus(outRefundNo,
                        response.getTradeNo(),
                        JSON.toJSONString(response),
                        PaymentStatusEnum.REFUND_FAIL);
                return Message.build(false, "退款申请失败");
            }
        } catch (AlipayApiException e) {
            logger.error(e.getMessage(), e);
            return Message.build(false, "退款申请异常");
        }

        logger.debug("支付退款申请处理完成,请求参数：{}", param);

        return Message.build(true);
    }

    @Override
    public String execRefundCallback(String callbackRequest) {
        throw new UnsupportedOperationException("支付宝无退款回调处理");
    }
    /**
     * 根据付款订单构建退款订单编号
     * @param param 退款订单创建参数
     * @return 退款订单编号
     */
    @Override
    protected String buildRefundOrderNo(PaymentRefundParam param) {
        return "R" + param.getPayOrder().getPaymentNum();
    }

}
