package com.bxm.localnews.channel;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.beans.Transient;
import java.util.Collection;
import java.util.Map;

/**
 * 数据通道代理服务,一种数据可能会写入到多个最终的数据源
 */
@Component
public class DataChannelProxy implements DataChannelService, InitializingBean, ApplicationContextAware {

    private static final Logger LOGGER = LoggerFactory.getLogger(DataChannelProxy.class);

    private ApplicationContext applicationContext;

    private Map<Class<?>, Collection<DataChannelService<?>>> channelList;

    @Override
    public void save(Object data) {
        for (DataChannelService service : this.getMatchServices(data)) {
            service.clear(data);
            service.save(data);
        }
    }

    @Override
    public void remove(Object data) {
        for (DataChannelService service : this.getMatchServices(data)) {
            service.remove(data);
        }
    }

    @Override
    public boolean modify(Object data) {
        boolean result = false;
        for (DataChannelService service : this.getMatchServices(data)) {
            result = service.modify(data);
            if (!result) {
                return result;
            }
        }
        return result;
    }

    private Collection<DataChannelService<?>> getMatchServices(Object data) {
        Collection<DataChannelService<?>> matchServices = this.channelList.get(data.getClass());
        if (matchServices.size() == 0) {
            LOGGER.error("{}数据类型没有找到对应的处理器", data.getClass());
        }
        return matchServices;
    }

    @Override
    public void afterPropertiesSet() {
        Map<String, DataChannelService> beans = this.applicationContext.getBeansOfType(DataChannelService.class);
        this.channelList = Maps.newHashMap();
        for (DataChannelService service : beans.values()) {
            Collection<DataChannelService<?>> matchBeans = this.channelList.get(service.supports());
            if (null == matchBeans) {
                matchBeans = Lists.newArrayList();
            }
            matchBeans.add(service);
            this.channelList.put(service.supports(), matchBeans);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
