package com.bxm.localnews.channel.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.analysis.TFIDFAnalyzesService;
import com.bxm.localnews.channel.DataChannelService;
import com.bxm.localnews.convert.impl.NewsConverter;
import com.bxm.localnews.service.CommonTagService;
import com.bxm.localnews.sync.enums.NewsTagTypeEnum;
import com.bxm.localnews.sync.primary.dao.*;
import com.bxm.localnews.sync.vo.Keyword;
import com.bxm.localnews.sync.vo.local.*;
import com.bxm.newidea.component.tools.StringUtils;
import org.mybatis.spring.SqlSessionTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

abstract class AbstractNewsDataChannelServiceImpl<T> implements DataChannelService<T> {

    @Autowired
    @Qualifier("primarySessionTemplate")
    protected SqlSessionTemplate primarySessionTemplate;

    Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired
    protected LocalnewsPersistentMapper localnewsPersistentMapper;

    @Autowired
    protected LocalnewsAgingMapper localnewsAgingMapper;

    @Autowired
    protected HotnewsAgingMapper hotnewsAgingMapper;

    @Autowired
    protected NewsRecommendedMapper newsRecommendedMapper;

    @Autowired
    protected NewsAgingRecommendedMapper newsAgingRecommendedMapper;

    @Autowired
    protected NewsPersistentRecommendedMapper newsPersistentRecommendedMapper;

    @Autowired
    protected HotnewsAgingRecommendedMapper hotnewsAgingRecommendedMapper;

    @Autowired
    protected LocalnewsPersistentRecommendedMapper localnewsPersistentRecommendedMapper;

    @Autowired
    protected LocalnewsAgingRecommendedMapper localnewsAgingRecommendedMapper;

    @Autowired
    protected NewsReplyMapper newsReplyMapper;

    @Autowired
    protected UserReplyMapper userReplyMapper;

    @Autowired
    NewsStatisticsMapper newsStatisticsMapper;

    @Autowired
    NewsMapper newsMapper;

    @Autowired
    NewsTagMapper newsTagMapper;

    @Autowired
    CommonTagService commonTagService;

    @Autowired
    TFIDFAnalyzesService tfidfAnalyzesService;

    @Autowired
    NewsConverter newsConverter;

    @Autowired
    NewsAgingTagMapper newsAgingTagMapper;

    @Autowired
    NewsAgingMapper newsAgingMapper;

    @Autowired
    NewsPersistentMapper newsPersistentMapper;

    @Autowired
    private NewsPersistentTagMapper newsPersistentTagMapper;

    @Autowired
    private LocalnewsPersistentTagMapper localnewsPersistentTagMapper;

    @Autowired
    private LocalnewsAgingTagMapper localnewsAgingTagMapper;

    /**
     * 清空推荐表，在记录中将数据置为失效
     *
     * @param news
     */
    void remove(News news) {
        //总推荐表和时效推荐表
        clearNewsAging(news);
        this.newsMapper.updateStatusById(news.getId(), news.getStatus());
    }

    /**
     * 清除新闻记录news_total
     *
     * @param newsId
     */
    private void cleanNewsTotal(Long newsId) {
        //清除总记录
        this.newsMapper.deleteById(newsId);
        this.newsTagMapper.deleteByNewsId(newsId);
        this.newsRecommendedMapper.deleteByNewsId(newsId);
    }

    /**
     * 数据保存前清理脏数据
     *
     * @param data 新闻
     */
    void clearDirtyData(News data) {
        //清除推荐表
        clearNewsAging(data);
        //清除记录表
        cleanNewsTotal(data.getId());
    }

    /**
     * 数据保存前清理脏数据
     *
     * @param data 新闻
     */
    void clearModifyDirtyData(News data) {
        //清除推荐表
        clearNewsAging(data);
        //清除记录表
        this.newsTagMapper.deleteByNewsId(data.getId());
    }

    /**
     * 清除推荐表中的数据
     *
     * @param data
     */
    void clearNewsAging(News data) {
        //查询旧的新闻
        News oldNews = newsMapper.selectByPrimaryKey(data.getId());
        if (null != oldNews) {
            //清除总推荐表和时效推荐表
            this.newsAgingMapper.deleteById(data.getId());
            this.newsPersistentMapper.deleteById(data.getId());

            //清除总标签表和时效标签表
            this.newsAgingTagMapper.deleteByNewsId(data.getId());
            this.newsPersistentTagMapper.deleteByNewsId(data.getId());

            //清除总推荐记录表和时效推荐记录表
            this.newsAgingRecommendedMapper.deleteByNewsId(data.getId());
            this.newsPersistentRecommendedMapper.deleteByNewsId(data.getId());

            //清除本地表数据
            if (StringUtils.isNotEmpty(oldNews.getAreaDetail())) {
                String[] areaCodeList = oldNews.getAreaDetail().split(",");
                for (String areaCode : areaCodeList) {
                    Long province = Long.valueOf(areaCode.substring(0, 2));

                    LocalnewsAging localnewsAging = new LocalnewsAging();
                    localnewsAging.setProvince(province);
                    localnewsAging.setId(data.getId());
                    localnewsPersistentMapper.deleteByModel(localnewsAging);
                    localnewsAgingMapper.deleteByModel(localnewsAging);

                    NewsTag newsTag = new NewsTag();
                    newsTag.setProvince(province);
                    newsTag.setNewsId(data.getId());
                    localnewsPersistentTagMapper.deleteByModel(newsTag);
                    localnewsAgingTagMapper.deleteByModel(newsTag);

                    NewsRecommended newsRecommended = new NewsRecommended();
                    newsRecommended.setProvince(province);
                    newsRecommended.setNewsId(data.getId());
                    localnewsPersistentRecommendedMapper.deleteByModel(newsRecommended);
                    localnewsAgingRecommendedMapper.deleteByModel(newsRecommended);
                }
            }

            //清除热门表
            hotnewsAgingMapper.deleteById(data.getId());
            hotnewsAgingRecommendedMapper.deleteByNewsId(data.getId());
        }
    }

    /**
     * 在同步爬虫数据时，
     * 默认是删了之后重新分析tag再录入
     *
     * @param news
     * @return
     */
    boolean modifyNews(News news) {
        //判断新闻是否存在
        News oldNews = newsMapper.selectByPrimaryKey(news.getId());
        if (oldNews != null) {
            //删除历史数据
            clearModifyDirtyData(oldNews);
            //保存新数据
            updateNews(news, oldNews);

            logger.debug("数据变更:{}", news.getId());

            return true;
        }

        return false;
    }

    /**
     * 保存总记录新闻标签
     * 包含新闻标题和内容解析的标签和新闻渠道标签
     *
     * @param news 新闻内容
     */
    private List<NewsTag> saveNewsTag(News news) {
        List<NewsTag> newsTagList = listNewsTag(news);

        //总标签表
        MybatisBatchBuilder.create(NewsTagMapper.class, newsTagList).sessionTemplate(primarySessionTemplate).run(NewsTagMapper::save);

        return newsTagList;
    }

    /**
     * 得到标签列表
     *
     * @param news
     * @return
     */
    private List<NewsTag> listNewsTag(News news) {
        List<NewsTag> newsTagList = new ArrayList<>();
        if (StringUtils.isNotBlank(news.getKindName())) {
            Keyword keyword = new Keyword();
            keyword.setName(news.getKindName());
            generateTag(newsTagList, keyword, NewsTagTypeEnum.TITLE_TAG.getCode(), news.getId());

        }

        if (news.getKeywordList() != null) {
            for (Keyword keyword : news.getKeywordList()) {
                generateTag(newsTagList, keyword, NewsTagTypeEnum.CHANNEL_TAG.getCode(), news.getId());
            }
        }
        return newsTagList;
    }

    /**
     * 组装标签列表
     *
     * @param newsTagList
     * @param keyword
     * @param type
     * @param newsId
     */
    private void generateTag(List<NewsTag> newsTagList, Keyword keyword, Byte type, Long newsId) {
        NewsTag newsTag = new NewsTag();
        newsTag.setAddTime(Calendar.getInstance().getTime());
        newsTag.setName(keyword.getName());
        newsTag.setTagType(type);
        newsTag.setNewsId(newsId);

        //如果是标签分类则显示权重为0
        if (type.equals(NewsTagTypeEnum.TITLE_TAG.getCode())) {
            newsTag.setWeight(0D);
        } else {
            newsTag.setWeight(keyword.getTfidfvalue());
        }

        newsTag.setTagId(commonTagService.saveAndGetTag(keyword.getName()));

        newsTagList.add(newsTag);
    }


    /**
     * 保存新闻都在这个
     * 如果是审核通过且是置顶的状态则保存置顶新闻
     * 如果是审核通过且是热门的状态则保存热门
     * 如果是审核通过且是本地的状态则保存本地新闻
     * 如果是审核通过且是全国的状态则保存全国新闻
     *
     * @param data
     */
    void saveNews(News data) {
        //保存标签-总记录表
        List<NewsTag> newsTagList = this.saveNewsTag(data);

        //保存新闻-总记录表-news_total
        this.newsMapper.save(data);
        if (!CollectionUtils.isEmpty(data.getNewsReplyList())) {
            List<NewsReply> newsReplyList = data.getNewsReplyList();

            MybatisBatchBuilder.create(NewsReplyMapper.class, newsReplyList).sessionTemplate(primarySessionTemplate).run(NewsReplyMapper::insertSelective);
            MybatisBatchBuilder.create(UserReplyMapper.class, newsReplyList).sessionTemplate(primarySessionTemplate).run(UserReplyMapper::insertSelective);
            //同步评论数
            long count = newsReplyList.stream().map(e -> e.getParentId() == 0).count();
            if (count > 0) {
                newsReplyMapper.updateNewsComments(data.getId(), (int) count);
            }

        }

        //未审核或者未上线的新闻不需要进入推荐池中
        if (2 != data.getReviewStatus() || 1 != data.getStatus()) {
            return;
        }

        //置顶
        if (data.getTop() == 2 || data.getKindTop() == 2) {
            return;
        }

        //热门
        if (data.getHot() == 2) {
            putNewsToHotRecommendPool(data, new ArrayList<>());
            return;
        }

        //本地
        if (null != data.getAreaDetail() && data.getDeliveryType() == 1) {
            putNewsToLocalRecommendPool(data, newsTagList, new ArrayList<>());
            return;
        }

        //全国表
        putNewsToNationalRecommendPool(data, newsTagList, new ArrayList<>());


    }

    /**
     * 保存新闻都在这个
     * 如果是审核通过且是置顶的状态则保存置顶新闻
     * 如果是审核通过且是热门的状态则保存热门
     * 如果是审核通过且是本地的状态则保存本地新闻
     * 如果是审核通过且是全国的状态则保存全国新闻
     *
     * @param data
     */
    void updateNews(News data, News oldNews) {
        //保存标签-总记录表
        List<NewsTag> newsTagList = this.saveNewsTag(data);
        List<NewsRecommended> newsRecommendedList = newsRecommendedMapper.selectByNewsId(data.getId());

        //保存新闻-总记录表
        data.setModifyTime(new Date());
        this.newsMapper.updateByPrimaryKeySelective(data);

        data.setShares(oldNews.getShares());
        data.setViews(oldNews.getViews());
        data.setRecommends(oldNews.getRecommends());
        data.setComments(oldNews.getComments());
        data.setActiveViews(oldNews.getActiveViews());
        data.setNewClicks(oldNews.getNewClicks());

        //未审核的新闻不需要进入推荐池中
        if (2 != data.getReviewStatus() || 1 != data.getStatus()) {
            return;
        }

        //置顶
        if (data.getTop() == 2 || data.getKindTop() == 2) {
            return;
        }

        //热门
        if (data.getHot() == 2) {
            putNewsToHotRecommendPool(data, newsRecommendedList);
            return;
        }

        //本地
        if (null != data.getAreaDetail() && data.getDeliveryType() == 1) {
            putNewsToLocalRecommendPool(data, newsTagList, newsRecommendedList);
            return;
        }

        //全国表
        putNewsToNationalRecommendPool(data, newsTagList, newsRecommendedList);


    }

    /**
     * 插入到热门推荐表
     * 热门表中以新闻id和地区做联合索引
     * 热门推荐池不需要标签
     *
     * @param data
     */
    void putNewsToHotRecommendPool(News data, List<NewsRecommended> newsRecommendedList) {
        if (StringUtils.isNotEmpty(data.getAreaDetail())) {
            List<String> areaCodeList = Arrays.asList(data.getAreaDetail().split(","));
            for (String areaCode : areaCodeList) {
                HotnewsAging hotnewsAging = new HotnewsAging();

                BeanUtils.copyProperties(data, hotnewsAging);
                hotnewsAging.setId(data.getId());
                hotnewsAging.setAreaCode(areaCode);
                hotnewsAging.setType((byte) 1);
                hotnewsAgingMapper.save(hotnewsAging);
            }
        } else {
            HotnewsAging hotnewsAging = new HotnewsAging();
            BeanUtils.copyProperties(data, hotnewsAging);
            hotnewsAging.setId(data.getId());
            hotnewsAging.setAreaCode(null);
            hotnewsAging.setType((byte) 1);
            hotnewsAgingMapper.save(hotnewsAging);
        }

        for (NewsRecommended newsRecommended : newsRecommendedList) {
            hotnewsAgingRecommendedMapper.insert(newsRecommended);
        }
    }

    /**
     * 插入到本地推荐表
     * 本地推荐以新闻id和地区做联合索引
     * 本地推荐表需要标签
     *
     * @param data
     */
    void putNewsToLocalRecommendPool(News data, List<NewsTag> newsTagList, List<NewsRecommended> newsRecommendedList) {
        Date issueTime = data.getIssueTime();
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR, -(24 * 10));
        Date agingTime = calendar.getTime();

        List<String> areaCodeList = Arrays.asList(data.getAreaDetail().split(","));
        for (String areaCode : areaCodeList) {
            Long province = Long.valueOf(areaCode.substring(0, 2));
            LocalnewsAging localnewsAging = new LocalnewsAging();
            BeanUtils.copyProperties(data, localnewsAging);

            localnewsAging.setAreaCode(areaCode);
            localnewsAging.setProvince(province);
            localnewsAging.setId(data.getId());

            localnewsPersistentMapper.save(localnewsAging);

            if (issueTime.compareTo(agingTime) > 0) {
                localnewsAgingMapper.save(localnewsAging);
            }

            newsTagList = newsTagList.stream().map(e->e.setProvince(province)).collect(Collectors.toList());
            newsRecommendedList = newsRecommendedList.stream().map(e->e.setProvince(province)).collect(Collectors.toList());

            MybatisBatchBuilder.create(LocalnewsPersistentTagMapper.class,newsTagList).sessionTemplate(primarySessionTemplate).run(LocalnewsPersistentTagMapper::save);
            if (issueTime.compareTo(agingTime) > 0) {
                MybatisBatchBuilder.create(LocalnewsAgingTagMapper.class,newsTagList).sessionTemplate(primarySessionTemplate).run(LocalnewsAgingTagMapper::save);
            }

            MybatisBatchBuilder.create(LocalnewsPersistentRecommendedMapper.class,newsRecommendedList).sessionTemplate(primarySessionTemplate).run(LocalnewsPersistentRecommendedMapper::insert);
            if (issueTime.compareTo(agingTime) > 0) {
                MybatisBatchBuilder.create(LocalnewsAgingRecommendedMapper.class,newsRecommendedList).sessionTemplate(primarySessionTemplate).run(LocalnewsAgingRecommendedMapper::insert);
            }

        }
    }

    /**
     * 插入全国推荐表中
     * 全国推荐表中不含热门、置顶、频道置顶、本地的新闻
     *
     * @param data
     * @param newsTagList
     */
    void putNewsToNationalRecommendPool(News data, List<NewsTag> newsTagList, List<NewsRecommended> newsRecommendedList) {
        Date issueTime = data.getIssueTime();
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.HOUR, -(24 * 10));
        Date agingTime = calendar.getTime();

        NewsAging newsAging = new NewsAging();
        newsAging.setActiveViews(data.getActiveViews())
                .setId(data.getId())
                .setComments(data.getComments())
                .setNewClicks(data.getNewClicks())
                .setRecommends(data.getRecommends())
                .setShares(data.getShares())
                .setCollects(data.getCollects())
                .setViews(data.getViews())
                .setIssueTime(data.getIssueTime())
                .setKindId(data.getKindId());
        newsPersistentMapper.save(newsAging);
        if (issueTime.compareTo(agingTime) > 0) {
            newsAgingMapper.save(newsAging);
        }

        MybatisBatchBuilder.create(NewsPersistentTagMapper.class,newsTagList).sessionTemplate(primarySessionTemplate).run(NewsPersistentTagMapper::save);
        if (issueTime.compareTo(agingTime) > 0) {
            MybatisBatchBuilder.create(NewsAgingTagMapper.class,newsTagList).sessionTemplate(primarySessionTemplate).run(NewsAgingTagMapper::save);
        }

        MybatisBatchBuilder.create(NewsPersistentRecommendedMapper.class,newsRecommendedList).sessionTemplate(primarySessionTemplate).run(NewsPersistentRecommendedMapper::insert);
        if (issueTime.compareTo(agingTime) > 0) {
            MybatisBatchBuilder.create(NewsAgingRecommendedMapper.class,newsRecommendedList).sessionTemplate(primarySessionTemplate).run(NewsAgingRecommendedMapper::insert);
        }

    }

}
