package com.bxm.localnews.channel.impl;

import com.bxm.localnews.sync.primary.dao.NewsMapper;
import com.bxm.localnews.sync.primary.dao.NewsStatisticsMapper;
import com.bxm.localnews.sync.primary.dao.NewsTagMapper;
import com.bxm.localnews.sync.vo.local.News;
import com.bxm.localnews.sync.vo.local.NewsRecommended;
import com.bxm.localnews.sync.vo.local.NewsStatistics;
import com.bxm.newidea.component.tools.SpringContextHolder;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.session.SqlSession;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * 新闻推荐记录数据服务
 */
@Component
public class NewsRecommendedChannelServiceImpl extends AbstractNewsDataChannelServiceImpl<NewsRecommended> {

    private static final String primarySessionTemplateName = "primarySessionTemplate";

    @Autowired(required = false)
    public NewsRecommendedChannelServiceImpl(
            NewsMapper newsMapper,
            NewsStatisticsMapper newsStatisticsMapper,
            NewsTagMapper newsTagMapper) {
        this.newsMapper = newsMapper;
        this.newsStatisticsMapper = newsStatisticsMapper;
        this.newsTagMapper = newsTagMapper;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(NewsRecommended data) {
        logger.debug("新增用户推荐记录");
        data.setRecommendType((byte) 1);
        saveRecommended(data);
    }

    @Override
    public void remove(NewsRecommended data) {
        //nothing to do
    }

    @Override
    public Class<NewsRecommended> supports() {
        return NewsRecommended.class;
    }

    /**
     * @param data
     * @return
     */
    @Override
    public boolean modify(NewsRecommended data) {
        //判断新闻是否存在
        if (newsMapper.exists(data.getNewsId()) > 0) {
            //判断用户对此新闻的推荐记录是否存在
            List<NewsRecommended> newsRecommendedList = newsRecommendedMapper.selectByModel(data);
            if (!CollectionUtils.isEmpty(newsRecommendedList)) {
                logger.debug("该用户对此新闻已有推荐记录:user:{},news:{}", data.getUserId(), data.getNewsId());
                return true;
            }
        }

        return false;
    }

    private void saveRecommended(NewsRecommended data) {
        News news = newsMapper.selectByPrimaryKey(data.getNewsId());

        SqlSessionTemplate primarySessionTemplate = SpringContextHolder.getBean(primarySessionTemplateName);
        SqlSession session = primarySessionTemplate.getSqlSessionFactory().openSession(ExecutorType.BATCH);
        try {
            if (null != news) {

                //未审核或者状态不为上线的新闻不需要进入推荐池中
                if (2 != news.getReviewStatus() || 1 != news.getStatus()) {
                    return;
                }

                //保存新闻推荐记录
                data.setNewsAddTime(new Date());
                newsRecommendedMapper.insert(data);

                //更新新闻静态数据
                this.updateNewsStatistic(data);

                //置顶
                if (news.getTop() == 2 || news.getKindTop() == 2) {
                    return;
                }

                //热门
                if (news.getHot() == 2) {
                    putNewsToHotRecommendPool(data);
                    return;
                }

                //本地
                if (null != news.getAreaDetail()) {
                    putNewsToLocalRecommendPool(data, news.getAreaDetail());
                    return;
                }

                //全国表
                putNewsToNationalRecommendPool(data);
            }
        } catch (Exception e) {
            session.rollback();
            this.logger.error(e.getMessage(), e);
        } finally {
            session.commit();
            session.clearCache();
            session.close();
        }
    }

    /**
     * 加入热门推荐池的推荐记录
     *
     * @param data
     */
    private void putNewsToHotRecommendPool(NewsRecommended data) {
        hotnewsAgingRecommendedMapper.insert(data);
    }

    /**
     * 加入本地推荐池的推荐记录
     *
     * @param data
     */
    private void putNewsToLocalRecommendPool(NewsRecommended data, String areaDetail) {
        List<String> areaCodeList = Arrays.asList(areaDetail.split(","));
        for (String areaCode : areaCodeList) {
            Long province = Long.valueOf(areaCode.substring(0, 2));
            data.setProvince(province);
            localnewsAgingRecommendedMapper.insert(data);
            localnewsPersistentRecommendedMapper.insert(data);
        }
    }

    /**
     * 加入全国推荐池的推荐记录
     *
     * @param data
     */
    private void putNewsToNationalRecommendPool(NewsRecommended data) {
        newsAgingRecommendedMapper.insert(data);
        newsPersistentRecommendedMapper.insert(data);
    }

    /**
     * 更新新闻的推荐次数
     *
     * @param data
     */
    private void updateNewsStatistic(NewsRecommended data) {
        NewsStatistics newsStatistics = new NewsStatistics();
        newsStatistics.setNewsId(data.getNewsId());
        newsStatistics.setRecommends(1);
        this.newsMapper.updateStatisticByPrimaryKeySelective(newsStatistics);
    }
}
