package com.bxm.localnews.channel.impl;

import com.bxm.localnews.sync.primary.dao.NewsMapper;
import com.bxm.localnews.sync.primary.dao.NewsStatisticsMapper;
import com.bxm.localnews.sync.primary.dao.NewsTagMapper;
import com.bxm.localnews.sync.vo.local.News;
import com.bxm.localnews.sync.vo.local.NewsStatistics;
import com.bxm.newidea.component.tools.SpringContextHolder;
import org.apache.ibatis.session.ExecutorType;
import org.apache.ibatis.session.SqlSession;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

@Component
public class NewsStatisticDataChannelServiceImpl extends AbstractNewsDataChannelServiceImpl<NewsStatistics> {


    private final NewsMapper newsMapper;

    @Autowired(required = false)
    public NewsStatisticDataChannelServiceImpl(
            NewsMapper newsMapper,
            NewsStatisticsMapper newsStatisticsMapper,
            NewsTagMapper newsTagMapper
    ) {
        this.newsMapper = newsMapper;
        this.newsStatisticsMapper = newsStatisticsMapper;
        this.newsTagMapper = newsTagMapper;
    }

    @Override
    public void save(NewsStatistics data) {
    }

    @Override
    public void clear(NewsStatistics data) {
    }

    @Override
    public void remove(NewsStatistics data) {
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean modify(NewsStatistics data) {
        return modifyNewsStatistics(data);
    }

    @Override
    public Class<NewsStatistics> supports() {
        return NewsStatistics.class;
    }


    /**
     * 对新闻总表 -> 热门 -> 本地 -> 全国 的静态数据进行更新
     *
     * @param data
     * @return
     */
    private boolean modifyNewsStatistics(NewsStatistics data) {

        News news = newsMapper.selectByPrimaryKey(data.getNewsId());
        //判断新闻是否存在
        if (news != null) {

            //保存新数据
            updateNews(data, news);

            logger.debug("新闻静态数据变更:{}", news.getId());

            return true;
        }

        return false;
    }

    /**
     * 更新新闻以及各个推荐池表的静态数据
     * @param data
     * @param news
     */
    private void updateNews(NewsStatistics data, News news) {
        SqlSessionTemplate primarySessionTemplate = SpringContextHolder.getBean("primarySessionTemplate");
        SqlSession session = primarySessionTemplate.getSqlSessionFactory().openSession(ExecutorType.BATCH);
        try {
            //更新总表中的静态数据
            if (data.getComments() == null || data.getComments() == 0) {
                this.newsMapper.updateStatisticByPrimaryKeySelective(data);
            }

            //未审核的新闻不需要进入推荐池中
            if (2 != news.getReviewStatus() || 1 != news.getStatus()) {
                return;
            }

            //置顶
            if (news.getTop() == 2 || news.getKindTop() == 2) {
                return;
            }

            //热门
            if (news.getHot() == 2) {
                putNewsToHotRecommendPool(data);
                return;
            }

            //本地
            if (null != news.getAreaDetail()) {
                putNewsToLocalRecommendPool(data, news.getAreaDetail());
                return;
            }

            //全国表
            putNewsToNationalRecommendPool(data);
        } catch (Exception e) {
            session.rollback();
            this.logger.error(e.getMessage(), e);
        } finally {
            session.commit();
            session.clearCache();
            session.close();
        }

    }

    /**
     * 更新热门推荐池的静态数据
     * @param data
     */
    private void putNewsToHotRecommendPool(NewsStatistics data) {
        hotnewsAgingMapper.updateStatisticByPrimaryKeySelective(data);
    }

    /**
     * 更新本地推荐池的静态数据
     * @param data
     * @param areaDetail
     */
    private void putNewsToLocalRecommendPool(NewsStatistics data, String areaDetail) {
        String[] areaCodeList = areaDetail.split(",");
        for (String areaCode : areaCodeList) {
            Long province = Long.valueOf(areaCode.substring(0, 2));
            data.setProvince(province);
            localnewsAgingMapper.updateStatisticByPrimaryKeySelective(data);
            localnewsPersistentMapper.updateStatisticByPrimaryKeySelective(data);
        }
    }

    /**
     * 更新全国推荐池的静态数据
     * @param data
     */
    private void putNewsToNationalRecommendPool(NewsStatistics data) {
        newsAgingMapper.updateStatisticByPrimaryKeySelective(data);
        newsPersistentMapper.updateStatisticByPrimaryKeySelective(data);
    }
}
